package com.liveperson.monitoring.requests

import android.content.Context
import android.content.pm.PackageManager
import com.liveperson.infra.log.LPLog
import com.liveperson.infra.network.http.request.HttpPostRequest
import com.liveperson.infra.network.http.request.HttpRequest
import com.liveperson.lp_monitoring_sdk.R
import com.liveperson.monitoring.MonitoringFactory
import com.liveperson.monitoring.model.EngagementWithSession
import com.liveperson.monitoring.model.LPMonitoringIdentity
import com.liveperson.monitoring.model.ModelKeyNames
import com.liveperson.monitoring.sdk.MonitoringParams
import com.liveperson.monitoring.sdk.callbacks.EngagementCallback
import com.liveperson.monitoring.sdk.callbacks.MonitoringErrorType
import com.liveperson.monitoring.sdk.responses.LPEngagementResponse
import org.json.JSONObject


/**
 * Created by nirni on 11/22/17.
 *
 * This is the GetEngagement HTTP request. This request an engagement from the monitoring server (Shark) and return
 * it using the provided callback.
 * The request's body is built using the given MonitoringInitParams object
 */
class GetEngagementRequest(context: Context, identities: List<LPMonitoringIdentity>?, monitoringParams: MonitoringParams?, val callback: EngagementCallback) :
            BaseMonitoringRequest(context, identities, monitoringParams) {

    companion object {
        private const val TAG = "GetEngagementRequest"

        private const val OS_ANDROID = "ANDROID"
        private const val DEVICE_FAMILY_MOBILE = "MOBILE"

        // Client properties keys
        private const val KEY_OS = "os"
        private const val KEY_OS_VERSION = "osVersion"
        private const val KEY_APP_VERSION = "appVersion"
        private const val KEY_DEVICE_FAMILY = "deviceFamily"
        private const val KEY_CLIENT_PROPERTIES = "clientProperties"
    }

    override fun getRequest(): HttpRequest = HttpPostRequest(buildRequestUrl())

    override fun getRequestUrl(): String = context.resources.getString(R.string.get_engagement_url)

    /**
     * Handle the response
     */
    override fun handleResponse(response: String) {

        // Build the engagement from the request
        val engagementWithSession = EngagementWithSession(response)

        // Save the sessionId and/or visitorId if returned in the engagement
        engagementWithSession.let { paramsCache?.sessionId = it.sessionId }
        engagementWithSession.let { paramsCache?.visitorId = it.visitorId }

        try {
            engagementWithSession.engagementDetailsList!![0]
                .let { paramsCache?.connectorId = it.connectorId }
        } catch (e: NullPointerException) {
            callErrorCallback(MonitoringErrorType.PARAMETER_MISSING,
                IllegalArgumentException("No connectorId because EngagementDetails is null"))
            return
        }

        // Post onSuccess to host app
        MonitoringFactory.monitoring.postOnMainThread(Runnable {

            callback.onSuccess(LPEngagementResponse(engagementWithSession))
        })

        getEngagementLanguage(response)
    }

    /**
     *
     */
    override fun callErrorCallback(monitoringErrorType: MonitoringErrorType,exception : Exception?) {

        MonitoringFactory.monitoring.postOnMainThread(Runnable {
            callback.onError(monitoringErrorType, exception)
        })
    }

    /**
     *
     */
    override fun buildRequestBody(identities: List<LPMonitoringIdentity>?, monitoringParams: MonitoringParams?): JSONObject {
        val getEngagementBody = super.buildRequestBody(identities, monitoringParams)
        getEngagementBody.put(KEY_CLIENT_PROPERTIES, buildClientProperties())

        return getEngagementBody
    }

    /**
     * Build the client properties data
     */
    private fun buildClientProperties(): JSONObject {

        val clientProperties = JSONObject();

        clientProperties.put(KEY_OS, OS_ANDROID)
        clientProperties.put(KEY_OS_VERSION, android.os.Build.VERSION.RELEASE)
        val version: String? = try {
            val pInfo = context.packageManager.getPackageInfo(context.packageName, 0)
            pInfo.versionName
        } catch (e: PackageManager.NameNotFoundException) {
            "Unavailable"
        }

        clientProperties.put(KEY_APP_VERSION, version)
        clientProperties.put(KEY_DEVICE_FAMILY, DEVICE_FAMILY_MOBILE)

        return clientProperties
    }

    /**
     * Make acCdn request to retrieve language assigned for this engagement.
     */
    private fun getEngagementLanguage(engagementResponse: String?) {

        var campaignId: String?
        var engagementId:String?
        var engagementRevision:String?

        try {
            val responseArray = JSONObject(engagementResponse).getJSONArray(ModelKeyNames.ENGAGEMENT_DETAILS)
            val engagementDetails = JSONObject(responseArray[0].toString())

            campaignId = engagementDetails.getString(ModelKeyNames.CAMPAIGN_ID)
            engagementId = engagementDetails.getString(ModelKeyNames.ENGAGEMENT_ID)
            engagementRevision = engagementDetails.getString(ModelKeyNames.ENGAGEMENT_REVISION)
        } catch (exception: Exception) {
            LPLog.e(TAG, "Failed to parse engagement details.", exception)
            return
        }

        AcCdnRequest(campaignId, engagementId, engagementRevision).execute()
    }

}
