package com.liveperson.monitoring.requests

import android.content.Context
import android.text.TextUtils
import com.liveperson.infra.ICallback
import com.liveperson.infra.Infra
import com.liveperson.infra.errors.ErrorCode.*
import com.liveperson.infra.log.LPLog
import com.liveperson.infra.log.LPLog.mask
import com.liveperson.infra.network.http.HttpHandler
import com.liveperson.infra.network.http.body.LPJSONObjectBody
import com.liveperson.infra.network.http.request.HttpRequest
import com.liveperson.monitoring.Command
import com.liveperson.monitoring.MonitoringFactory
import com.liveperson.monitoring.model.LPMonitoringIdentity
import com.liveperson.monitoring.sdk.MonitoringParams
import com.liveperson.monitoring.sdk.callbacks.MonitoringErrorType
import org.json.JSONArray
import org.json.JSONException
import org.json.JSONObject

/**
 * Created by nirni on 12/6/17.
 *
 * This is a base class for sending a monitoring request (sendSde or getEngagement)
 */
abstract class BaseMonitoringRequest(val context: Context, private val identities: List<LPMonitoringIdentity>?, private val monitoringParams: MonitoringParams?) : Command {

    companion object {
        private const val TAG = "BaseMonitoringRequest"

        protected const val NUM_OF_RETRIES = 4 // TODO: 12/7/17 get from configuration
        const val DEFAULT_AUTH_ACR = "loa1"
        const val DEFAULT_UNAUTH_ACR = "0"

        const val ISSUER = "iss"
        const val ACR = "acr"
        const val SUB = "sub"
        // Body properties
        private const val KEY_CONSUMER_ID = "consumerId"
        private const val KEY_ENTRY_POINTS = "entryPoints"
        private const val KEY_ENGAGEMENT_ATTRIBUTES = "engagementAttributes"
        private const val KEY_IDENTITIES = "identities"
    }

    private val baseSessionVisitorString = "&vid=%s&sid=%s"

    protected val paramsCache = MonitoringFactory.monitoring.paramsCache

    private var retryCounter = 1


    /////////// Abstract methods ///////////////

    // Get the HttpRequest object to be send
    protected abstract fun getRequest() : HttpRequest

    // Handle the response
    protected abstract fun handleResponse(response: String)

    protected abstract fun callErrorCallback(monitoringErrorType: MonitoringErrorType,exception : Exception?)

    // Get the request URL
    protected abstract fun getRequestUrl(): String


    /**
     * Execute this request
     */
    override fun execute() {

        // Get the request
        val httpRequest = getRequest()
        // Build the request body
        val body = buildRequestBody(identities, monitoringParams)
        // Set to application/json request and set body to request
        val httpPostBody = LPJSONObjectBody(body)
        httpRequest.setBody(httpPostBody)

        LPLog.d(TAG, "Sending body: ${mask(body.toString(4))}")

        // Set a callback to the request
        httpRequest.callback = object : ICallback<String, Exception> {
            override fun onSuccess(value: String?) {

                // Format json message
                try {
                    val messageObject = JSONObject(value)
                    LPLog.d(TAG, "Response: ${mask(messageObject.toString(4))}")
                } catch (e: JSONException) {
                    LPLog.w(TAG, "Error parsing response: ${mask(value)}", e)
                }

                if(value != null) {
                    handleResponse(value)
                }
            }

            override fun onError(exception: Exception?) {
                val message = exception?.message
                if (message != null && message.contains("internalCode\":20")){
                    extractAndSetVid(message)

                    //creating the updated url for the new request, with the VisitorID from the response.
                    httpRequest.url = buildRequestUrl()

                    LPLog.d(TAG, "Account is not loaded yet. Retry...")
                }
                else {
                    if (exception != null) {
                        LPLog.w(TAG, "Request error.", exception)
                    }
                }

                handleError(exception, httpRequest)
            }
        }

        // Execute the request
        HttpHandler.execute(httpRequest)
    }

    /**
     * extracting the visitorID from the message.
     */
    protected fun extractAndSetVid(message: String?)  {
        // extracting the vid (VisitorID) from the response (in the exception) and creating a new url request for the next try with a the responded
        val temp = message?.substringAfter("body:")
        val jsonObject = JSONObject(temp)
        val mess = jsonObject.getString("message")
        val vid = mess.substringAfter("vid: ")

        paramsCache?.visitorId = vid
    }
    /**
     * Handle error
     */
    protected open fun handleError(exception: Exception?, httpRequest: HttpRequest) {

        if (retryCounter > NUM_OF_RETRIES) {
            LPLog.d(TAG, "Done with retries (retry number $retryCounter)." )
            if (exception != null) {
                LPLog.e(TAG, ERR_00000052, "Error: ", exception)
            }
            callErrorCallback(MonitoringErrorType.REQUEST_ERROR,exception)
            return
        }

        LPLog.d(TAG, "Retry No. ${retryCounter++}")
        HttpHandler.executeDelayed(httpRequest, (retryCounter*1000).toLong())

    }

    /**
     * Build the relevant GetEngagementUrl: Set the Monitoring server domain
     */
    protected open fun buildRequestUrl(): String {

        var requestUrlString = String.format(getRequestUrl(), paramsCache?.sharkDomain, paramsCache?.brandId, paramsCache?.appInstallId)

        // If both sessionId and visitorId exist, add them as url params
        if (paramsCache?.sessionId != null || paramsCache?.visitorId != null) {
            LPLog.d(TAG, "SessionId and VisitorId exist. Add them as request params (SessionId=${paramsCache.sessionId}, VisitorId=${paramsCache.visitorId})")
            requestUrlString += String.format(baseSessionVisitorString, paramsCache.visitorId, paramsCache.sessionId)
        }

        return requestUrlString
    }

    /**
     * Build the request body
     */
    protected open fun buildRequestBody(identities: List<LPMonitoringIdentity>?, monitoringParams: MonitoringParams?): JSONObject {

        val getEngagementBody = JSONObject()
        val identitiesBody = JSONArray()


        identities?.forEach {

            val identity = JSONObject()
            // Add the consumerId only if it's not empty

            if (!TextUtils.isEmpty(it.consumerId)) {

                if (!TextUtils.isEmpty(it.issuer)){
                    identity.put(ISSUER,it.issuer)
                }
                identity.put(ACR, DEFAULT_AUTH_ACR)
                identity.put(SUB,it.consumerId)

            }else{

                //un auth
                identity.put(ACR, DEFAULT_UNAUTH_ACR)

                //issuer - url for anonymous IDP server
                val brandId = paramsCache?.brandId

                identity.put(ISSUER, getIDPAnonymousUrl())// un auth - url of anonymous IDP server

                //consumer id- in case we logged in already, extracted from JWT
                val originalConsumerId = getOriginalConsumerId(brandId)

                if(!originalConsumerId.isNullOrBlank()){
                    identity.put(SUB, originalConsumerId)
                }

            }
            identitiesBody.put(identity)

        }


        getEngagementBody.put(KEY_IDENTITIES, identitiesBody)

        // Add SDE data if available
        monitoringParams?.entryPoints?.let{ getEngagementBody.put(KEY_ENTRY_POINTS, it) }
        monitoringParams?.engagementAttributes?.let{ getEngagementBody.put(KEY_ENGAGEMENT_ATTRIBUTES, it) }

        return getEngagementBody
    }

    private fun getOriginalConsumerId(brandId: String?) : String? {
        var originalConsumerId: String? = null
        if (Infra.instance.consumerManager.hasActiveConsumer()) {
            originalConsumerId = Infra.instance.consumerManager.getActiveConsumer()?.originalConsumerId
        }
        return originalConsumerId
    }

    private fun getIDPAnonymousUrl(): String {
        val idpDomain = paramsCache?.idpDomain
        return "https://$idpDomain"
    }


}
