package com.liveperson.monitoring.cache

import com.liveperson.infra.errors.ErrorCode.*
import com.liveperson.infra.log.LPLog
import com.liveperson.infra.managers.PreferenceManager
import com.liveperson.lp_monitoring_sdk.R
import com.liveperson.monitoring.MonitoringFactory
import com.liveperson.monitoring.model.LPMonitoringIdentity
import com.liveperson.monitoring.utils.Clearable
import java.util.*
import kotlin.collections.ArrayList

/**
 * Created by nirni on 12/3/17.
 *
 * This class holds all parameters that are relevant to the session.
 * All parameters are stored in memory in addition to the Shared Preferences. Upon initialization
 * all parameters are loaded from Shared Preferences into memory. When saving new value, it is saved to
 * memory and to Shared Preferences
 */
open class MonitoringParamsCache(val brandId: String) : ParamsCache, Clearable {

    companion object {
        private const val TAG = "MonitoringParamsCache"

        // CSDS key names
        const val CSDS_SHARK_DOMAIN_KEY = "smt" // Shark domain in CSDS
        const val CSDS_LOGGOS_DOMAIN_KEY = "loggos" // Loggos domain in CSDS
        const val CSDS_IDP_DOMAIN_KEY = "idp" // IDP domain in CSDS
        const val CSDS_ACCDN_DOMAIN_KEY = "acCdnDomain"

        // Shared preferences keys
        const val LAST_CSDS_UPDATE_TIMESTAMP_KEY = "lastCsdsUpdateTimestampKey"
        const val SESSION_ID_KEY = "sessionIdKey"
        const val VISITOR_ID_KEY = "visitorIdKey"
        const val CONNECTOR_ID_KEY = "connectorIdKey"
        const val APP_INSTALL_ID_KEY = "appInstallIdKey"
        const val SDK_INFO_SEND_KEY = "sdkInfoSendKey"
        const val MONITORING_IDENTITIES_KEY = "monitoringIdentitiesKey"
    }

    // Holds the CSDS domain (QA or production)
    var csdsMainDomain: String? = null
    // Monitoring server's (Shark) domain fetched from CSDS
    var sharkDomain: String? = null
    // Loggos domain
    var loggosDomain: String? = null
    var idpDomain: String? = null
    var acCdnDomain: String? = null
    // Last timestamp when CSDS data was updated
    var lastCsdsUpdateTimestamp : Long = 0
        set(value) {
            field = value
            PreferenceManager.getInstance().setLongValue(LAST_CSDS_UPDATE_TIMESTAMP_KEY, brandId, value)
        }

    // Session ID received from Shark response
    var sessionId : String? = null
        set(value) {
            field = value
            PreferenceManager.getInstance().setStringValue(SESSION_ID_KEY, brandId, value)
        }

    // Visitor ID received from Shark response
    var visitorId : String? = null
        set(value) {
            field = value
            PreferenceManager.getInstance().setStringValue(VISITOR_ID_KEY, brandId, value)
        }

    var connectorId: String? = null
        set(value) {
            field = value
            PreferenceManager.getInstance().setStringValue(CONNECTOR_ID_KEY, brandId, value)
        }


    // AppInstallId value
    var appInstallId : String? = null
        set(value) {
            field = value
            PreferenceManager.getInstance().setStringValue(APP_INSTALL_ID_KEY, brandId, value)
        }

    // SDK info sent
    var sdkInfoSent = false
        set(value) {
            field = value
            PreferenceManager.getInstance().setBooleanValue(SDK_INFO_SEND_KEY, brandId, value)
        }

    var monitoringIdentities = ArrayList<LPMonitoringIdentity>()
        set(value) {
            field = value
            if (value.isNotEmpty()) {
                PreferenceManager.getInstance()
                    .setStringsSet(MONITORING_IDENTITIES_KEY, brandId,
                        value.map { identity ->
                        "${identity.consumerId} : ${identity.issuer}"
                    }.toMutableSet())
            }
        }

    // Update memory from Shared Preferences
    init {
        sharkDomain = PreferenceManager.getInstance().getStringValue(CSDS_SHARK_DOMAIN_KEY, brandId, null)
        loggosDomain = PreferenceManager.getInstance().getStringValue(CSDS_LOGGOS_DOMAIN_KEY, brandId, null)
        idpDomain = PreferenceManager.getInstance().getStringValue(CSDS_IDP_DOMAIN_KEY, brandId, null)
        acCdnDomain = PreferenceManager.getInstance().getStringValue(CSDS_ACCDN_DOMAIN_KEY, brandId, null)
        lastCsdsUpdateTimestamp = PreferenceManager.getInstance().getLongValue(LAST_CSDS_UPDATE_TIMESTAMP_KEY, brandId, 0)
        sessionId = PreferenceManager.getInstance().getStringValue(SESSION_ID_KEY, brandId, null)
        visitorId = PreferenceManager.getInstance().getStringValue(VISITOR_ID_KEY, brandId, null)
        connectorId = PreferenceManager.getInstance().getStringValue(CONNECTOR_ID_KEY, brandId, null)
        appInstallId = PreferenceManager.getInstance().getStringValue(APP_INSTALL_ID_KEY, brandId, null)
        sdkInfoSent = PreferenceManager.getInstance().getBooleanValue(SDK_INFO_SEND_KEY, brandId, false)

        monitoringIdentities = ArrayList(PreferenceManager.getInstance().getStringSet(
            MONITORING_IDENTITIES_KEY, brandId,
            mutableSetOf()
        ).map { identity ->
            LPMonitoringIdentity(identity.substringBefore(":"), identity.substringAfter(":"))
        })

        csdsMainDomain = if (brandId.startsWith("qa") || brandId.startsWith("le")) {
            MonitoringFactory.monitoring.applicationContext?.resources?.getString(R.string.csds_qa_url)
        } else {
            MonitoringFactory.monitoring.applicationContext?.resources?.getString(R.string.csds_url)
        }
    }

    /**
     * Update CSDS domains in SharedPreferences and in memory.
     * Return true if one of the domains that are relevant to the SDK are changed, false if non changed.
     * If the SharedPreferences was empty before updating - return true
     * @param csdsDomains
     * @return
     */
    override fun updateCsdsDomains(csdsDomains: HashMap<String, String>): Boolean {
        var updated = false

        if (csdsDomains.isEmpty()) {
            LPLog.e(TAG, ERR_00000001, "updateCsdsDomains: no domains received")
            return false
        }

        // --- Set all required domains ---
        // Set Shark domain
        if (updateDomain(csdsDomains, CSDS_SHARK_DOMAIN_KEY, sharkDomain)) {
            updated = true
        }

        // Set Loggos domain
        if (updateDomain(csdsDomains, CSDS_LOGGOS_DOMAIN_KEY, loggosDomain)) {
            updated = true
        }

        // Set Loggos domain
        if (updateDomain(csdsDomains, CSDS_IDP_DOMAIN_KEY, idpDomain)) {
            updated = true
        }

        if (updateDomain(csdsDomains, CSDS_ACCDN_DOMAIN_KEY, acCdnDomain)) {
            updated = true
        }

        // Update last update time when CSDS was updated
        lastCsdsUpdateTimestamp = System.currentTimeMillis()

        return updated
    }

    /**
     * Get a service domain from memory. Null if domain is not available
     */
    override fun getServiceDomain(serviceName: String): String? {
        var domainToReturn: String? = null

        when (serviceName) {
            CSDS_SHARK_DOMAIN_KEY -> domainToReturn = sharkDomain
            CSDS_LOGGOS_DOMAIN_KEY -> domainToReturn = loggosDomain
            CSDS_IDP_DOMAIN_KEY -> domainToReturn = idpDomain
        }

        return domainToReturn
    }

    /**
     * Check if CSDS data is already filled. This checks if the required domains are in memory
     */
    override fun isCsdsFilled(): Boolean {
        return sharkDomain != null
    }

    override fun clear() {
        PreferenceManager.getInstance().remove(CSDS_SHARK_DOMAIN_KEY, brandId)
        PreferenceManager.getInstance().remove(CSDS_LOGGOS_DOMAIN_KEY, brandId)
        PreferenceManager.getInstance().remove(CSDS_IDP_DOMAIN_KEY, brandId)
        PreferenceManager.getInstance().remove(CSDS_ACCDN_DOMAIN_KEY, brandId)
        PreferenceManager.getInstance().remove(LAST_CSDS_UPDATE_TIMESTAMP_KEY, brandId)
        PreferenceManager.getInstance().remove(SESSION_ID_KEY, brandId)
        PreferenceManager.getInstance().remove(VISITOR_ID_KEY, brandId)
        PreferenceManager.getInstance().remove(APP_INSTALL_ID_KEY, brandId)
        PreferenceManager.getInstance().remove(SDK_INFO_SEND_KEY, brandId)
        PreferenceManager.getInstance().remove(MONITORING_IDENTITIES_KEY, brandId)

        csdsMainDomain = null
        clearAllDomains()
        lastCsdsUpdateTimestamp = 0
        sessionId = null
        visitorId = null
        connectorId = null
        appInstallId = null
        sdkInfoSent = false
        monitoringIdentities.clear()
    }

    private fun clearAllDomains(){
        sharkDomain = null
        loggosDomain = null
        acCdnDomain = null
    }

    /**
     * Update specific domain from the given CSDS map.
     * @param csdsDomains
     * @param csdsDomainKey
     * @param oldDomainValue
     * @return true - if the value is a new one, false - if same as old value
     */

    private fun updateDomain(csdsDomains: HashMap<String, String>, csdsDomainKey: String, oldDomainValue: String?): Boolean {
        var updated = false

        val domain = csdsDomains[csdsDomainKey]
        if (domain != null) {
            PreferenceManager.getInstance().setStringValue(csdsDomainKey, brandId, domain)
            if (oldDomainValue != null && domain != oldDomainValue) {
                updated = true
            }
            when (csdsDomainKey) {
                CSDS_SHARK_DOMAIN_KEY -> sharkDomain = domain
                CSDS_LOGGOS_DOMAIN_KEY -> loggosDomain = domain
                CSDS_IDP_DOMAIN_KEY -> idpDomain = domain
                CSDS_ACCDN_DOMAIN_KEY -> acCdnDomain = domain
            }
        }

        return updated
    }

}
