package com.liveperson.monitoring

import android.content.Context
import android.os.Handler
import android.os.Looper
import android.text.TextUtils
import com.liveperson.infra.InternetConnectionService
import com.liveperson.infra.auth.LPAuthenticationParams
import com.liveperson.infra.log.LPLog
import com.liveperson.infra.log.LPLog.mask
import com.liveperson.monitoring.cache.MonitoringParamsCache
import com.liveperson.monitoring.connection.MonitoringTasksExecutor
import com.liveperson.monitoring.managers.MonitoringRequestManager
import com.liveperson.monitoring.model.LPMonitoringIdentity
import com.liveperson.monitoring.sdk.MonitoringInternalInitParams
import com.liveperson.monitoring.sdk.MonitoringParams
import com.liveperson.monitoring.sdk.callbacks.EngagementCallback
import com.liveperson.monitoring.sdk.callbacks.MonitoringErrorType
import com.liveperson.monitoring.sdk.callbacks.SdeCallback

/**
 * Created by nirni on 11/21/17.
 */
open class Monitoring : IMonitoring {

    companion object {
        private const val TAG = "Monitoring"
    }

    private var appHandler : Handler? = null
    private var requestManager : MonitoringRequestManager? = null

    // Indicates that Monitoring was initialized
    var initialized = false

    var paramsCache : MonitoringParamsCache? = null
    var brandId : String? = null
    var applicationContext : Context? = null
    private var monitoringTasksExecutor: MonitoringTasksExecutor? = null

    fun init(monitoringInternalInitParams: MonitoringInternalInitParams): Boolean {

        LPLog.d(TAG, "init: Monitoring module version = " + com.liveperson.lp_monitoring_sdk.BuildConfig.VERSION_NAME)

        LPLog.d(TAG, "Initializing Monitoring SDK. MonitoringInternalInitParams: $monitoringInternalInitParams")

        appHandler = Handler(Looper.getMainLooper())
        applicationContext = monitoringInternalInitParams.applicationContext
        brandId = monitoringInternalInitParams.accountId
        if (TextUtils.isEmpty(brandId)) {
            LPLog.w(TAG, "BrandId is empty. Aborting")
            return false
        }
        val lBrandId = brandId
        lBrandId ?.let { paramsCache = MonitoringParamsCache(lBrandId)}
        // Fail if paramsCache is not initialized properly
        paramsCache ?: return false
        val lParamsCache = paramsCache
        paramsCache?.appInstallId = monitoringInternalInitParams.appInstallId

        lParamsCache?.let {requestManager = MonitoringRequestManager(brandId, lParamsCache)}
        monitoringTasksExecutor = MonitoringTasksExecutor(brandId!!, applicationContext!!, paramsCache!!)
        initialized = true

        return true
    }

    override fun logout(): Boolean {

        LPLog.d(TAG, "Logging out...")

        if (!initialized) {
            LPLog.w(TAG, "Not initialized")
            return true
        }

        paramsCache?.clear()
        paramsCache = null
        brandId = null
        applicationContext = null

        initialized = false
        return true
    }

    override fun isInitialized() : Boolean {
        return initialized
    }

    override fun  getEngagement(
        context: Context,
        identities: List<LPMonitoringIdentity>?,
        monitoringParams: MonitoringParams?,
        lpAuthenticationParams: LPAuthenticationParams?,
        callback: EngagementCallback
    ) {

        LPLog.d(TAG, "GetEngagement. LPMonitoringIdentity: ${mask(identities?.takeIf { it.isNotEmpty() }?.get(0))}, SDE: ${mask(monitoringParams)}")

        if (!initialized) {
            LPLog.w(TAG, "Not initialized")
            callback.onError(MonitoringErrorType.NOT_INITIALIZED, null)
            return
        }

        if (!InternetConnectionService.isNetworkAvailable()) {
            LPLog.d(TAG, "No network. calling callback.onError")

            callback.onError(MonitoringErrorType.NO_NETWORK, null)
            return
        }

        monitoringTasksExecutor?.runEngagementTask(lpAuthenticationParams, identities, monitoringParams, callback)
    }

    override fun sendSde(context: Context, identities: List<LPMonitoringIdentity>?, monitoringParams: MonitoringParams?, callback: SdeCallback) {

        LPLog.d(TAG, "sendSde. ConsumerId: ${mask(identities)}, SDE: ${mask(monitoringParams)}")

        if (!initialized) {
            LPLog.w(TAG, "Not initialized")
            callback.onError(MonitoringErrorType.NOT_INITIALIZED, null)
            return
        }

        if (identities == null || identities.isEmpty()) {
            callback.onError(MonitoringErrorType.PARAMETER_MISSING, Exception("ConsumerId is mandatory"))
            return
        }

        if (!InternetConnectionService.isNetworkAvailable()) {
            LPLog.d(TAG, "No network. calling callback.onError")
            callback.onError(MonitoringErrorType.NO_NETWORK, null)
            return
        }


        monitoringTasksExecutor?.runSendSdeTask(identities, monitoringParams, callback)
    }

    /**
     * Post the given Runnable on the main thread
     */
    fun postOnMainThread(runnable: Runnable) {
        appHandler?.post(runnable)
    }
}
