package com.liveperson.monitoring.connection.tasks

import com.liveperson.infra.ICallback
import com.liveperson.infra.auth.LPAuthenticationParams
import com.liveperson.infra.errors.ErrorCode
import com.liveperson.infra.log.LPLog
import com.liveperson.infra.network.http.requests.CSDSRequest
import com.liveperson.monitoring.MonitoringFactory
import com.liveperson.monitoring.cache.MonitoringParamsCache
import com.liveperson.monitoring.connection.MonitoringTaskHandler
import com.liveperson.monitoring.sdk.callbacks.IMonitoringCallback
import com.liveperson.monitoring.sdk.callbacks.MonitoringErrorType
import com.liveperson.monitoring.sdk.responses.LPSdeResponse

class CSDSTask(val brandId : String,
               val domain: String,
               val paramsCache: MonitoringParamsCache,
               val lpAuthenticationParams: LPAuthenticationParams?,
               val callback: IMonitoringCallback<out LPSdeResponse, Exception>,
               val nextTask: MonitoringTaskHandler
               ) : MonitoringTaskHandler {

    companion object {
        private const val TAG = "CSDSTask"
    }

    override fun execute(args: Map<String, Any?>) {
        LPLog.i(TAG, "executeNextTask: Starting $TAG")

        CSDSRequest(domain, brandId, lpAuthenticationParams?.certificatePinningKeys, object : ICallback<HashMap<String, String>, Exception> {
            override fun onSuccess(csdsUrls: HashMap<String, String>?) {
                if (csdsUrls != null && csdsUrls.isNotEmpty()) {
                    LPLog.i(TAG, csdsUrls.toString())

                    // Update the CSDS data to cache
                    paramsCache.updateCsdsDomains(csdsUrls)

                    LPLog.d(TAG, "Calling the next request...")
                    nextTask.execute()

                } else {
                    LPLog.e(TAG, ErrorCode.ERR_0000000D, "execute: Received empty CSDS domain map")
                    MonitoringFactory.monitoring.postOnMainThread(Runnable {
                        callback.onError(MonitoringErrorType.CSDS_ERROR, null)
                    })

                }
            }

            override fun onError(exception: Exception?) {
                exception.let {
                    LPLog.e(
                        TAG,
                        ErrorCode.ERR_0000000E, "onError: ", it)
                }
                MonitoringFactory.monitoring.postOnMainThread(Runnable {
                    callback.onError(MonitoringErrorType.CSDS_ERROR, exception)
                })
            }

        }).execute()
    }
}