package com.liveperson.monitoring.connection.tasks

import android.content.Context
import com.liveperson.infra.auth.LPAuthenticationParams
import com.liveperson.infra.callbacks.AuthCallBack
import com.liveperson.infra.log.LPLog
import com.liveperson.infra.model.Consumer
import com.liveperson.infra.model.errors.AuthError
import com.liveperson.infra.network.http.requests.AuthRequest
import com.liveperson.monitoring.MonitoringFactory
import com.liveperson.monitoring.cache.MonitoringParamsCache
import com.liveperson.monitoring.connection.MonitoringTaskHandler
import com.liveperson.monitoring.sdk.callbacks.IMonitoringCallback
import com.liveperson.monitoring.sdk.callbacks.MonitoringErrorType
import com.liveperson.monitoring.sdk.responses.LPSdeResponse

class IdpTask(val brandId: String,
              val context: Context,
              val lpAuthenticationParams: LPAuthenticationParams,
              val paramsCache: MonitoringParamsCache,
              val callback: IMonitoringCallback<out LPSdeResponse, Exception>,
              val nextTask: MonitoringTaskHandler) : MonitoringTaskHandler {

    private val TAG = "IdpTask"
    private var connectorId: String? = null

    override fun execute(args: Map<String, Any?>) {
        LPLog.i(TAG, "executeNextTask: Starting $TAG")

        connectorId = args["connectorId"] as String?

        val idpDomain = paramsCache.idpDomain.takeIf {
            !it.isNullOrEmpty()
        } ?: run {
            onTaskError(Exception("Domain not found"))
            return
        }

        val callback = object : AuthCallBack {

            override fun onAuthSuccess(consumer: Consumer) {
                LPLog.i(TAG, "On Auth success: ${LPLog.mask(consumer)}")
                nextTask.execute(mapOf("token" to consumer.lpToken))
            }

            override fun onConsumerSwitch(oldConsumer: Consumer, newConsumer: Consumer) {
                // We ignore consumer switch for monitoring request
            }

            override fun onAuthFailed(error: AuthError) {
                LPLog.w(TAG, "onAuthFailed: $error")
                error.exception?.let { onTaskError(it) }
            }
        }

        AuthRequest(
            context,
            brandId,
            idpDomain,
            lpAuthenticationParams,
            paramsCache.hostVersion,
            lpAuthenticationParams.certificatePinningKeys,
            null,
            connectorId,
            false,
            callback
        ).also {
            it.authenticate()
        }

    }

    private fun onTaskError(exception: Exception) {
        MonitoringFactory.monitoring.postOnMainThread(Runnable {
            callback.onError(MonitoringErrorType.IDP_ERROR, exception)
        })
    }
}