package com.liveperson.monitoring.managers

import android.content.Context
import com.liveperson.infra.errors.ErrorCode.*
import com.liveperson.infra.log.LPLog
import com.liveperson.monitoring.Command
import com.liveperson.monitoring.MonitoringFactory
import com.liveperson.monitoring.cache.MonitoringParamsCache
import com.liveperson.monitoring.requests.CSDSRequest
import com.liveperson.monitoring.sdk.callbacks.IMonitoringCallback
import com.liveperson.monitoring.sdk.callbacks.MonitoringErrorType
import com.liveperson.monitoring.sdk.responses.LPSdeResponse
import com.liveperson.monitoring.utils.DomainBuilder

/**
 * Created by nirni on 12/7/17.
 */
class MonitoringRequestManager(val brandId : String?, val paramsCache: MonitoringParamsCache) {

    companion object {
        private const val TAG = "MonitoringRequestManager"

        const val TIME_24H_IN_MILLIS = 1000 * 60 * 60 * 24 // TODO: 12/4/17 Get from resources
    }

    /**
     * This sends the given request, but first checks whether the CSDS needs to be updated. If so it first request CSDS data
     * and after data receives sends the given request.
     */
    fun csdsDependantRequest(context: Context, request : Command, callback: IMonitoringCallback<out LPSdeResponse, Exception>) {

        val now = System.currentTimeMillis()
        val expireTime = paramsCache.lastCsdsUpdateTimestamp + TIME_24H_IN_MILLIS

        LPLog.d(TAG, "Current time: $now, last CSDS update time: ${paramsCache.lastCsdsUpdateTimestamp}, expire time: ${expireTime}")

        // If no CSDS data is available, or the data is old
        if (!paramsCache.isCsdsFilled() || (now > expireTime)) {

            LPLog.d(TAG, "CSDS data does not exist or expired. Call CSDS")

            // Get domain for the given brandId
            val csdsDomain = DomainBuilder.getCsdsDomain(context, brandId)

            CSDSRequest(csdsDomain, brandId, object : IMonitoringCallback<HashMap<String, String>, Exception> {

                override fun onSuccess(csdsDomainMap: HashMap<String, String>?) {

                    if (csdsDomainMap != null) {
                        LPLog.i(TAG, csdsDomainMap.toString())

                        // Update the CSDS data to cache
                        paramsCache.updateCsdsDomains(csdsDomainMap)

                        // Init Loggos
// TODO: 2/5/18 Check that indeed not needed
//                        brandId ?.let {initLoggos(brandId)}

                        // Request
                        LPLog.d(TAG, "Calling the next request...")
                        request.execute()

                    } else {
                        LPLog.e(TAG, ERR_0000000D, "csdsDependantRequest: Received empty CSDS domain map")
                        MonitoringFactory.monitoring.postOnMainThread(Runnable {
                            callback.onError(MonitoringErrorType.CSDS_ERROR, null)
                        })

                    }
                }

                override fun onError(errorType: MonitoringErrorType, e: Exception?) {
                    if (e != null) {
                        LPLog.e(TAG, ERR_0000000E, "onError: $errorType", e)
                    }
                    MonitoringFactory.monitoring.postOnMainThread(Runnable {
                        callback.onError(MonitoringErrorType.CSDS_ERROR, e)
                    })
                }

            }).execute()
        } else { // No update for CSDS is needed

            LPLog.d(TAG, "CSDS data exist in cache. Proceed to request")
            request.execute()
        }
    }
}
