package com.liveperson.monitoring.requests

import androidx.annotation.VisibleForTesting
import com.liveperson.infra.ICallback
import com.liveperson.infra.errors.ErrorCode
import com.liveperson.infra.errors.ErrorCode.*
import com.liveperson.infra.log.LPLog
import com.liveperson.infra.network.http.HttpHandler
import com.liveperson.infra.network.http.request.HttpGetRequest
import com.liveperson.infra.otel.LPTraceType
import com.liveperson.infra.utils.LocaleUtils
import com.liveperson.monitoring.Command
import com.liveperson.monitoring.MonitoringFactory
import com.liveperson.monitoring.cache.MonitoringParamsCache
import com.liveperson.monitoring.model.ModelKeyNames
import org.json.JSONObject

/**
 * Created by abembalkar on 06/27/2019.
 * A request to fetch engagement language
 */

class AcCdnRequest(val campaignId: String?, val engagementId: String?, val engagementRevision: String?): Command {

    companion object {
        private const val TAG = "ACCDNRequest"
        private const val ACCDN_URL = "https://%1\$s/api/account/%2\$s/configuration/le-campaigns/campaigns/%3\$s/engagements/%4\$s/revision/%5\$s/?v=3.0"
    }

    private var mParamsCache = MonitoringFactory.monitoring.paramsCache

    override fun execute() {

        val acCdnRequestUrl = getRequestUrl()
        LPLog.d(TAG, "acCdn request url: $acCdnRequestUrl")

        val httpGetRequest = HttpGetRequest(acCdnRequestUrl,
            LPTraceType.ACCDN_ENGAGEMENT_REQ)

        httpGetRequest.callback = object: ICallback<String, Exception> {
            override fun onSuccess(response: String) {
                LPLog.d(TAG, "AcCdn response: $response")
                parseAcCdnResponse(response)
            }

            override fun onError(exception: Exception) {
                LPLog.e(TAG, ERR_00000050, "Failed to get engagement language.", exception)
            }
        }

        HttpHandler.execute(httpGetRequest)
    }

    /**
     * Parse acCdn response to get engagement language.
     */
    fun parseAcCdnResponse(acCdnResponse: String) {
        try {
            val acCdnJSON = JSONObject(acCdnResponse)
            val language = acCdnJSON.getString(ModelKeyNames.LANGUAGE)
            LocaleUtils.getInstance().setEngagementLanguageCode(language)

            // check if it has attached configured window - welcome message
            val windowId = acCdnJSON.optString(ModelKeyNames.WINDOW_ID)
            if (windowId.isNotEmpty()) {
                getConfiguredWelcomeMessage(windowId)
            }
        } catch (exception: Exception) {
            LPLog.e(TAG, ERR_00000051, "Failed to parse engagement language.", exception)
        }
    }

    private fun getConfiguredWelcomeMessage(windowId: String) {
        GetWelcomeMessageRequest(windowId).execute()
    }

    fun getRequestUrl(): String = String.format(ACCDN_URL, mParamsCache?.acCdnDomain, mParamsCache?.brandId, campaignId, engagementId, engagementRevision)

    @VisibleForTesting
    fun setParamsCache(paramsCache: MonitoringParamsCache) {
        mParamsCache = paramsCache
    }

}
