package com.liveperson.monitoring.requests

import com.liveperson.infra.ICallback
import com.liveperson.infra.errors.ErrorCode.*
import com.liveperson.infra.log.LPLog
import com.liveperson.infra.model.LPWelcomeMessage
import com.liveperson.infra.network.http.HttpHandler
import com.liveperson.infra.network.http.request.HttpGetRequest
import com.liveperson.infra.otel.LPTraceType
import com.liveperson.monitoring.Command
import com.liveperson.monitoring.MonitoringFactory
import com.liveperson.monitoring.ccuiConfig.toMessageOptions
import org.json.JSONObject

/**
 * A request to fetch configured welcome message from CCUI based on given windowId
 */
class GetWelcomeMessageRequest(private val windowId: String) : Command {

    companion object {
        private const val TAG = "GetWelcomeMessageRequest"
        private const val WINDOW_CONFIG_URL =
            "https://%1\$s/api/account/%2\$s/configuration/engagement-window/window-confs/%3\$s?v=1.0"
        // https://va-a.ac.liveperson.net/api/account/63644995/configuration/engagement-window/window-confs/4722424232?v=1.0

        private const val IS_ENABLE_KEY = "isEnabled"
        private const val WHEN_TO_PRESENT_KEY = "whenToPresent"
        private const val FREQUENCY_EVERY_TIME_VALUE = "EVERY_TIME"
        private const val TEXT_KEY = "text"
        private const val QUICK_REPLIES_KEY = "quickReplies"
        private const val JSON_KEY = "json"
        private const val WELCOME_MESSAGE_KEY = "welcomeMessage"
    }

    private var mParamsCache = MonitoringFactory.monitoring.paramsCache

    override fun execute() {
        val windowConfUrl = getRequestUrl()
        LPLog.d(TAG, "windowConfig request url: $windowConfUrl")

        val httpGetRequest = HttpGetRequest(
            windowConfUrl,
            LPTraceType.ACCDN_WINDOW_CONFIG_REQ
        )

        httpGetRequest.callback = object : ICallback<String, Exception> {
            override fun onSuccess(response: String) {
                LPLog.d(TAG, "windowConfUrl response: $response")
                parseWindowConfigResponse(response)
            }

            override fun onError(exception: Exception) {
                LPLog.e(TAG, ERR_0000016E, "Failed to get windowConf welcome message.", exception)
            }
        }

        HttpHandler.execute(httpGetRequest)
    }

    /**
     * Parse response to get configured welcome message.
     * Store welcome message in monitoring object whether it has value or not (null)
     */
    private fun parseWindowConfigResponse(windowConfigResponse: String) {
        try {
            var welcomeMessage : LPWelcomeMessage? = null

            val windowConfigJson = JSONObject(windowConfigResponse)
            val welcomeMsgJson =
                windowConfigJson?.optJSONObject(JSON_KEY)?.optJSONObject(WELCOME_MESSAGE_KEY)
            LPLog.d(TAG, "welcomeMessage: ${welcomeMsgJson?.toString()}")

            if (welcomeMsgJson != null) {
                welcomeMessage = parseWelcomeMessageFromJson(welcomeMsgJson)
            }

            if (mParamsCache?.brandId != null) {
                MonitoringFactory.monitoring.welcomeMessageMap[mParamsCache?.brandId!!] = welcomeMessage
                LPLog.d(TAG, "Stored welcome message ($welcomeMessage) from ccui")
            }

        } catch (exception: Exception) {
            LPLog.e(TAG, ERR_0000016F, "Failed to parse configured welcome message.", exception)
        }
    }

    /**
     * parse json object to get return welcome message if it's valid
     * - if it's not enable, return null
     */
    private fun parseWelcomeMessageFromJson(wmJson: JSONObject): LPWelcomeMessage? = with(wmJson) {
        if (!wmJson.optBoolean(IS_ENABLE_KEY)) {
            return null
        }

        val welcomeMessage = LPWelcomeMessage(optString(TEXT_KEY)).apply {
            messageFrequency = if (FREQUENCY_EVERY_TIME_VALUE == optString(WHEN_TO_PRESENT_KEY, ""))
                LPWelcomeMessage.MessageFrequency.EVERY_CONVERSATION else LPWelcomeMessage.MessageFrequency.FIRST_TIME_CONVERSATION
            messageOptions = optJSONArray(QUICK_REPLIES_KEY)?.toMessageOptions() ?: emptyList()
        }

        return welcomeMessage
    }

    private fun getRequestUrl(): String =
        String.format(WINDOW_CONFIG_URL, mParamsCache?.acCdnDomain, mParamsCache?.brandId, windowId)
}
