package com.liveperson.monitoring.connection.tasks

import com.liveperson.infra.ICallback
import com.liveperson.infra.auth.LPAuthenticationParams
import com.liveperson.infra.errors.ErrorCode
import com.liveperson.infra.log.LPLog
import com.liveperson.infra.network.http.requests.ACCDNConnectorsRequest
import com.liveperson.monitoring.MonitoringFactory
import com.liveperson.monitoring.cache.MonitoringParamsCache
import com.liveperson.monitoring.connection.MonitoringTaskHandler
import com.liveperson.monitoring.sdk.callbacks.IMonitoringCallback
import com.liveperson.monitoring.sdk.callbacks.MonitoringErrorType
import com.liveperson.monitoring.sdk.responses.LPSdeResponse
import org.json.JSONArray

class ACCDNConnectorsTask(val brandId: String,
                          val lpAuthenticationParams: LPAuthenticationParams,
                          val paramsCache: MonitoringParamsCache,
                          val callback: IMonitoringCallback<out LPSdeResponse, Exception>,
                          val nextTask: MonitoringTaskHandler) : MonitoringTaskHandler {

    companion object {
        private const val TAG = "ACCDNConnectorsTask"
        private const val CONFIGURATION_KEY = "configuration"
        private const val ISSUER_DISPLAY_NAME_KEY = "issuerDisplayName"
        private const val ID_KEY = "id"
    }

    override fun execute(args: Map<String, Any?>) {
        if (getIssuerName() == null) {
            nextTask.execute()
            return
        }

        LPLog.i(TAG, "executeNextTask: Starting $TAG")
        val certificates: List<String>? = lpAuthenticationParams.certificatePinningKeys
        val acCdnDomain = paramsCache.acCdnDomain.takeIf {
            !it.isNullOrEmpty()
        } ?: run {
            onTaskError(Exception("Domain not found"))
            return
        }

        ACCDNConnectorsRequest(acCdnDomain, brandId, certificates, object : ICallback<String, Exception> {
            override fun onSuccess(value: String?) {
                LPLog.i(TAG, "Successfully got connector information: $value")
                if (!value.isNullOrEmpty()) {
                    try {
                        val connectorId = extractConnectorId(JSONArray(value))
                        nextTask.execute(mapOf("connectorId" to connectorId))
                    } catch (exception: Exception) {
                        onTaskError(exception)
                    }
                } else {
                    // If no connector details found, we can still run default Idp request without connectorId
                    nextTask.execute()
                }
            }

            override fun onError(exception: Exception?) {
                onTaskError(exception)
            }

        }).execute()

    }

    private fun onTaskError(exception: Exception?) {
        LPLog.e(TAG, ErrorCode.ERR_00000163, "Failed to get connector information: $exception")
        MonitoringFactory.monitoring.postOnMainThread(Runnable {
            callback.onError(MonitoringErrorType.ACCDN_ERROR, exception)
        })
    }

    /**
     * Extract auth connector Id from response data.
     */
    private fun extractConnectorId(data: JSONArray): String? {
        val issuerDisplayName = getIssuerName()

        for (i in 0 until data.length()) {
            val connector = data.getJSONObject(i)

            if (connector.has(ID_KEY) && connector.has(CONFIGURATION_KEY)) {
                val configuration = connector.getJSONObject(CONFIGURATION_KEY)
                if (issuerDisplayName != null && configuration.has(ISSUER_DISPLAY_NAME_KEY) &&
                    issuerDisplayName.equals(configuration.getString(ISSUER_DISPLAY_NAME_KEY), true)
                ) {

                    val connectorId = connector.get(ID_KEY).toString()
                    LPLog.i(TAG, "Found valid connector id of $issuerDisplayName: $connectorId")
                    return connectorId
                }
            }
        }
        return null
    }

    /**
     * Get Issuer Display Name if it's available
     * return null if it's not
     */
    private fun getIssuerName(): String? = lpAuthenticationParams.issuerDisplayName?.takeIf {
            it.isNotBlank()
    }
}