package com.liveperson.monitoring.requests;

import android.text.TextUtils;


import com.liveperson.infra.ICallback;
import com.liveperson.infra.log.LPLog;
import com.liveperson.infra.network.http.HttpHandler;
import com.liveperson.infra.network.http.request.HttpGetRequest;
import com.liveperson.infra.otel.LPTraceType;
import com.liveperson.monitoring.Command;
import com.liveperson.monitoring.sdk.callbacks.MonitoringErrorType;
import com.liveperson.monitoring.sdk.callbacks.IMonitoringCallback;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.util.HashMap;

/**
 * Created by Ilya Gazman on 11/3/2015.
 * <p/>
 * A request to fetch CSDS
 */
public class CSDSRequest implements Command {

    private static final String TAG = CSDSRequest.class.getSimpleName();
    private static final String CSDS_URL = "https://%1$s/csdr/account/%2$s/service/baseURI.json?version=1.0";
    private static final int CSDS_REQUEST_TIMEOUT = 30000;
    private IMonitoringCallback mCallback;
    private String mBrandId;
    private String mDomain;

    public CSDSRequest(String domain, String brandId, IMonitoringCallback callback) {
        mCallback = callback;
        mBrandId = brandId;
        mDomain = domain;
    }


    /**
     * Execute CSDS request
     */
    @Override
    public void execute() {

        final String cSDSRequestUrl = String.format(CSDS_URL, mDomain, mBrandId);
        HttpGetRequest httpGetRequest = new HttpGetRequest(cSDSRequestUrl,
                LPTraceType.CSDS_DOMAINS_REQ);
        httpGetRequest.setTimeout(CSDS_REQUEST_TIMEOUT);
        httpGetRequest.setCallback(new ICallback<String, Exception>() {
            @Override
            public void onSuccess(String csdsString) {
                if(!TextUtils.isEmpty(csdsString)){
                    try{
                        JSONObject csdsJson = new JSONObject(csdsString);
                        JSONArray csdsResponse = csdsJson.optJSONArray("baseURIs");
                        if (csdsResponse.length() == 0) {
                            mCallback.onError(MonitoringErrorType.CSDS_ERROR, new Exception( "CSDS response: Brand not found. url = " + cSDSRequestUrl));
                        } else {
                            HashMap<String, String> mCSDSUrls = parseCSDSResponse(csdsResponse);
                            mCallback.onSuccess(mCSDSUrls);
                        }
                    } catch (JSONException je) {

                    }
                }
                else {
					mCallback.onError(MonitoringErrorType.CSDS_ERROR, new Exception( "CSDS response: csdsString is empty"));
				}
            }

            @Override
            public void onError(Exception exception) {
                mCallback.onError(MonitoringErrorType.CSDS_ERROR, new Exception("CSDS failed! url = " + cSDSRequestUrl +". error: " + exception.getMessage()));
            }
        });
        HttpHandler.execute(httpGetRequest);
    }

    /**
     * Parsing server data
     *
     * @param cSDSResponse
     * @return
     */
    private HashMap<String, String> parseCSDSResponse(JSONArray cSDSResponse) throws JSONException {
        HashMap<String, String> mCSDSUrls = new HashMap<>(cSDSResponse.length());
        for (int i = 0; i < cSDSResponse.length(); i++) {
            JSONObject service = cSDSResponse.optJSONObject(i);
            String serviceName = service.optString("service");
            String baseURIName = service.optString("baseURI");
            LPLog.INSTANCE.d(TAG, "serviceName: " + serviceName + " : " + baseURIName);
            mCSDSUrls.put(serviceName, baseURIName);
        }

        return mCSDSUrls;
    }
}
