package com.liveperson.infra.ui.view.uicomponents;

import android.content.Context;
import android.graphics.Bitmap;
import android.graphics.PorterDuff;
import android.os.AsyncTask;
import android.os.Build;
import android.text.Editable;
import android.text.Html;
import android.text.TextUtils;
import android.text.TextWatcher;
import android.util.AttributeSet;
import android.util.DisplayMetrics;
import android.util.TypedValue;
import android.view.KeyEvent;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.view.WindowManager;
import android.view.animation.Animation;
import android.view.animation.AnimationUtils;
import android.view.inputmethod.EditorInfo;
import android.webkit.URLUtil;
import android.widget.Button;
import android.widget.EditText;
import android.widget.ImageButton;
import android.widget.ImageSwitcher;
import android.widget.ImageView;
import android.widget.LinearLayout;
import android.widget.TextView;
import android.widget.Toast;
import android.widget.ViewSwitcher;

import com.liveperson.infra.ForegroundService;
import com.liveperson.infra.configuration.Configuration;
import com.liveperson.infra.log.LPMobileLog;
import com.liveperson.infra.managers.PreferenceManager;
import com.liveperson.infra.ui.R;
import com.liveperson.infra.ui.view.utils.ImageUrlUtil;
import com.liveperson.infra.utils.ImageUtils;
import com.liveperson.infra.utils.LinkPreviewCallback;
import com.liveperson.infra.utils.SourceContent;
import com.liveperson.infra.utils.TextCrawler;

import java.util.ArrayList;
import java.util.List;

/**
 * Layout used in the bottom of the conversation fragment
 * includes an EditText and a send button
 */
public abstract class BaseEnterMessage extends LinearLayout implements IConnectionChangedCustomView,IUpdateFromBrand{

	private static final String TAG = BaseEnterMessage.class.getSimpleName();
	private static final int ATTACH_ICON_RESOURCE = R.drawable.lpinfra_ui_ic_attach;
	public static final int RECORDING_LAYOUT_ANIMATION_DURATION_MILLIS = 200;

	private EditText mTextInput;
	protected Button mTextSendButton;
	protected ImageButton mImageSendButton;
	private ImageSwitcher mAttachBtn;
	protected ImageButton mTrashButton;
	private View mDivider;
	protected ViewSwitcher mEnterMessageLayoutSwitcher;

	protected IConversationProvider mBrandIdProvider;
	protected IEnterMessageListener mEnterMessageListener;
	private Animation mAnimationOut;
	private Animation mAnimationIn;
	protected boolean mIsConnected;
	protected boolean mIsUpdated;
	private boolean mPhotoSharingKillSwitchEnabled;
	private boolean mPhotoSharingSiteSettingsEnabled;
	private IOverflowMenu mOverflowMenu;

	// views for link preview ui
	private ViewGroup mDropPreview;
	private LinearLayout mLinearLayout;
	private TextCrawler mTextCrawler;
	private String mCurrentTitle, mCurrentDescription, mCurrentImageURL, mCurrentUrl, mSiteName, mCurrentText;
	private Bitmap mCurrentImage;
	private boolean mIsRealTimePreviewEnabled, mIsLinkPreviewEnabled, mIsSufficientToDisplayLinkPreview, mIsDivider;

	private enum InputState {HAS_TEXT, NONE}

	private InputState mInputState = InputState.NONE;

	/**
	 * @param context
	 */
	public BaseEnterMessage(Context context) {
		super(context);
		init(context);
	}

	/**
	 * @param context
	 * @param attrs
	 */
	public BaseEnterMessage(Context context, AttributeSet attrs) {
		super(context, attrs);
		init(context);
	}

	/**
	 * @param context
	 * @param attrs
	 * @param defStyleAttr
	 */
	public BaseEnterMessage(Context context, AttributeSet attrs, int defStyleAttr) {
		super(context, attrs, defStyleAttr);
		init(context);
	}

	/**
	 * @param context
	 */
	private void init(Context context) {
		LayoutInflater.from(context).inflate(R.layout.lpinfra_ui_enter_message_layout, this);
	}

	/**
	 * @param provider
	 */
	public void setBrandIdProvider(IConversationProvider provider) {
		mBrandIdProvider = provider;
	}


	@Override
	protected void onFinishInflate() {
		super.onFinishInflate();
		mTextInput = findViewById(R.id.lpui_enter_message_text);
		mTextSendButton = findViewById(R.id.lpui_enter_message_send);
		mImageSendButton = findViewById(R.id.lpui_enter_message_send_button);
		mAttachBtn = findViewById(R.id.lpui_attach_file);
		mTrashButton = findViewById(R.id.lpui_voice_trash_button);
		mDivider = findViewById(R.id.lpui_enter_message_divider);
		mEnterMessageLayoutSwitcher = findViewById(R.id.lpui_enter_view_switcher);

		mIsRealTimePreviewEnabled = Configuration.getBoolean(R.bool.link_preview_enable_real_time_preview);
		mIsLinkPreviewEnabled = Configuration.getBoolean(R.bool.link_preview_enable_feature);
		mDropPreview = (ViewGroup) findViewById(R.id.lpui_drop_preview_view);
		mTextCrawler = new TextCrawler();

		setEditTextPreferences();
		setSendButtonPreferences();
		setAttachButtonPreferences();
	}


	/**
	 * Setting the edit text preferences
	 * We set those parameters here, because we want to insure that if the host app
	 * overrides the edit text implementation, they need to define the least needed
	 */
	private void setEditTextPreferences() {
		mTextInput.setHint(R.string.lp_enter_message);
		mTextInput.setImeOptions(EditorInfo.IME_ACTION_SEND);
		mTextInput.setInputType(EditorInfo.TYPE_CLASS_TEXT | EditorInfo.TYPE_TEXT_FLAG_IME_MULTI_LINE | EditorInfo.TYPE_TEXT_FLAG_CAP_SENTENCES);
		mTextInput.setSingleLine(false);
		mTextInput.setMaxLines(3);
		mTextInput.setTextSize(TypedValue.COMPLEX_UNIT_SP, 14);
		mTextInput.setTextColor(Configuration.getColor(R.color.lp_enter_msg_text));
		mTextInput.setHintTextColor(Configuration.getColor(R.color.lp_enter_msg_hint));
		mTextInput.setLinksClickable(false);

		mTextInput.setOnEditorActionListener(new TextView.OnEditorActionListener() {
			@Override
			public boolean onEditorAction(TextView v, int actionId, KeyEvent event) {
				if (actionId == EditorInfo.IME_NULL && event.getAction() == KeyEvent.ACTION_DOWN) {
					sendMessage();
				} else if (actionId == EditorInfo.IME_ACTION_SEND) {
					sendMessage();
				}
				return true;
			}
		});
		mTextInput.cancelLongPress();

		mTextInput.addTextChangedListener(new TextWatcher() {

			public void afterTextChanged(Editable s) {
				onAfterChangedText(s.toString());
			}

			public void beforeTextChanged(CharSequence s, int start, int count, int after) {
				onBeforeChangedText();
			}

			public void onTextChanged(CharSequence s, int start, int before, int count) {
				String msgStr = mTextInput.getText().toString().trim();
				mCurrentText = msgStr;

				if (!TextUtils.isEmpty(msgStr)) {
					if (mIsLinkPreviewEnabled) {
						generateMessageWithURL(msgStr);
					}
					if (mInputState != InputState.HAS_TEXT) {
						mInputState = InputState.HAS_TEXT;
						onHasText(true);
						updateSendButtonState();
					}
				} else {
					mCurrentUrl = "";
					clean();
					if (mInputState == InputState.HAS_TEXT) {
						mInputState = InputState.NONE;
						onHasText(false);
						updateSendButtonState();
					}

				}
			}
		});
	}

	/**
	 * Setting the send button preferences
	 */
	private void setSendButtonPreferences() {

		// Select between Sent text button and Send image button
		if (Configuration.getBoolean(R.bool.use_send_image_button)) {
			mImageSendButton.setVisibility(VISIBLE);
			mTextSendButton.setVisibility(GONE);
			mImageSendButton.setOnClickListener(new OnClickListener() {
				@Override
				public void onClick(View v) {
					sendMessage();
				}
			});
		} else {
			mImageSendButton.setVisibility(GONE);
			mTextSendButton.setVisibility(VISIBLE);
			mTextSendButton.setOnClickListener(new OnClickListener() {
				@Override
				public void onClick(View v) {
					sendMessage();
				}
			});
		}
		setEnterTextMinWidth();
	}

	/**
	 *
	 */
	private void setEnterTextMinWidth() {
		DisplayMetrics metrics = new DisplayMetrics();
		WindowManager wm = (WindowManager) getContext().getSystemService(Context.WINDOW_SERVICE);
		wm.getDefaultDisplay().getMetrics(metrics);
		mTextSendButton.setMaxWidth(metrics.widthPixels / 2);
	}

	/**
	 * Setting the attach button preferences
	 */
	private void setAttachButtonPreferences() {
		// Get the general configuration key for enabling photo sharing
		mPhotoSharingKillSwitchEnabled = PreferenceManager.getInstance().getBooleanValue(PreferenceManager.KILL_SWITCH_PHOTO_SHARING_ENABLED_PREFERENCE_KEY, PreferenceManager.APP_LEVEL_PREFERENCES, true);
		mPhotoSharingSiteSettingsEnabled = PreferenceManager.getInstance().getBooleanValue(PreferenceManager.SITE_SETTINGS_PHOTO_SHARING_ENABLED_PREFERENCE_KEY, PreferenceManager.APP_LEVEL_PREFERENCES, true);

		// Disable photo sharing if set to false
		if (!Configuration.getBoolean(R.bool.enable_photo_sharing) || !mPhotoSharingKillSwitchEnabled || !mPhotoSharingSiteSettingsEnabled) {
			mAttachBtn.setVisibility(GONE);
		}

		mAnimationOut = AnimationUtils.loadAnimation(getContext(), R.anim.menu_icon_amination_out);
		mAnimationIn = AnimationUtils.loadAnimation(getContext(), R.anim.menu_icon_amination_in);

		mAttachBtn.setFactory(new ViewSwitcher.ViewFactory() {
			public View makeView() {
				ImageView imageView = new ImageView(getContext());
				imageView.setScaleType(ImageView.ScaleType.CENTER);
				imageView.setLayoutParams(new ImageSwitcher.LayoutParams(LayoutParams.MATCH_PARENT, LayoutParams.MATCH_PARENT));
				return imageView;
			}
		});

		mAttachBtn.setImageResource( R.drawable.lpinfra_ui_ic_attach);
		disableAttachButton();

		mAttachBtn.setOnClickListener(new OnClickListener() {

			private ICloseMenuListener menuListener = new ICloseMenuListener() {
				@Override
				public void onCloseMenu() {
					mAttachBtn.setImageResource( R.drawable.lpinfra_ui_ic_attach);
					mAttachBtn.setContentDescription(getResources().getString(R.string.lp_accessibility_attachment_menu_button_collapsed));
					requestFocus();
				}
			};

			@Override
			public void onClick(View v) {
				if (mOverflowMenu != null) {

					if (mOverflowMenu.isMenuOpen()) {
						//close the menu, it'll call the onCloseMenu listener and will put the attach icon.
						mOverflowMenu.hide();
					} else {
						// if the link preview layout is visible close it
						if (mDropPreview != null) {
							mDropPreview.setVisibility(GONE);
						}
						//show the menu and put the X icon.
						mAttachBtn.setImageResource(R.drawable.lpinfra_ui_ic_close);
						mAttachBtn.setContentDescription(getResources().getString(R.string.lp_accessibility_attachment_menu_button_expanded));
						mOverflowMenu.show();
						mOverflowMenu.setOnCloseListener(menuListener);
					}
				}

			}
		});
	}

	/**
	 * Disable the attach button
	 */
	private void disableAttachButton() {
		mAttachBtn.setEnabled(false);
		mAttachBtn.setAlpha(0.5f);
	}

	/**
	 * @param msgStr
	 */
	private void generateMessageWithURL(String msgStr) {
		String[] urls = extractLinks(msgStr);
		if (urls.length > 0) {
			mTextCrawler.makePreview(callback, urls[0]);
		} else {
			mCurrentUrl = "";
		}
	}


	/**
	 * extractLinks - a method to extract only the url from the whole message
	 * for later parse
	 *
	 * @param text - the original msg
	 * @return String[] of all the links
	 */
	public static String[] extractLinks(String text) {
		List<String> links = new ArrayList<>();
		String[] l = text.split("\\s+");
		for (int i = 0; i < l.length; i++) {
			if (URLUtil.isValidUrl(l[i])) {
				links.add(l[i]);
			}
		}
		return links.toArray(new String[links.size()]);

	}


	/**
	 * Callback to update the preview view.
	 */
	private LinkPreviewCallback callback = new LinkPreviewCallback() {
		@Override
		public void onPre() {
		}

		/**
		 * validateSufficientConditionsToDisplayPreview
		 * Only in case of at least 2 OG: tags we will show and send a url message
		 */
		private boolean validateSufficientConditionsToDisplayPreview(SourceContent sourceContent) {
			int counter = 0;
			counter = sourceContent.getSiteName().isEmpty() ? --counter : ++counter;
			counter = sourceContent.getDescription().isEmpty() ? --counter : ++counter;
			counter = sourceContent.getTitle().isEmpty() ? --counter : ++counter;
			counter = sourceContent.getImages().isEmpty() ? --counter : ++counter;

			LPMobileLog.d(TAG, "**** URL Conditions " +
					"Site = " + !sourceContent.getSiteName().isEmpty() +
					" Description = " + !sourceContent.getDescription().isEmpty() +
					" Title = " + !sourceContent.getTitle().isEmpty() +
					" Images = " + !sourceContent.getImages().isEmpty());

			return mIsSufficientToDisplayLinkPreview = (counter >= 0);
		}

		@Override
		public void onPos(final SourceContent sourceContent, boolean isNull) {
			// if parsing failed - show error message
			if (isNull || sourceContent == null || TextUtils.isEmpty(sourceContent.getFinalUrl())) {
				mDropPreview.setVisibility(GONE);
				BaseEnterMessage.this.clean();
				mCurrentUrl = "";

			} else {
				if (mTextInput.getText().toString().trim().isEmpty() || !mCurrentText.equals(mTextInput.getText().toString().trim())) {
					return;
				}
				if ( validateSufficientConditionsToDisplayPreview(sourceContent) && mIsRealTimePreviewEnabled) {
					final ImageView imageSet = (ImageView) mDropPreview.findViewById(R.id.image_post_set);
					final TextView titleTextView = (TextView) mDropPreview.findViewById(R.id.title);
					final TextView descriptionTextView = (TextView) mDropPreview.findViewById(R.id.description);
					final ImageButton closeBtn = (ImageButton) mDropPreview.findViewById(R.id.close_btn);
					imageSet.setImageDrawable(null);
					mDropPreview.setVisibility(VISIBLE);

					// click on the close button will
					closeBtn.setOnClickListener(new OnClickListener() {
						@Override
						public void onClick(View v) {
							clean();
							mDropPreview.setVisibility(GONE);
						}
					});

					// this is to fix a bug MOB-7097
					sourceContent.setImages(ImageUrlUtil.completeToValidImageUrl(sourceContent.getFinalUrl(), sourceContent.getImages()));

					// set the data (images + texts) in the layout
					AsyncTask.execute(new Runnable() {
						@Override
						public void run() {

							if (sourceContent.getImages().isEmpty()) {
								LPMobileLog.d(TAG, "image isEmpty()");

								imageSet.post(new Runnable() {
									@Override
									public void run() {
										imageSet.setVisibility(GONE);
										imageSet.setImageDrawable(null);
									}
								});
								return;
							}
							mCurrentImage = ImageUtils.createBitmapFromURL(BaseEnterMessage.this.getContext(), sourceContent.getImages(), imageSet.getWidth(), imageSet.getHeight());
							if (mCurrentImage != null) {
								LPMobileLog.d(TAG, "image mCurrentUrl != null");

								imageSet.post(new Runnable() {
									@Override
									public void run() {
										imageSet.setImageBitmap(mCurrentImage);
										imageSet.setVisibility(VISIBLE);
									}
								});
							} else {
								LPMobileLog.d(TAG, "Exception in loading image ");
								imageSet.post(new Runnable() {
									@Override
									public void run() {
										imageSet.setVisibility(GONE);
										imageSet.setImageDrawable(null);
									}
								});
							}
						}
					});
					if (Build.VERSION.SDK_INT >= 24) {
						titleTextView.setText(Html.fromHtml(sourceContent.getTitle(), Html.FROM_HTML_MODE_COMPACT));
						descriptionTextView.setText(Html.fromHtml(sourceContent.getDescription(), Html.FROM_HTML_MODE_COMPACT));
					} else {
						titleTextView.setText(Html.fromHtml(sourceContent.getTitle()));
						descriptionTextView.setText(Html.fromHtml(sourceContent.getDescription()));
					}
				}
				// save the data to the local members for later use
				mCurrentTitle = sourceContent.getTitle();
				mCurrentDescription = sourceContent.getDescription();
				mCurrentUrl = sourceContent.getUrl();
				mCurrentImageURL = sourceContent.getImages();
				mSiteName = sourceContent.getSiteName();

				LPMobileLog.i(TAG, "**** URL finished parsing: mCurrentUrl " + mCurrentUrl +
						" mIsRealTimePreviewEnabled: " + mIsRealTimePreviewEnabled +
						" mIsSufficientToDisplayLinkPreview: " + mIsSufficientToDisplayLinkPreview +
						" ***");

			}
		}

		@Override
		public void clean() {
			LayoutInflater inflater = (LayoutInflater) BaseEnterMessage.this.getContext().getSystemService(Context.LAYOUT_INFLATER_SERVICE);
			final View content = inflater.inflate(R.layout.lpinfra_ui_enter_message_preview_content_layout, mLinearLayout);
			final ImageView imageSet = (ImageView) content.findViewById(R.id.image_post_set);
			imageSet.setImageDrawable(null);
			imageSet.setVisibility(GONE);
		}
	};

	private void clean() {
		mCurrentTitle = mCurrentDescription = mCurrentUrl = mCurrentImageURL = "";
		mCurrentImage = null;

		if (callback != null) {
			callback.clean();
		}
		if (mDropPreview != null) {
			mDropPreview.setVisibility(GONE);
		}
	}

	/**
	 * Clear the edit text
	 * and the link preview area
	 */
	protected void clearText() {
		mTextInput.setText("");
		clean();
	}

	/**
	 * Show toast message when there is no connection
	 */
	protected void showNoNetworkMessage() {
		if (ForegroundService.getInstance().isBrandForeground()) {
			Toast.makeText(getContext(), R.string.lp_no_network_toast_message, Toast.LENGTH_LONG).show();
		}
	}


	/**
	 * Get the text from the input text and sends it
	 */
	protected void sendMessage() {
		String msgStr = mTextInput.getText().toString().trim();
		if (!TextUtils.isEmpty(msgStr)) {
			// if the text to send doesnt contain a url we will send a regular text message
			// also in case there isnt sufficient tag to display as a link
			if (TextUtils.isEmpty(mCurrentUrl) || !mIsSufficientToDisplayLinkPreview) {
				LPMobileLog.d(TAG, "url send Message, mIsSufficientToDisplayLinkPreview: " + mIsSufficientToDisplayLinkPreview);
				sendMessage(msgStr);
			} else {
				LPMobileLog.d(TAG, "url sendMessageWithURL");
				sendMessageWithURL(msgStr, mCurrentUrl, mCurrentTitle, mCurrentImageURL, mCurrentDescription, mSiteName);
			}
			clean();
		}
	}


	/**
	 * Setting the views as disables/enabled
	 *
	 * @param enabled
	 */
	@Override
	public void setEnabled(boolean enabled) {
		super.setEnabled(enabled);
		mTextInput.setEnabled(enabled);
		mAttachBtn.setEnabled(enabled);

		// If enabled == true, update the send button according to text box and connection state
		if (enabled) {
			updateAttachButton();
		} else {
			mTextSendButton.setEnabled(enabled);
			mImageSendButton.setEnabled(enabled);
		}
	}

	/**
	 * Update callback,to check if
	 * we are updated after the
	 * internet connection is established
	 *
	 * @param isUpdated
	 */
	@Override
	public void onUpdate(boolean isUpdated) {
		mIsUpdated = isUpdated;
		updateSendButtonState();
	}

	/**
	 * Connection callback
	 *
	 * @param isConnected
	 */
	@Override
	public void onConnectionChanged(boolean isConnected) {
		mIsConnected = isConnected;
		updateAttachButton();

		// Enable/disable attachment button according to if the PhotoSharingEnabled remote configuration was changed
		mPhotoSharingKillSwitchEnabled = PreferenceManager.getInstance().getBooleanValue(PreferenceManager.KILL_SWITCH_PHOTO_SHARING_ENABLED_PREFERENCE_KEY, PreferenceManager.APP_LEVEL_PREFERENCES, true);
		mPhotoSharingSiteSettingsEnabled = PreferenceManager.getInstance().getBooleanValue(PreferenceManager.SITE_SETTINGS_PHOTO_SHARING_ENABLED_PREFERENCE_KEY, PreferenceManager.APP_LEVEL_PREFERENCES, true);

		// Disable photo sharing if set to false
		if (!Configuration.getBoolean(R.bool.enable_photo_sharing) || !mPhotoSharingKillSwitchEnabled || !mPhotoSharingSiteSettingsEnabled) {
			mAttachBtn.setVisibility(GONE);
		} else {
			mAttachBtn.setVisibility(VISIBLE);
		}
	}

	/**
	 * Set the animation to the attach button
	 */
	private void setAnimationOnAttachButton() {
		mAttachBtn.setInAnimation(mAnimationIn);
		mAttachBtn.setOutAnimation(mAnimationOut);
	}

	/**
	 * Cancel the animation on the attach button
	 */
	private void cancelAnimationOnAttachButton() {
		mAttachBtn.setInAnimation(null);
		mAttachBtn.setOutAnimation(null);
	}


	/**
	 * Update the send button color
	 * Depends on the connection state and if the edittext contains characters
	 */
	protected void updateSendButtonState() {

		if (!shouldUpdateSendButton()) {
			return;
		}

		String msgStr = mTextInput.getText().toString().trim();
		int sendBtnColor;

		if (!TextUtils.isEmpty(msgStr) && mIsConnected && mIsUpdated) {
			sendBtnColor = R.color.lp_send_button_text_enable;
			mTextSendButton.setEnabled(true);
			mImageSendButton.setEnabled(true);
		} else if (TextUtils.isEmpty(msgStr)) {
			sendBtnColor = R.color.lp_send_button_text_disable;
			mTextSendButton.setEnabled(false);
			mImageSendButton.setEnabled(false);
			clean();
			// not empty and not connected
		} else {
			sendBtnColor = R.color.lp_send_button_text_disable;
			mTextSendButton.setEnabled(false);
			mImageSendButton.setEnabled(false);
		}
		mTextSendButton.setTextColor(Configuration.getColor(sendBtnColor));
		mImageSendButton.getDrawable().setColorFilter(Configuration.getColor(sendBtnColor), PorterDuff.Mode.SRC_IN);
	}

	protected void enableSendButton() {
		int sendBtnColor;
		sendBtnColor = R.color.lp_send_button_text_enable;
		mTextSendButton.setEnabled(true);
		mImageSendButton.setEnabled(true);
		mTextSendButton.setTextColor(Configuration.getColor(sendBtnColor));
		mImageSendButton.getDrawable().setColorFilter(Configuration.getColor(sendBtnColor), PorterDuff.Mode.SRC_IN);
	}

	/**
	 * Update the attach button according to the connection state
	 */
	private void updateAttachButton() {

		if (mIsConnected) {
			// Enable attach button
			cancelAnimationOnAttachButton();
			enableAttachButton();
			setAnimationOnAttachButton();

		} else {
			// If the attachment menu is open close it
			if (mOverflowMenu.isMenuOpen()) {
				mOverflowMenu.hide();
			}

			// Disable the attach button
			cancelAnimationOnAttachButton();
			disableAttachButton();
		}
	}

	public void setOverflowMenu(IOverflowMenu menu) {
		mOverflowMenu = menu;
		mOverflowMenu.setOnCloseListener(new ICloseMenuListener() {
			@Override
			public void onCloseMenu() {
				// TODO: 8/7/16 animate attachment icon
			}
		});
	}


	/**
	 * Enable the attach button
	 */
	private void enableAttachButton() {
		mAttachBtn.setEnabled(true);
		mAttachBtn.setAlpha(1f);
	}


	/**
	 * Set message sent callback
	 *
	 * @param enterMessageListener
	 */
	public void setEnterMessageListener(IEnterMessageListener enterMessageListener) {
		mEnterMessageListener = enterMessageListener;
	}

	/**
	 * Insert text to the input text
	 *
	 * @param msg
	 */
	public void setText(String msg) {
		if (!TextUtils.isEmpty(msg) && mTextInput != null) {
			mTextInput.setText(msg);
		} else {
			LPMobileLog.d(TAG, "Failed to set text message with text: " + msg);
		}
	}


	protected abstract void sendMessageWithURL(final String message, final String urlToParse, final String title, final String imageURL, final String description, final String siteName);


	protected abstract void sendMessage(final String message);

	/**
	 * Called when the EditText status is changed between has text or empty
	 *
	 * @param isHasText
	 */
	protected abstract void onHasText(final boolean isHasText);

	/**
	 * Called everytime a text has changed. The changed text is given
	 *
	 * @param text
	 */
	protected abstract void onAfterChangedText(String text);

	/**
	 * This is called before any change in the text is about to take place
	 */
	protected abstract void onBeforeChangedText();

	/**
	 * Indicate whether to update the send button or not
	 * @return
	 */
	protected abstract boolean shouldUpdateSendButton();

}
