/* Copyright (c) LivePerson, Inc. */
package com.liveperson.infra.ui.view.adapter.viewholder;

import android.content.Context;
import android.os.Bundle;
import android.support.v4.text.util.LinkifyCompat;
import android.support.v7.widget.RecyclerView;
import android.text.Spannable;
import android.text.TextUtils;
import android.text.util.Linkify;
import android.util.Patterns;
import android.view.View;
import android.view.accessibility.AccessibilityNodeInfo;
import android.widget.TextView;

import com.liveperson.infra.configuration.Configuration;
import com.liveperson.infra.model.Message;
import com.liveperson.infra.ui.R;
import com.liveperson.infra.utils.DateUtils;

import java.text.DateFormat;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * Created by nirni on 11/11/15.
 * This is the base view holder that all view holders that are used in the recycler view should extend
 */
public abstract class BaseViewHolder extends RecyclerView.ViewHolder {

	private int position;
	protected TextView mMessageTextView;
	protected String mTimestampAccessibilityString;
	private boolean isNumericTimestampOnly;
	protected long mTimeStamp;
	public enum LinkType {URL, PHONE, EMAIL}
	private LinkType linkType;

	/**
	 * Transform filter to return the matched text as a group
	 */
	private static Linkify.TransformFilter mFilter = new Linkify.TransformFilter() {
		public final String transformUrl(final Matcher match, String url) {
			return match.group();
		}
	};

	/**
	 * Filters out web URL matches that occur after an at-sign (@).  This is
	 * to prevent turning the domain name in an email address into a web link.
	 */
	private static Linkify.MatchFilter mUrlMatchFilter = new Linkify.MatchFilter() {
		public final boolean acceptMatch(CharSequence s, int start, int end) {
			if (start == 0) {
				return true;
			}

			if (s.charAt(start - 1) == '@') {
				return false;
			}

			return true;
		}
	};

	/**
	 * Filters our phone numbers that has a character before of after that is not ' ', '+', '#', '+'
	 */
	private static Linkify.MatchFilter mPhoneMatchFilter = new Linkify.MatchFilter() {
		public final boolean acceptMatch(CharSequence s, int start, int end) {
			if (start == 0) {
				return true;
			}

			if (s.charAt(start - 1) != ' ' || s.charAt(start - 1) != '+' || s.charAt(start - 1) != '*' || s.charAt(start - 1) != '#' ||
					s.charAt(end + 1) != ' ' || s.charAt(end + 1) != '+' || s.charAt(end + 1) != '*' || s.charAt(end + 1) != '#') {
				return false;
			}

			return true;
		}
	};

	public BaseViewHolder(View itemView) {
		super(itemView);
		mMessageTextView = (TextView) itemView.findViewById(R.id.lpui_message_text);
	}

	protected Context getContext() { return itemView != null ? itemView.getContext() : null; }

	public abstract void updateContentDescription();

	public void setTempPosition(int position) {
		this.position = position;
	}

	public void setMessageTextView(String message) {
		mMessageTextView.setText(message);
	}

	public int getTempPosition() {
		return position;
	}

	public String getTextToCopy() {
		return mMessageTextView.getText().toString();
	}

	public void setContentDescription(String message) {
		itemView.setContentDescription(message);
		itemView.setAccessibilityDelegate(new View.AccessibilityDelegate() {
			@Override
			public void onInitializeAccessibilityNodeInfo(View host, AccessibilityNodeInfo info) {
				super.onInitializeAccessibilityNodeInfo(host, info);
				info.setAccessibilityFocused(false);
			}
		});
	}

	public void setOnLongClickListener(View.OnLongClickListener onLongClickListener) {
		itemView.setLongClickable(true);
		itemView.setOnLongClickListener(onLongClickListener);
	}

	public void setOnClickListener(View.OnClickListener onClickListener) {
		itemView.setClickable(true);
		itemView.setOnClickListener(onClickListener);
	}

	protected void setMessageTextOnClickListener(View.OnClickListener onClickListener) {
		mMessageTextView.setOnClickListener(onClickListener);
	}

	protected void setMessageTextOnLongClickListener(View.OnLongClickListener onLongClickListener) {
		mMessageTextView.setOnLongClickListener(onLongClickListener);
	}

	public void setContextualBehaviorOnClick(View.OnClickListener onClickListener) {
		itemView.setClickable(true);
		onClickListener.onClick(itemView);
	}

	public boolean setContextualBehaviorOnLongClick(View.OnLongClickListener onClickListener) {
		itemView.setLongClickable(true);
		return onClickListener.onLongClick(itemView);
	}

	public void setTimestampTextView(long timestamp) {

		mTimeStamp = timestamp;
		// Set the string for accessibility
		mTimestampAccessibilityString = getAccessibilityDateTimeFormat(timestamp);
	}

	protected String getTimeFormat(long timestamp) {
		isNumericTimestampOnly = Configuration.getBoolean(R.bool.message_status_numeric_timestamp_only);
		if(!isNumericTimestampOnly) {
			long timeDiff = System.currentTimeMillis() - timestamp;
			if (timeDiff < 60 * 1000) {
				return itemView.getContext().getString(R.string.lp_message_time_now);
			}
			if (timeDiff < 60 * 1000 * 20) {
				long minutes = timeDiff / (60 * 1000);
				return String.format(itemView.getContext().getString(R.string.lp_message_time_min_ago), minutes);
			}
		}
		String customTimeFormat = itemView.getContext().getString(R.string.lp_time_format);

		return DateUtils.getFormattedTime(customTimeFormat, DateFormat.SHORT, timestamp);

	}

	/**
	 * Get the date and time format of the given timestamp.
	 * This is used for accessibility talkback usage
	 * @param timestamp
	 * @return
	 */
	protected String getAccessibilityDateTimeFormat(long timestamp) {

		String customTimeFormat = itemView.getContext().getString(R.string.lp_time_format);
		return DateUtils.getFormattedDateAndTime(customTimeFormat, DateFormat.SHORT, DateFormat.SHORT, timestamp);
	}

	/**
	 * Search and mark links in the given TextView. If a link was found this method returns true, else return false
	 * @param textView
	 * @return true - if found and marked a link, false - otherwise
	 */
	protected boolean linkifyText(TextView textView) {

		// Indicates whether the text was linkified. This is needed for accessibility purposes.
		// It's sets to true if the TextView's text was linkified
		boolean linkified = false;

		// custom link regex
		String customPhoneRegex = textView.getContext().getString(R.string.lp_bubble_phone_links_regex);
		String customUrlRegex = textView.getContext().getString(R.string.lp_bubble_url_links_regex);
		String customEmailRegex = textView.getContext().getString(R.string.lp_bubble_email_links_regex);

		if (!TextUtils.isEmpty(customPhoneRegex)) {
			Pattern customLinkPattern = Pattern.compile(customPhoneRegex);
			if (LinkifyCompat.addLinks((Spannable) textView.getText(), customLinkPattern, "tel:", null, mFilter)) {
				linkified = true;
				linkType = LinkType.PHONE;
			}
		} else {
			//default links
			// Link phone numbers
			if (LinkifyCompat.addLinks((Spannable) textView.getText(), Patterns.PHONE, "tel:", mPhoneMatchFilter, mFilter)) {
				linkified = true;
				linkType = LinkType.PHONE;
			}
		}

		if (!TextUtils.isEmpty(customUrlRegex)) {
			Pattern customLinkPattern = Pattern.compile(customUrlRegex);
			if (LinkifyCompat.addLinks((Spannable) textView.getText(), customLinkPattern, null, mUrlMatchFilter, mFilter)) {
				linkified = true;
				linkType = LinkType.URL;
			}
		} else {
			//default links
			if (LinkifyCompat.addLinks((Spannable) textView.getText(), Patterns.WEB_URL, "http://", new String[]{"https://"}, mUrlMatchFilter, mFilter)) {
				linkified = true;
				linkType = LinkType.URL;
			}
		}

		if (!TextUtils.isEmpty(customEmailRegex)) {
			Pattern customLinkPattern = Pattern.compile(customEmailRegex);
			if (LinkifyCompat.addLinks((Spannable) textView.getText(), customLinkPattern, "mailto:", null, mFilter)) {
				linkified = true;
				linkType = LinkType.EMAIL;
			}
		} else {
			//default links
			// Link email addresses
			if (LinkifyCompat.addLinks((Spannable) textView.getText(), Patterns.EMAIL_ADDRESS, "mailto:", null, mFilter)) {
				linkified = true;
				linkType = LinkType.EMAIL;
			}
		}
		// Use the default linkify if non of the custom url or email regex are set. We don't create default phone regex as it could be any random number
		if (!linkified) {
			if (LinkifyCompat.addLinks((Spannable) textView.getText(), Linkify.WEB_URLS)) {
				linkified = true;
				linkType = LinkType.URL;
			} else if (LinkifyCompat.addLinks((Spannable) textView.getText(), Linkify.EMAIL_ADDRESSES)) {
				linkified = true;
				linkType = LinkType.EMAIL;
			}
		}

		return linkified;
	}

	public void applyChanges(Bundle changes, Message message) {
		String newText = changes.getString(Message.EXTRA_MESSAGE_TEXT, null);
		if (!TextUtils.isEmpty(newText)) {
			setMessageTextView(newText);
			updateContentDescription();
		}
	}

    public void onBind() {
	}

	public void recycle() {
	}

	public LinkType getLinkType() {
		return linkType;
	}
}
