/* Copyright (c) LivePerson, Inc. */
package com.liveperson.infra.ui.view.adapter.viewholder;

import static android.util.TypedValue.COMPLEX_UNIT_SP;
import static com.liveperson.infra.ui.view.utils.UIUtils.containsHtmlTag;

import android.annotation.SuppressLint;
import android.content.Context;
import android.os.Build;
import android.os.Bundle;
import android.text.Html;
import android.text.Selection;
import android.text.Spannable;
import android.text.TextUtils;
import android.text.style.URLSpan;
import android.text.util.Linkify;
import android.util.Patterns;
import android.view.View;
import android.view.accessibility.AccessibilityNodeInfo;
import android.widget.TextView;

import androidx.core.text.util.LinkifyCompat;
import androidx.recyclerview.widget.RecyclerView;

import com.liveperson.infra.configuration.Configuration;
import com.liveperson.infra.model.Message;
import com.liveperson.infra.ui.R;
import com.liveperson.infra.ui.view.utils.ArithmeticUtilsKt;
import com.liveperson.infra.ui.view.utils.linkify.DefaultTransformFilter;
import com.liveperson.infra.ui.view.utils.linkify.PhoneMatchFilter;
import com.liveperson.infra.ui.view.utils.linkify.UrlMatchFilter;
import com.liveperson.infra.utils.DateUtils;
import com.liveperson.infra.utils.patterns.PatternsCompat;

import java.text.DateFormat;
import java.util.concurrent.TimeUnit;
import java.util.regex.Pattern;

/**
 * Created by nirni on 11/11/15.
 * This is the base view holder that all view holders that are used in the recycler view should extend
 */
public abstract class BaseViewHolder extends RecyclerView.ViewHolder implements Runnable {

	private static final long UPDATE_PERIOD = TimeUnit.MINUTES.toMillis(21);
	private static final long UPDATE_DELAY = TimeUnit.SECONDS.toMillis(15);
	private static final String TAG = "BaseViewHolder";

	protected TextView mMessageTextView;
	protected TextView mTimestampTextView;
	protected long mTimeStamp;

	public enum LinkType {URL, PHONE, EMAIL, NAVIGATION}
	private LinkType linkType;
	private OnRequestTimestampUpdateListener listener;

	/**
	 * Transform filter to return the matched text as a group
	 */
	private static Linkify.TransformFilter mFilter = new DefaultTransformFilter();

	/**
	 * Filters out web URL matches that occur after an at-sign (@).  This is
	 * to prevent turning the domain name in an email address into a web link.
	 */
	private static Linkify.MatchFilter mUrlMatchFilter = new UrlMatchFilter();

	/**
	 * Filters our phone numbers that has a character before of after that is not '+', '#', '*'
	 */
	private static Linkify.MatchFilter mPhoneMatchFilter = new PhoneMatchFilter();

	public BaseViewHolder(View itemView) {
		super(itemView);
		mMessageTextView = itemView.findViewById(R.id.lpui_message_text);
		mTimestampTextView = itemView.findViewById(R.id.lpui_message_timestamp);
		setupTimestamp();
	}

	protected Context getContext() {
		return itemView.getContext();
	}

	public abstract void updateContentDescription();

	public abstract void onUpdateTimestampText();

	public void setMessageTextView(String message) {
		if (containsHtmlTag(message)) {
			if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.N) {
				mMessageTextView.setText(Html.fromHtml(message, Html.FROM_HTML_MODE_LEGACY));
			} else {
				mMessageTextView.setText(Html.fromHtml(message));
			}
		} else {
			mMessageTextView.setText(message);
		}
	}

	public String getTextToCopy() {
		return mMessageTextView.getText().toString();
	}

	public void setContentDescription(String message) {
		itemView.setContentDescription(message);
		itemView.setAccessibilityDelegate(new View.AccessibilityDelegate() {
			@Override
			public void onInitializeAccessibilityNodeInfo(View host, AccessibilityNodeInfo info) {
				super.onInitializeAccessibilityNodeInfo(host, info);
				info.setAccessibilityFocused(false);
			}
		});
	}

	public void setOnLongClickListener(View.OnLongClickListener onLongClickListener) {
		itemView.setLongClickable(true);
		itemView.setOnLongClickListener(onLongClickListener);
	}

	public void setOnClickListener(View.OnClickListener onClickListener) {
		itemView.setClickable(true);
		itemView.setOnClickListener(onClickListener);
	}

	protected void setMessageTextOnClickListener(View.OnClickListener onClickListener) {
		mMessageTextView.setOnClickListener(onClickListener);
	}

	protected void setMessageTextOnLongClickListener(View.OnLongClickListener onLongClickListener) {
		mMessageTextView.setOnLongClickListener(onLongClickListener);
	}

	public void setContextualBehaviorOnClick(View.OnClickListener onClickListener) {
		itemView.setClickable(true);
		onClickListener.onClick(itemView);
	}

	public boolean setContextualBehaviorOnLongClick(View.OnLongClickListener onClickListener) {
		itemView.setLongClickable(true);
		return onClickListener.onLongClick(itemView);
	}

	public final void setTimestamp(long timestamp) {
		mTimeStamp = timestamp;
		onUpdateTimestampText();
	}

	protected void setupTimestamp(){
		if (mTimestampTextView != null) {
			if (isTimestampEnabled()) {
				mTimestampTextView.setVisibility(View.VISIBLE);
				mTimestampTextView.setTextSize(Configuration.getDimension(R.dimen.lp_timestamps_font_size, COMPLEX_UNIT_SP));
			} else {
				mTimestampTextView.setVisibility(View.GONE);
			}
		}
	}

	@SuppressLint("StringFormatMatches")
	protected String getTimeFormat(long timestamp) {
		boolean isNumericTimestampOnly = Configuration.getBoolean(R.bool.message_status_numeric_timestamp_only);
		if(!isNumericTimestampOnly) {
			long timeDiff = System.currentTimeMillis() - timestamp;
			if (timeDiff < 60 * 1000) {
				return itemView.getContext().getString(R.string.lp_message_time_now);
			}
			if (timeDiff < 60 * 1000 * 20) {
				long minutes = timeDiff / (60 * 1000);
				return String.format(itemView.getContext().getString(R.string.lp_message_time_min_ago), minutes);
			}
		}
		String customTimeFormat = itemView.getContext().getString(R.string.lp_time_format);

		return DateUtils.getFormattedTime(customTimeFormat, DateFormat.SHORT, timestamp);

	}

	/**
	 * Search and mark links in the given TextView. If a link was found this method returns true, else return false
	 * @param textView
	 * @return true - if found and marked a link, false - otherwise
	 */
	protected boolean linkifyText(TextView textView) {

		// Indicates whether the text was linkified. This is needed for accessibility purposes.
		// It's sets to true if the TextView's text was linkified
		boolean linkified = false;

		// custom link regex
		String customPhoneRegex = textView.getContext().getString(R.string.lp_bubble_phone_links_regex);
		String customUrlRegex = textView.getContext().getString(R.string.lp_bubble_url_links_regex);
		String customEmailRegex = textView.getContext().getString(R.string.lp_bubble_email_links_regex);

		if (!TextUtils.isEmpty(customPhoneRegex)) {
			Pattern customLinkPattern = Pattern.compile(customPhoneRegex);
			if (LinkifyCompat.addLinks((Spannable) textView.getText(), customLinkPattern, "tel:", null, mFilter)) {
				linkified = true;
				linkType = LinkType.PHONE;
			}
		} else {
			//default links
			// Link phone numbers
			if (LinkifyCompat.addLinks((Spannable) textView.getText(), Patterns.PHONE, "tel:", mPhoneMatchFilter, mFilter)) {
				linkified = true;
				linkType = LinkType.PHONE;
			}
		}

		if (!TextUtils.isEmpty(customUrlRegex)) {
			Pattern customLinkPattern = Pattern.compile(customUrlRegex);
			if (LinkifyCompat.addLinks((Spannable) textView.getText(), customLinkPattern, null, mUrlMatchFilter, mFilter)) {
				linkified = true;
				linkType = LinkType.URL;
			}
		} else {
			//default links
			if (LinkifyCompat.addLinks((Spannable) textView.getText(), PatternsCompat.AUTOLINK_WEB_URL, "http://", new String[]{"https://"}, mUrlMatchFilter, mFilter)) {
				linkified = true;
				linkType = LinkType.URL;
			}
		}

		if (!TextUtils.isEmpty(customEmailRegex)) {
			Pattern customLinkPattern = Pattern.compile(customEmailRegex);
			if (LinkifyCompat.addLinks((Spannable) textView.getText(), customLinkPattern, "mailto:", null, mFilter)) {
				linkified = true;
				linkType = LinkType.EMAIL;
			}
		} else {
			//default links
			// Link email addresses
			if (LinkifyCompat.addLinks((Spannable) textView.getText(), Patterns.EMAIL_ADDRESS, "mailto:", null, mFilter)) {
				linkified = true;
				linkType = LinkType.EMAIL;
			}
		}
		// Use the default linkify if non of the custom url or email regex are set. We don't create default phone regex as it could be any random number
		if (!linkified) {
			if (LinkifyCompat.addLinks((Spannable) textView.getText(), Linkify.WEB_URLS)) {
				linkified = true;
				linkType = LinkType.URL;
			} else if (LinkifyCompat.addLinks((Spannable) textView.getText(), Linkify.EMAIL_ADDRESSES)) {
				linkified = true;
				linkType = LinkType.EMAIL;
			}
		}

		return linkified;
	}

	public void applyChanges(Bundle changes, Message message) {
		String newText = changes.getString(Message.EXTRA_MESSAGE_TEXT, null);
		if (!TextUtils.isEmpty(newText)) {
			setMessageTextView(newText);
			updateContentDescription();
		}
	}

	protected boolean isTimestampEnabled() {
		return Configuration.getBoolean(R.bool.lp_enable_timestamps);
	}

	protected boolean isReadReceiptDisabled() {
		return !Configuration.getBoolean(R.bool.lp_enable_read_receipts);
	}

	protected boolean shouldUpdateTimestamps() {
		boolean isNumericTimestamp = Configuration.getBoolean(R.bool.message_status_numeric_timestamp_only);
		return isTimestampEnabled() && !isNumericTimestamp && Math.abs(System.currentTimeMillis() - mTimeStamp) < UPDATE_PERIOD;
	}

	public void setRequestTimestampListener(OnRequestTimestampUpdateListener listener) {
		this.listener = listener;
	}

	@Override
	public void run() {
		if (listener == null) {
			return;
		} else {
			listener.requestTimestampsUpdate(getAdapterPosition());
		}
		if (shouldUpdateTimestamps()) {
			updateTimestamp(getUpdateDelay(), this);
		}
	}

	public void onBind() {
	}

	public void observeTimestampChanges() {
		cancelUpdates(this);
		if (shouldUpdateTimestamps()) {
			updateTimestamp(0 , this);
		}
	}

	public void stopObservingTimestampsChanges() {
		cancelUpdates(this);
		listener = null;
	}

	public void recycle() {
	}

	public LinkType getLinkType() {
		return linkType;
	}

	public Long getUpdateDelay() {
		return UPDATE_DELAY;
	}

	protected void removeUrlSpans() {
		CharSequence text = mMessageTextView.getText();
		if (text instanceof Spannable) {
			Spannable spannableString = (Spannable) text;
			URLSpan[] urlSpans = spannableString.getSpans(0, spannableString.length(), URLSpan.class);
			if (urlSpans != null) {
				for (URLSpan span : urlSpans) {
					spannableString.removeSpan(span);
				}
			}
		}
	}

	protected void clearSelection() {
		CharSequence text = mMessageTextView.getText();
		if (text instanceof Spannable) {
			Selection.removeSelection((Spannable) text);
		}
	}

	protected int getTimestampAsInteger() {
		return ArithmeticUtilsKt.asIntegerValue(mTimeStamp);
	}

	private void updateTimestamp(long delay, Runnable runnable) {
		if (listener != null) {
			listener.onUpdateTimestampWithDelay(delay, runnable);
		}
	}

	private void cancelUpdates(Runnable runnable) {
		if (listener != null) {
			listener.cancelTimestampsUpdate(runnable);
		}
	}

	/**
	 * An interface to request updates for messages timestamps in runtime.
	 */
	public interface OnRequestTimestampUpdateListener {

		/**
		 * Method used to request updates of message timestamp for
		 * particular position.
		 * @param position - position of message required to be updated
		 */
		void requestTimestampsUpdate(int position);

		/**
		 * Method used to provide timestamps changes in queue.
		 * @param delay    - delay of particular changes;
		 * @param runnable - scope of works needed to be executed.
		 */
		void onUpdateTimestampWithDelay(long delay, Runnable runnable);

		/**
		 * Method used to cancel updates for particular item.
		 * @param runnable - object with scope of work needed to be canceled.
		 */
		void cancelTimestampsUpdate(Runnable runnable);
	}
}
