package com.liveperson.infra.ui.view.ui;

import android.content.Context;
import android.content.res.Resources;
import android.graphics.Canvas;
import android.graphics.Color;
import android.graphics.Paint;
import android.graphics.Path;
import android.graphics.RectF;
import android.util.AttributeSet;
import android.util.DisplayMetrics;
import android.util.TypedValue;
import android.widget.LinearLayout;

/**
 * A configurable corners radius and border width layout.
 * */
public class RoundedCornersLayout extends LinearLayout {

    private float mDensity = Resources.getSystem().getDisplayMetrics().density;
    private Paint mPaint = new Paint();
    private Path mPathCorners = new Path();
    private Path mPathCornersBorder = new Path();

    private RectF rect = new RectF(0, 0, 0, 0);
    private RectF rectBorder = new RectF(0, 0, 0, 0);

    private float mBottomLeftRadiusPx;
    private float mTopLeftRadiusPx;
    private float mTopRightRadiusPx;
    private float mBottomRightRadiusPx;

    private boolean mDrawBottomBorder;
    private boolean mDrawLeftBorder;
    private boolean mDrawTopBorder;
    private boolean mDrawRightBorder;

    private float mBorderWidth;
    private float mBorderHalf;
    private boolean mShowBorder = false;
    private int mBorderColor = Color.parseColor("#000000");
    private float mMaxRadiusPx = 50;

    private int save;

    public RoundedCornersLayout(Context context) {
        this(context, null);
    }

    public RoundedCornersLayout(Context context, AttributeSet attrs) {
        this(context, attrs, 0);
    }

    public RoundedCornersLayout(Context context, AttributeSet attrs, int defStyle) {
        super(context, attrs, defStyle);
        mDensity = getResources().getDisplayMetrics().density;

        mPaint.setAntiAlias(true);
        mPaint.setStyle(Paint.Style.STROKE);
        mPaint.setColor(mBorderColor);
    }

    /**
     * Set border width.
     * If width <= 0 the border will not be shown.
     * */
    public void setBorderWidth(float width) {
        mBorderWidth = width;

        if (mBorderWidth > 0) {
            mShowBorder = true;
        } else {
            mShowBorder = false;
        }

        mBorderHalf = mBorderWidth / 2;

        mPaint.setStrokeWidth(mBorderWidth);
        updateRectangleBorder();
        invalidate();
    }

    public void setBorderColor(int color) {
        mBorderColor = color;
        mPaint.setColor(color);
        invalidate();
    }

    /**
     * Set corner radius for each corner in dp.
     * */
    public void setCornersRadius(float bottomLeftRadiusDp, float topLeftRadiusDp, float topRightRadiusDp, float bottomRightRadiusDp) {
        // Create a scale factor (which depends on the display density) to resemble the scale of the rounded corners that android creates when using an xml drawable.
        float scale = mDensity / 2.2f;

        // Convert to actual pixels and divide by the scale.
        mBottomLeftRadiusPx = dpToPx(bottomLeftRadiusDp) / scale;
        mTopLeftRadiusPx = dpToPx(topLeftRadiusDp) / scale;
        mTopRightRadiusPx = dpToPx(topRightRadiusDp) / scale;
        mBottomRightRadiusPx = dpToPx(bottomRightRadiusDp) / scale;

        // Find and save the highest radius for later use.
        mMaxRadiusPx = findMax(mTopLeftRadiusPx, mTopRightRadiusPx, mBottomLeftRadiusPx, mBottomRightRadiusPx);

    }

    /**
     * Helper method that converts dp to screen pixels.
     * */
    private int dpToPx(float dp) {
        DisplayMetrics displayMetrics = Resources.getSystem().getDisplayMetrics();
        float px = TypedValue.applyDimension(TypedValue.COMPLEX_UNIT_DIP, dp, displayMetrics);
        return Math.round(px);
    }


    /**
     * Can be used to control which borders will be shown.
     * */
    public void setBorders(boolean bottomBorder, boolean leftBorder, boolean topBorder, boolean rightBorder) {
        mDrawBottomBorder = bottomBorder;
        mDrawLeftBorder = leftBorder;
        mDrawTopBorder = topBorder;
        mDrawRightBorder = rightBorder;
    }

    @Override
    protected void onSizeChanged(int w, int h, int oldw, int oldh) {
        super.onSizeChanged(w, h, oldw, oldh);

        // Content rectangle:
        // Set the bounds of the content rectangle with respect to the border width.
        // Otherwise, if any of the corners are not desired, create them outside of the bounds of the visible view.
        rect.bottom = mDrawBottomBorder ? h - mBorderHalf : h + mMaxRadiusPx;
        rect.left = mDrawLeftBorder ? 0 + mBorderHalf : -mMaxRadiusPx;
        rect.top = mDrawTopBorder ? 0 + mBorderHalf : -mMaxRadiusPx;
        rect.right = mDrawRightBorder ? w - mBorderHalf : w + mMaxRadiusPx;


        // Create the path of the rectangle the will be used for clipping the content.
        mPathCorners.reset();
        mPathCorners = roundedRect(rect, mTopLeftRadiusPx, mTopRightRadiusPx, mBottomRightRadiusPx, mBottomLeftRadiusPx);
        mPathCorners.close();

        updateRectangleBorder();
    }

    /**
     * Creates the border path given content bounds.
     * */
    private void updateRectangleBorder() {
        // Border rectangle.
        // Set the bounds of the borders rectangle.
        // If any of the borders are not desired, create them outside of the bounds of the visible view.
        rectBorder.bottom = mDrawBottomBorder ? rect.bottom : rect.bottom + mMaxRadiusPx;
        rectBorder.left = mDrawLeftBorder ? rect.left : rect.left - mMaxRadiusPx;
        rectBorder.top = mDrawTopBorder ? rect.top : rect.top - mMaxRadiusPx;
        rectBorder.right = mDrawRightBorder ? rect.right : rect.right + mMaxRadiusPx;

        // Create the path of the rectangle the will be used for the borders around the clipped content.
        mPathCornersBorder.reset();
        mPathCornersBorder = roundedRect(rectBorder, mTopLeftRadiusPx, mTopRightRadiusPx, mBottomRightRadiusPx, mBottomLeftRadiusPx);
        mPathCornersBorder.close();

    }

    @Override
    protected void dispatchDraw(Canvas canvas) {
        save = canvas.save();
        // Clip the content using the corners path created on size changed.
        canvas.clipPath(mPathCorners);
        super.dispatchDraw(canvas);
        canvas.restoreToCount(save);

        // Draw the border path if it is set
        if (mShowBorder) {
            canvas.drawPath(mPathCornersBorder, mPaint);
        }
    }

    /**
     * Create a rounded corners path using the provided rectangle and the corners radius.
     * */
    private Path roundedRect(RectF rect, float topLeftDiameter, float topRightDiameter, float bottomRightDiameter, float bottomLeftDiameter) {
        Path path = new Path();

        path.moveTo(rect.left + topLeftDiameter / 2, rect.top);
        path.lineTo(rect.right - topRightDiameter / 2, rect.top);

        path.quadTo(rect.right, rect.top, rect.right, rect.top + topRightDiameter / 2);
        path.lineTo(rect.right, rect.bottom - bottomRightDiameter / 2);

        path.quadTo(rect.right, rect.bottom, rect.right - bottomRightDiameter / 2, rect.bottom);
        path.lineTo(rect.left + bottomLeftDiameter / 2, rect.bottom);

        path.quadTo(rect.left, rect.bottom, rect.left, rect.bottom - bottomLeftDiameter / 2);
        path.lineTo(rect.left, rect.top + topLeftDiameter / 2);

        path.quadTo(rect.left, rect.top, rect.left + topLeftDiameter / 2, rect.top);
        path.close();

        return path;
    }

    private float findMax(float a, float b, float c, float d) {
        float max = mMaxRadiusPx;

        if (a > max) { max = a; }
        if (b > max) { max = b; }
        if (c > max) { max = c; }
        if (d > max) { max = d; }

        return max;
    }
}
