package com.liveperson.infra.ui.view.header.caching;

import androidx.collection.LongSparseArray;
import androidx.recyclerview.widget.LinearLayoutManager;
import androidx.recyclerview.widget.RecyclerView;
import android.view.View;
import android.view.ViewGroup;

import com.liveperson.infra.ui.view.adapter.viewholder.BaseViewHolder;
import com.liveperson.infra.ui.view.header.StickyRecyclerHeadersAdapter;
import com.liveperson.infra.ui.view.header.util.OrientationProvider;


/**
 * An implementation of {@link HeaderProvider} that creates and caches header views
 */
public class HeaderViewCache implements HeaderProvider {

    private final StickyRecyclerHeadersAdapter mAdapter;
    private final LongSparseArray<BaseViewHolder> mHeaderViews = new LongSparseArray<>();
    private final OrientationProvider mOrientationProvider;

    public HeaderViewCache(StickyRecyclerHeadersAdapter adapter,
                           OrientationProvider orientationProvider) {
        mAdapter = adapter;
        mOrientationProvider = orientationProvider;
    }

    @Override
    public View getHeader(RecyclerView parent, int position) {
        long headerId = mAdapter.getHeaderId(position);

        BaseViewHolder headerHolder = mHeaderViews.get(headerId);
        if (headerHolder == null) {
            //TODO - recycle views
            headerHolder = (BaseViewHolder)mAdapter.onCreateHeaderViewHolder(parent);
            mHeaderViews.put(headerId, headerHolder);
        }
        mAdapter.onBindHeaderViewHolder(headerHolder, position);
        measureHeader(parent, headerHolder);
        return headerHolder.itemView;
    }

    @Override
    public void invalidate() {
        mHeaderViews.clear();
    }


    private void measureHeader(RecyclerView parent, BaseViewHolder viewHolder){
        View header = viewHolder.itemView;
        if (header.getLayoutParams() == null) {
            header.setLayoutParams(new ViewGroup.LayoutParams(
                    ViewGroup.LayoutParams.WRAP_CONTENT, ViewGroup.LayoutParams.WRAP_CONTENT));
        }

        int widthSpec;
        int heightSpec;

        if (mOrientationProvider.getOrientation(parent) == LinearLayoutManager.VERTICAL) {
            widthSpec = View.MeasureSpec.makeMeasureSpec(parent.getWidth(), View.MeasureSpec.EXACTLY);
            heightSpec = View.MeasureSpec.makeMeasureSpec(parent.getHeight(), View.MeasureSpec.UNSPECIFIED);
        } else {
            widthSpec = View.MeasureSpec.makeMeasureSpec(parent.getWidth(), View.MeasureSpec.UNSPECIFIED);
            heightSpec = View.MeasureSpec.makeMeasureSpec(parent.getHeight(), View.MeasureSpec.EXACTLY);
        }

        int childWidth = ViewGroup.getChildMeasureSpec(widthSpec,
                parent.getPaddingLeft() + parent.getPaddingRight(), header.getLayoutParams().width);
        int childHeight = ViewGroup.getChildMeasureSpec(heightSpec,
                parent.getPaddingTop() + parent.getPaddingBottom(), header.getLayoutParams().height);
        header.measure(childWidth, childHeight);
        header.layout(0, 0, header.getMeasuredWidth(), header.getMeasuredHeight());
    }
}
