package com.liveperson.infra.ui.view.ui.progress

import android.animation.ValueAnimator
import android.content.Context
import android.content.res.TypedArray
import android.graphics.Canvas
import android.graphics.Color
import android.graphics.drawable.Drawable
import android.os.Build
import android.util.AttributeSet
import android.util.TypedValue
import android.view.View
import com.liveperson.infra.ui.R

class LPProgressBar  @JvmOverloads constructor(
    context: Context,
    attrs: AttributeSet? = null,
    defStyleAttr: Int = 0,
    defStyleRes: Int = 0
) : View(context, attrs, defStyleAttr, defStyleRes) {

    companion object {
        private const val DEFAULT_PROGRESS_TRACK = 0.5f
        private const val DEFAULT_MINIMUM_HEIGHT_IN_DP = 4f
        private const val DEFAULT_DURATION = 1250
    }

    private var progressTrackWeight = DEFAULT_PROGRESS_TRACK

    private val progressDrawable: ProgressDrawable = ProgressDrawable()

    private val progressAnimator: ValueAnimator = ValueAnimator.ofFloat(0f, 1f)
        .apply { repeatCount = ValueAnimator.INFINITE }
        .apply { repeatMode = ValueAnimator.RESTART }

    init {
        context.readFromAttributes(attrs, defStyleAttr, defStyleRes) {
            progressDrawable.color = it.getColor(
                R.styleable.LPProgressBar_progress_color,
                Color.BLUE
            )
            progressDrawable.cornerRadius = it.getDimension(
                R.styleable.LPProgressBar_progress_corner_radius,
                0f
            )
            progressTrackWeight = it.getFloat(
                R.styleable.LPProgressBar_progress_weight,
                DEFAULT_PROGRESS_TRACK
            )
            val animationDuration = it.getInt(
                R.styleable.LPProgressBar_progress_animation_duration,
                DEFAULT_DURATION
            )
            progressAnimator.duration = animationDuration.toLong()
        }
    }

    private var progress: Float = 0f
        set(value) {
            field = value
            progressDrawable.changeProgressBounds(value, measuredWidth, measuredHeight)
        }



    override fun onAttachedToWindow() {
        super.onAttachedToWindow()
        progressDrawable.callback = this
    }

    override fun onMeasure(widthMeasureSpec: Int, heightMeasureSpec: Int) {
        val widthMode = MeasureSpec.getMode(widthMeasureSpec)
        val widthSize = MeasureSpec.getSize(widthMeasureSpec)
        val heightMode = MeasureSpec.getMode(heightMeasureSpec)
        val heightSize = MeasureSpec.getSize(heightMeasureSpec)

        val width: Int = when (widthMode) {
            MeasureSpec.EXACTLY -> {
                widthSize
            }
            MeasureSpec.AT_MOST -> {
                suggestedMinimumWidth.coerceAtMost(widthSize)
            }
            else -> {
                suggestedMinimumWidth
            }
        }

        val height: Int = when (heightMode) {
            MeasureSpec.EXACTLY -> {
                heightSize
            }
            MeasureSpec.AT_MOST -> {
                suggestedMinimumHeight.coerceAtMost(heightSize)
            }
            else -> {
                suggestedMinimumHeight
            }
        }

        setMeasuredDimension(width, height)
        progressDrawable.changeProgressBounds(progress, width, height)
    }

    override fun onDraw(canvas: Canvas) {
        super.onDraw(canvas)
        progressDrawable.draw(canvas)
    }

    override fun onDetachedFromWindow() {
        progressDrawable.callback = null
        stopAnimation()
        super.onDetachedFromWindow()
    }

    override fun verifyDrawable(who: Drawable): Boolean {
        return super.verifyDrawable(who) || who == progressDrawable
    }

    override fun setVisibility(visibility: Int) {
        super.setVisibility(visibility)
        when (visibility) {
            VISIBLE -> startAnimation()
            else -> stopAnimation()
        }
    }

    private inline fun Context.readFromAttributes(
        attrs: AttributeSet?,
        defStyleAttr: Int,
        defStyleRes: Int,
        block: (TypedArray) -> Unit
    ) {
        obtainStyledAttributes(attrs, R.styleable.LPProgressBar, defStyleAttr, defStyleRes)
            .apply(block)
            .recycle()
    }

    private fun Context.toPx(dp: Float): Float {
        return TypedValue.applyDimension(TypedValue.COMPLEX_UNIT_DIP, dp, resources.displayMetrics)
    }

    private fun startAnimation() {
        progressAnimator.takeUnless { it.isStarted || it.isRunning }
            ?.also {
                if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP_MR1) {
                    it.setCurrentFraction(progress)
                }
            }
            ?.apply { addUpdateListener { progress = it.animatedValue as? Float ?: 0f } }
            ?.start()
    }

    override fun getSuggestedMinimumHeight(): Int {
        return context.toPx(DEFAULT_MINIMUM_HEIGHT_IN_DP).toInt()
    }

    private fun stopAnimation() {
        progressAnimator.apply { removeAllUpdateListeners() }
            .takeIf { it.isStarted || it.isRunning }
            ?.cancel()
    }

    private fun ProgressDrawable.changeProgressBounds(
        progress: Float,
        width: Int,
        height: Int
    ) {
        val progressWidth = width.toFloat() * progressTrackWeight
        val offsetX = progress * (width + progressWidth) - progressWidth
        setBounds(offsetX.toInt(), 0, (progressWidth + offsetX).toInt(), height)
    }
}