package com.liveperson.infra.ui.view.uicomponents;

import android.app.Activity;
import android.content.Context;
import android.graphics.Bitmap;
import android.graphics.PorterDuff;
import android.graphics.drawable.Drawable;
import android.os.Handler;
import android.os.Looper;
import android.text.Editable;
import android.text.SpannableString;
import android.text.TextUtils;
import android.text.TextWatcher;
import android.util.AttributeSet;
import android.util.DisplayMetrics;
import android.util.TypedValue;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.view.WindowManager;
import android.view.animation.Animation;
import android.view.animation.AnimationUtils;
import android.view.inputmethod.EditorInfo;
import android.view.inputmethod.InputMethodManager;
import android.widget.Button;
import android.widget.EditText;
import android.widget.ImageButton;
import android.widget.ImageSwitcher;
import android.widget.ImageView;
import android.widget.LinearLayout;
import android.widget.TextView;
import android.widget.Toast;
import android.widget.ViewSwitcher;

import androidx.annotation.Nullable;
import androidx.core.content.ContextCompat;

import com.liveperson.infra.ForegroundService;
import com.liveperson.infra.configuration.Configuration;
import com.liveperson.infra.controller.DBEncryptionHelper;
import com.liveperson.infra.controller.DBEncryptionService;
import com.liveperson.infra.html.HtmlUtils;
import com.liveperson.infra.log.LPLog;
import com.liveperson.infra.managers.PreferenceManager;
import com.liveperson.infra.ui.R;
import com.liveperson.infra.ui.view.utils.ImageUrlUtil;
import com.liveperson.infra.utils.EncryptionVersion;
import com.liveperson.infra.utils.ImageUtils;
import com.liveperson.infra.utils.LinkPreviewCallback;
import com.liveperson.infra.utils.SourceContent;
import com.liveperson.infra.utils.patterns.PatternsCompat;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

import static androidx.appcompat.content.res.AppCompatResources.getDrawable;
import static com.liveperson.infra.managers.PreferenceManager.KEY_TYPED_TEXT;
import static com.liveperson.infra.utils.LpRecognitionListener.TEXT_SOURCE_SPEECH_RECOGNITION;

/**
 * Layout used in the bottom of the conversation fragment
 * includes an EditText and a send button
 */
public abstract class BaseEnterMessage extends LinearLayout implements IConnectionChangedCustomView,IUpdateFromBrand{

	private static final String TAG = "BaseEnterMessage";
	private static final String KEY_TYPED_TEXT_ENCRYPT_VERSION = "KEY_TYPED_TEXT_ENCRYPT_VERSION";

	public static final int RECORDING_LAYOUT_ANIMATION_DURATION_MILLIS = 200;
	private ExecutorService executorService;
	private Handler mainHandler;

	protected EditText textInput;
	protected Button mTextSendButton;
	protected ImageButton mImageSendButton;
	private ImageSwitcher mAttachBtn;
	protected ImageButton trashButton;
	protected ViewSwitcher mEnterMessageLayoutSwitcher;

	protected IConversationProvider mBrandIdProvider;
	protected IEnterMessageListener mEnterMessageListener;
	private Animation mAnimationOut;
	private Animation mAnimationIn;
	protected boolean mIsConnected;
	protected boolean mIsUpdated;
	private boolean mPhotoSharingKillSwitchEnabled;
	private boolean mPhotoSharingSiteSettingsEnabled;
	private IOverflowMenu mOverflowMenu;

	// views for link preview ui
	private ViewGroup mDropPreview;
	private String mCurrentTitle, mCurrentDescription, mCurrentImageURL, mCurrentUrl, mSiteName, mCurrentText;
	private Bitmap mCurrentImage;
	private boolean mIsRealTimePreviewEnabled, mIsSufficientToDisplayLinkPreview;

	protected boolean mIsOfflineMessagingEnabled;

	private enum InputState {HAS_TEXT, NONE}

	private InputState mInputState = InputState.NONE;

	/**
	 * @param context
	 */
	public BaseEnterMessage(Context context) {
		super(context);
		init(context);
	}

	/**
	 * @param context
	 * @param attrs
	 */
	public BaseEnterMessage(Context context, AttributeSet attrs) {
		super(context, attrs);
		init(context);
	}

	/**
	 * @param context
	 * @param attrs
	 * @param defStyleAttr
	 */
	public BaseEnterMessage(Context context, AttributeSet attrs, int defStyleAttr) {
		super(context, attrs, defStyleAttr);
		init(context);
	}

	/**
	 * @param context
	 */
	private void init(Context context) {
		executorService = Executors.newSingleThreadExecutor();
		mainHandler = new Handler(Looper.getMainLooper());

		LayoutInflater.from(context).inflate(R.layout.lpinfra_ui_enter_message_layout, this);
		initializeViews();
	}

	/**
	 * @param provider
	 */
	public void setBrandIdProvider(IConversationProvider provider) {
		mBrandIdProvider = provider;
	}

	protected void initializeViews() {
		textInput = findViewById(R.id.lpui_enter_message_text);
		mTextSendButton = findViewById(R.id.lpui_enter_message_send);
		mImageSendButton = findViewById(R.id.lpui_enter_message_send_button);
		mAttachBtn = findViewById(R.id.lpui_attach_file);
		trashButton = findViewById(R.id.lpui_voice_trash_button);
		mEnterMessageLayoutSwitcher = findViewById(R.id.lpui_enter_view_switcher);

		mIsRealTimePreviewEnabled = Configuration.getBoolean(R.bool.link_preview_enable_real_time_preview);
		mDropPreview = findViewById(R.id.lpui_drop_preview_view);

		setEditTextPreferences();
		setSendButtonPreferences();
		setAttachButtonPreferences();
	}
	/**
	 * Setting the edit text preferences
	 * We set those parameters here, because we want to insure that if the host app
	 * overrides the edit text implementation, they need to define the least needed
	 */
	private void setEditTextPreferences() {
		textInput.setHint(R.string.lp_enter_message);
		//Multi-line and send action can't be enable at the same time.
		if (Configuration.getBoolean(R.bool.enable_ime_options_action_send)) {
			textInput.setInputType(EditorInfo.TYPE_CLASS_TEXT | EditorInfo.TYPE_TEXT_FLAG_IME_MULTI_LINE | EditorInfo.TYPE_TEXT_FLAG_CAP_SENTENCES);
			textInput.setImeOptions(EditorInfo.IME_ACTION_SEND);
			textInput.setOnEditorActionListener((v, actionId, event) -> {
				boolean handled = false;
				if (actionId == EditorInfo.IME_ACTION_SEND) {
					sendMessage();
					handled = true;
				}
				return handled;
			});
		} else {
			textInput.setInputType(EditorInfo.TYPE_CLASS_TEXT | EditorInfo.TYPE_TEXT_FLAG_MULTI_LINE | EditorInfo.TYPE_TEXT_FLAG_CAP_SENTENCES);
		}
		textInput.setTextSize(TypedValue.COMPLEX_UNIT_SP, 14);
		textInput.setTextColor(ContextCompat.getColor(getContext(), R.color.lp_enter_msg_text));
		textInput.setHintTextColor(ContextCompat.getColor(getContext(), R.color.lp_enter_msg_hint));
		textInput.setLinksClickable(false);
		textInput.cancelLongPress();

		textInput.addTextChangedListener(new TextWatcher() {

			public void afterTextChanged(Editable s) {
				onAfterChangedText(s.toString());
			}

			public void beforeTextChanged(CharSequence s, int start, int count, int after) {
				onBeforeChangedText();
			}

			public void onTextChanged(CharSequence s, int start, int before, int count) {
				Object tag = textInput.getTag();
				boolean isDictation = tag == TEXT_SOURCE_SPEECH_RECOGNITION;
				if (isSpeechRecognitionInProgress() && !isDictation) {
					stopSpeechRecognitionIfNeeded();
				}
				String msgStr = textInput.getText().toString().trim();
				mCurrentText = msgStr;

				if (!TextUtils.isEmpty(msgStr)) {
					if (mIsRealTimePreviewEnabled) {
						generateLinkPreview(msgStr);
					}
					if (mInputState != InputState.HAS_TEXT) {
						mInputState = InputState.HAS_TEXT;
						onHasText(true);
						updateSendButtonState();
					}
				} else {
					mCurrentUrl = "";
					clean();
					if (mInputState == InputState.HAS_TEXT) {
						mInputState = InputState.NONE;
						onHasText(false);
						updateSendButtonState();
					}

				}
			}
		});
	}

	/**
	 * Setting the send button preferences
	 */
	private void setSendButtonPreferences() {

		// Select between Sent text button and Send image button
		if (Configuration.getBoolean(R.bool.use_send_image_button)) {
			mImageSendButton.setVisibility(VISIBLE);
			mTextSendButton.setVisibility(GONE);
			mImageSendButton.setOnClickListener(v -> sendMessage());
		} else {
			mImageSendButton.setVisibility(GONE);
			mTextSendButton.setVisibility(VISIBLE);
			mTextSendButton.setOnClickListener(v -> sendMessage());
		}
		setEnterTextMinWidth();
	}

	private void setEnterTextMinWidth() {
		DisplayMetrics metrics = new DisplayMetrics();
		WindowManager wm = (WindowManager) getContext().getSystemService(Context.WINDOW_SERVICE);
		wm.getDefaultDisplay().getMetrics(metrics);
		mTextSendButton.setMaxWidth(metrics.widthPixels / 2);
	}

	/**
	 * Setting the attach button preferences
	 */
	private void setAttachButtonPreferences() {
		// Get the general configuration key for enabling photo sharing
		mPhotoSharingKillSwitchEnabled = PreferenceManager.getInstance().getBooleanValue(PreferenceManager.KILL_SWITCH_PHOTO_SHARING_ENABLED_PREFERENCE_KEY, PreferenceManager.APP_LEVEL_PREFERENCES, true);
		mPhotoSharingSiteSettingsEnabled = PreferenceManager.getInstance().getBooleanValue(PreferenceManager.SITE_SETTINGS_PHOTO_SHARING_ENABLED_PREFERENCE_KEY, PreferenceManager.APP_LEVEL_PREFERENCES, true);

		// Disable photo sharing if set to false
		if (!Configuration.getBoolean(R.bool.enable_photo_sharing) || !mPhotoSharingKillSwitchEnabled || !mPhotoSharingSiteSettingsEnabled) {
			mAttachBtn.setVisibility(GONE);
		}

		mAnimationOut = AnimationUtils.loadAnimation(getContext(), R.anim.menu_icon_amination_out);
		mAnimationIn = AnimationUtils.loadAnimation(getContext(), R.anim.menu_icon_amination_in);

		mAttachBtn.setFactory(() -> {
			ImageView imageView = new ImageView(getContext());
			imageView.setScaleType(ImageView.ScaleType.CENTER);
			imageView.setLayoutParams(new ImageSwitcher.LayoutParams(LayoutParams.MATCH_PARENT, LayoutParams.MATCH_PARENT));
			return imageView;
		});

		Drawable attachIcon = getDrawable(getContext(), R.drawable.lpinfra_ui_ic_attach);
		attachIcon.setColorFilter(getResources().getColor(R.color.lp_file_attach_icon_clip_color), PorterDuff.Mode.MULTIPLY);
		mAttachBtn.setImageDrawable(attachIcon);

		disableAttachButton();

		mAttachBtn.setOnClickListener(new OnClickListener() {

			private ICloseMenuListener menuListener = new ICloseMenuListener() {
				@Override
				public void onCloseMenu() {
					Drawable attachIcon = getDrawable(getContext(), R.drawable.lpinfra_ui_ic_attach);
					attachIcon.setColorFilter(getResources().getColor(R.color.lp_file_attach_icon_clip_color), PorterDuff.Mode.MULTIPLY);
					mAttachBtn.setImageDrawable(attachIcon);

					mAttachBtn.setContentDescription(getResources().getString(R.string.lp_accessibility_attachment_menu_button_collapsed));
					requestFocus();
				}
			};

			@Override
			public void onClick(View v) {
				if (mOverflowMenu != null) {

					if (mOverflowMenu.isMenuOpen()) {
						//close the menu, it'll call the onCloseMenu listener and will put the attach icon.
						mOverflowMenu.hide();
					} else {
						// if the link preview layout is visible close it
						if (mDropPreview != null) {
							mDropPreview.setVisibility(GONE);
						}

						Drawable closeIcon = getDrawable(getContext(), R.drawable.lpinfra_ui_ic_close);
						closeIcon.setColorFilter(getResources().getColor(R.color.lp_file_close_icon_clip_color), PorterDuff.Mode.MULTIPLY);
						mAttachBtn.setImageDrawable(closeIcon);

						mAttachBtn.setContentDescription(getResources().getString(R.string.lp_accessibility_attachment_menu_button_expanded));

						mOverflowMenu.show();
						mOverflowMenu.setOnCloseListener(menuListener);
					}
				}

			}
		});
	}

	/**
	 * Disable the attach button
	 */
	private void disableAttachButton() {
		mAttachBtn.setEnabled(false);
		mAttachBtn.setAlpha(0.5f);
	}

	protected abstract void generateLinkPreview(String input);

	protected abstract String prepareLink(String input);

	/**
	 * Callback to update the preview view.
	 */
	protected LinkPreviewCallback callback = new LinkPreviewCallback() {

		/**
		 * validateSufficientConditionsToDisplayPreview
		 * Only in case of at least 2 OG: tags we will show and send a url message
		 */
		private boolean validateSufficientConditionsToDisplayPreview(SourceContent sourceContent) {
			int counter = 0;
			counter = sourceContent.getSiteName().isEmpty() ? --counter : ++counter;
			counter = sourceContent.getDescription().isEmpty() ? --counter : ++counter;
			counter = sourceContent.getTitle().isEmpty() ? --counter : ++counter;
			counter = sourceContent.getImages().isEmpty() ? --counter : ++counter;
			return mIsSufficientToDisplayLinkPreview = (counter >= 0);
		}

		@Override
		public void onPos(final SourceContent sourceContent, boolean isNull) {
			// if parsing failed - show error message
			if (isNull || sourceContent == null || TextUtils.isEmpty(sourceContent.getFinalUrl())) {
				mDropPreview.setVisibility(GONE);
				BaseEnterMessage.this.clean();
				mCurrentUrl = "";

			} else {
				if (textInput.getText().toString().trim().isEmpty()
						|| !mCurrentText.equals(textInput.getText().toString().trim())
						|| !sourceContent.getFinalUrl().equalsIgnoreCase(prepareLink(mCurrentText))) {
					return;
				}
				if ( validateSufficientConditionsToDisplayPreview(sourceContent) && mIsRealTimePreviewEnabled) {
					final ImageView imageSet = mDropPreview.findViewById(R.id.image_post_set);
					final TextView titleTextView = mDropPreview.findViewById(R.id.title);
					final TextView descriptionTextView = mDropPreview.findViewById(R.id.description);
					final ImageButton closeBtn = mDropPreview.findViewById(R.id.close_btn);
					imageSet.setImageDrawable(null);
					mDropPreview.setVisibility(VISIBLE);

					// click on the close button will
					closeBtn.setOnClickListener(v -> {
						clean();
						mDropPreview.setVisibility(GONE);
					});

					// this is to fix a bug MOB-7097
					sourceContent.setImages(ImageUrlUtil.completeToValidImageUrl(sourceContent.getFinalUrl(), sourceContent.getImages()));

					executorService.execute(() -> {
						if (sourceContent.getImages().isEmpty()) {
							mainHandler.post(() -> {
								imageSet.setVisibility(GONE);
								imageSet.setImageDrawable(null);
							});
							return;
						}
						mCurrentImage = ImageUtils.createBitmapFromURL(getContext(), sourceContent.getImages(), imageSet.getWidth(), imageSet.getHeight());
						if (mCurrentImage != null) {
							mainHandler.post(() -> {
								imageSet.setImageBitmap(mCurrentImage);
								imageSet.setVisibility(VISIBLE);
							});
						} else {
							mainHandler.post(() -> {
								imageSet.setVisibility(GONE);
								imageSet.setImageDrawable(null);
							});
						}
					});

					SpannableString title = HtmlUtils.convertHtmlToStyledString(sourceContent.getTitle());
					SpannableString description = HtmlUtils.convertHtmlToStyledString(sourceContent.getDescription());
					titleTextView.setText(title);
					descriptionTextView.setText(description);
				}
				// save the data to the local members for later use
				mCurrentTitle = sourceContent.getTitle();
				mCurrentDescription = sourceContent.getDescription();
				mCurrentUrl = sourceContent.getUrl();
				mCurrentImageURL = sourceContent.getImages();
				mSiteName = sourceContent.getSiteName();
			}
		}

		@Override
		public void clean() {
			// What does this method even do? It inflates a layout to a value that was never assigned
			// (I since changed that value to a hardcoded null, instead of a hidden null), and then
			// changes that layout, but never actually uses it?
			LayoutInflater inflater = (LayoutInflater) BaseEnterMessage.this.getContext().getSystemService(Context.LAYOUT_INFLATER_SERVICE);
			final View content = inflater.inflate(R.layout.lpinfra_ui_enter_message_preview_content_layout, null);
			final ImageView imageSet = content.findViewById(R.id.image_post_set);
			imageSet.setImageDrawable(null);
			imageSet.setVisibility(GONE);
		}
	};

	protected void clean() {
		mCurrentTitle = mCurrentDescription = mCurrentUrl = mCurrentImageURL = "";
		mCurrentImage = null;

		if (callback != null) {
			callback.clean();
		}
		if (mDropPreview != null) {
			mDropPreview.setVisibility(GONE);
		}
	}

	/**
	 * Clear the edit text
	 * and the link preview area
	 */
	protected void clearText() {
		textInput.setText("");
		clean();
	}

	/**
	 * Show toast message when there is no connection
	 */
	protected void showNoNetworkMessage() {
		if (ForegroundService.getInstance().isBrandForeground()) {
			Toast.makeText(getContext(), R.string.lp_no_network_toast_message, Toast.LENGTH_LONG).show();
		}
	}

	/**
	 * Get the text from the input text and sends it
	 */
	protected void sendMessage() {
		String msgStr = textInput.getText().toString().trim();
		if (!TextUtils.isEmpty(msgStr)) {
			// if the text to send doesn't contain a url we will send a regular text message
			// also in case there isn't sufficient tag to display as a link
			if (TextUtils.isEmpty(mCurrentUrl) || !mIsSufficientToDisplayLinkPreview) {
				LPLog.INSTANCE.d(TAG, "url send Message, mIsSufficientToDisplayLinkPreview: " + mIsSufficientToDisplayLinkPreview);
				sendMessage(msgStr);
			} else {
				LPLog.INSTANCE.d(TAG, "url sendMessageWithURL");
				sendMessageWithURL(msgStr, mCurrentUrl, mCurrentTitle, mCurrentImageURL, mCurrentDescription, mSiteName);
			}
			clean();
		}
	}

	/**
	 * Setting the views as disables/enabled
	 *
	 * @param enabled
	 */
	@Override
	public void setEnabled(boolean enabled) {
		super.setEnabled(enabled);
		textInput.setEnabled(enabled);
		mAttachBtn.setEnabled(enabled);

		// If enabled == true, update the send button according to text box and connection state
		if (enabled) {
			updateAttachButton();
		}

		updateSendButtonState();
	}

	/**
	 * Update callback,to check if
	 * we are updated after the
	 * internet connection is established
	 *
	 * @param isUpdated
	 */
	@Override
	public void onUpdate(boolean isUpdated) {
		mIsUpdated = isUpdated;
		updateSendButtonState();
	}

	/**
	 * Connection callback
	 *
	 * @param isConnected
	 */
	@Override
	public void onConnectionChanged(boolean isConnected) {
		mIsConnected = isConnected;
		updateAttachButton();

		// Enable/disable attachment button according to if the PhotoSharingEnabled remote configuration was changed
		mPhotoSharingKillSwitchEnabled = PreferenceManager.getInstance().getBooleanValue(PreferenceManager.KILL_SWITCH_PHOTO_SHARING_ENABLED_PREFERENCE_KEY, PreferenceManager.APP_LEVEL_PREFERENCES, true);
		mPhotoSharingSiteSettingsEnabled = PreferenceManager.getInstance().getBooleanValue(PreferenceManager.SITE_SETTINGS_PHOTO_SHARING_ENABLED_PREFERENCE_KEY, PreferenceManager.APP_LEVEL_PREFERENCES, true);

		// Disable photo sharing if set to false
		if (!Configuration.getBoolean(R.bool.enable_photo_sharing) || !mPhotoSharingKillSwitchEnabled || !mPhotoSharingSiteSettingsEnabled) {
			mAttachBtn.setVisibility(GONE);
		} else {
			mAttachBtn.setVisibility(VISIBLE);
		}
	}

	/**
	 * Set the animation to the attach button
	 */
	private void setAnimationOnAttachButton() {
		mAttachBtn.setInAnimation(mAnimationIn);
		mAttachBtn.setOutAnimation(mAnimationOut);
	}

	/**
	 * Cancel the animation on the attach button
	 */
	private void cancelAnimationOnAttachButton() {
		mAttachBtn.setInAnimation(null);
		mAttachBtn.setOutAnimation(null);
	}

	/**
	 * Update the send button color
	 * Depends on the connection state and if the EditText contains characters
	 */
	protected void updateSendButtonState() {

		if (!shouldUpdateSendButton()) {
			return;
		}

		if (isContentWaiting() && ((mIsConnected && mIsUpdated) || mIsOfflineMessagingEnabled) && isEnabled()) {
			mTextSendButton.setEnabled(true);
			mImageSendButton.setEnabled(true);
		} else if (!isContentWaiting()) {
			mTextSendButton.setEnabled(false);
			mImageSendButton.setEnabled(false);
			clean();
			// not empty and not connected
		} else {
			mTextSendButton.setEnabled(false);
			mImageSendButton.setEnabled(false);
		}
	}

	public void setIsOfflineMessagingEnabled(boolean isOfflineMessagingEnabled) {
		this.mIsOfflineMessagingEnabled = isOfflineMessagingEnabled;
		if (isOfflineMessagingEnabled) {
			updateAttachButton();
		}
	}

	protected boolean isContentWaiting() {
		return !TextUtils.isEmpty(textInput.getText().toString().trim());
	}

	/**
	 * Update the attach button according to the connection state
	 */
	private void updateAttachButton() {

		if (mIsConnected || mIsOfflineMessagingEnabled) {
			// Enable attach button
			cancelAnimationOnAttachButton();
			if (isEnabled()){
				enableAttachButton();
			}

			setAnimationOnAttachButton();

		} else {
			// If the attachment menu is open close it
			if (mOverflowMenu.isMenuOpen()) {
				mOverflowMenu.hide();
			}

			// Disable the attach button
			cancelAnimationOnAttachButton();
			disableAttachButton();
		}
	}

	public void setOverflowMenu(IOverflowMenu menu) {
		mOverflowMenu = menu;
		mOverflowMenu.setOnCloseListener(() -> {
			// TODO: 8/7/16 animate attachment icon
		});
	}

	/**
	 * Enable the attach button
	 */
	private void enableAttachButton() {
		mAttachBtn.setEnabled(true);
		mAttachBtn.setAlpha(1f);
	}

	/**
	 * Set message sent callback
	 *
	 * @param enterMessageListener
	 */
	public void setEnterMessageListener(IEnterMessageListener enterMessageListener) {
		mEnterMessageListener = enterMessageListener;
	}

	public String getText() {
		return textInput != null ? textInput.getText().toString() : "";
	}

	protected void hideSoftKeyboard(@Nullable final View focusedView) {
		if (focusedView != null && focusedView.getContext() != null) {
			InputMethodManager imm = (InputMethodManager) focusedView.getContext().getSystemService(Activity.INPUT_METHOD_SERVICE);
			if (imm != null) {
				imm.hideSoftInputFromWindow(focusedView.getWindowToken(), 0);
			}
		}
	}

	public void restoreTypedText() {
		// Restore typed text.
		String brandId = mBrandIdProvider.getBrandId();
		String message = PreferenceManager.getInstance().getStringValue(KEY_TYPED_TEXT, brandId, "");
		if (PreferenceManager.getInstance().contains(KEY_TYPED_TEXT_ENCRYPT_VERSION, brandId)) {
			int encryptionVersionNumber = PreferenceManager.getInstance().getIntValue(KEY_TYPED_TEXT_ENCRYPT_VERSION, brandId, 1);
			EncryptionVersion encryptionVersion = EncryptionVersion.fromInt(encryptionVersionNumber);
			String typedText = DBEncryptionHelper.decrypt(encryptionVersion, message);
			if (!TextUtils.isEmpty(typedText) && textInput != null) {
				textInput.setText(typedText);
			}
		}
	}

	public void onBackground() {
		// Save typed text.
		String typedText = getText();
		String brandId = mBrandIdProvider.getBrandId();
		EncryptionVersion encryptionVersion = DBEncryptionService.Companion.getAppEncryptionVersion();
		String encryptedMessage = DBEncryptionHelper.encrypt(encryptionVersion, typedText);
		PreferenceManager.getInstance().setStringValue(KEY_TYPED_TEXT, brandId, encryptedMessage);
		PreferenceManager.getInstance().setIntValue(KEY_TYPED_TEXT_ENCRYPT_VERSION, brandId, encryptionVersion.ordinal());
	}

	protected void requestFocusOnTextInput(){
		textInput.requestFocus();
		textInput.setSelection(textInput.length());
	}

	protected abstract void sendMessageWithURL(final String message, final String urlToParse, final String title, final String imageURL, final String description, final String siteName);


	protected abstract void sendMessage(final String message);

	/**
	 * Called when the EditText status is changed between has text or empty
	 *
	 * @param isHasText
	 */
	protected abstract void onHasText(final boolean isHasText);

	/**
	 * Called every time a text has changed. The changed text is given
	 *
	 * @param text
	 */
	protected abstract void onAfterChangedText(String text);

	/**
	 * This is called before any change in the text is about to take place
	 */
	protected abstract void onBeforeChangedText();

	/**
	 * Indicate whether to update the send button or not
	 * @return
	 */
	protected abstract boolean shouldUpdateSendButton();

	protected abstract boolean isSpeechRecognitionInProgress();

	protected abstract void stopSpeechRecognitionIfNeeded();

}
