@file:JvmName("LPTextUtils")

package com.liveperson.infra.ui.view.utils.text

import android.content.Context
import android.text.SpannableString
import android.util.Patterns
import androidx.core.text.clearSpans
import androidx.core.text.util.LinkifyCompat
import com.liveperson.infra.ui.R
import com.liveperson.infra.html.convertHtmlToStyledString
import com.liveperson.infra.log.LPLog
import com.liveperson.infra.md.convertMarkdownToHTML
import com.liveperson.infra.md.markdownToHtml
import com.liveperson.infra.ui.view.utils.text.linkify.DefaultTransformFilter
import com.liveperson.infra.ui.view.utils.text.linkify.PhoneMatchFilter
import com.liveperson.infra.ui.view.utils.text.linkify.UrlMatchFilter
import com.liveperson.infra.utils.patterns.PatternsCompat
import java.util.regex.Pattern

private const val TAG = "LPTextUtils"

private val PHONE_MATCH_FILTER = PhoneMatchFilter()
private val URL_MATCH_FILTER = UrlMatchFilter()
private val DEFAULT_TRANSFORM_FILTER = DefaultTransformFilter()

private const val SCHEME_TELEPHONE = "tel:"
private const val SCHEME_EMAIL = "mailto:"
private const val SCHEME_HTTP = "http://"
private const val SCHEME_HTTPS = "https://"

private const val CHAR_CUSTOM_ENDLINE = "\u2307"

/**
 * Method used to parse html tags with text and linkify phones, emails and links.
 *
 * @param text raw text with tags and links
 * @param markdownLinkColor color for markdown links
 * @param context application context
 *
 * @return spannable string with url spans and rich styles.
 */
fun Context.linkifyText(text: String, markdownLinkColor: Int): SpannableString {
    if (text.isBlank()) {
        return SpannableString(text)
    }
    return try {
        text.replace("\n", CHAR_CUSTOM_ENDLINE)
            .convertMarkdownToHTML(markdownLinkColor)
            .convertHtmlToStyledString()
            .withRestoredSpans { linkifyPhoneNumbers(it) }
            .withRestoredSpans { linkifyUrls(it) }
            .withRestoredSpans { linkifyEmails(it) }
            .changeString {
                // Android devices that are running on Android M
                // and below will use Legacy mode to format spannable string
                // which means that each tag will be devided by
                // 2 end-line characters (\n)
                // to adjust behavior across all supported version
                // all 2 end-line characters (\n) will be replaced with the single one
                val result = it.replace("\n\n", " \n")
                    .replace("$CHAR_CUSTOM_ENDLINE\n", " \n")
                    .replace(CHAR_CUSTOM_ENDLINE, "\n")
                if (result.endsWith('\n')) {
                    result.substring(0, result.lastIndex) + " "
                } else {
                    result
                }
            }
    } catch (throwable: Throwable) {
        LPLog.d(TAG, "Failed to parse content", throwable)
        SpannableString(text)
    }
}

/**
 * Extension method used to format raw message to remove md and html tags.
 *
 * @return formated message with md or html tags.
 */
fun String.asFormattedMessage(): String {
    if (isBlank()) {
        return this
    }
    return try {
        this.replace("\n", CHAR_CUSTOM_ENDLINE)
            .markdownToHtml()
            .convertHtmlToStyledString()
            .toString()
            .let {
                // Android devices that are running on Android M
                // and below will use Legacy mode to format spannable string
                // which means that each tag will be devided by
                // 2 end-line characters (\n)
                // to adjust behavior across all supported version
                // all 2 end-line characters (\n) will be replaced with the single one
                val result = it.replace("\n\n", " \n")
                    .replace("$CHAR_CUSTOM_ENDLINE\n", " \n")
                    .replace(CHAR_CUSTOM_ENDLINE, "\n")
                if (result.endsWith('\n')) {
                    result.substring(0, result.lastIndex) + " "
                } else {
                    result
                }
            }
    } catch (throwable: Throwable) {
        LPLog.d(TAG, "Failed to parse content", throwable)
        this
    }
}

private fun Context.linkifyPhoneNumbers(spannableString: SpannableString): Boolean {
    val customPhoneRegex: String = getString(R.string.lp_bubble_phone_links_regex)
    val (phonePattern, matchFilter) = if (customPhoneRegex.isEmpty()) {
        Patterns.PHONE to PHONE_MATCH_FILTER
    } else {
        Pattern.compile(customPhoneRegex) to null
    }
    return LinkifyCompat.addLinks(
        spannableString,
        phonePattern,
        SCHEME_TELEPHONE,
        matchFilter,
        DEFAULT_TRANSFORM_FILTER
    )
}

private fun Context.linkifyUrls(spannableString: SpannableString): Boolean {
    val customUrlRegex: String = getString(R.string.lp_bubble_url_links_regex)
    return if (customUrlRegex.isEmpty()) {
        LinkifyCompat.addLinks(
            spannableString,
            PatternsCompat.AUTOLINK_WEB_URL,
            SCHEME_HTTP,
            arrayOf(SCHEME_HTTPS),
            URL_MATCH_FILTER,
            DEFAULT_TRANSFORM_FILTER
        )
    } else {
        LinkifyCompat.addLinks(
            spannableString,
            Pattern.compile(customUrlRegex),
            null,
            URL_MATCH_FILTER,
            DEFAULT_TRANSFORM_FILTER
        )
    }
}


private fun Context.linkifyEmails(spannableString: SpannableString): Boolean {
    val customEmailRegex: String = getString(R.string.lp_bubble_email_links_regex)
    val emailPattern: Pattern = if (customEmailRegex.isEmpty()) {
        Patterns.EMAIL_ADDRESS
    } else {
        Pattern.compile(customEmailRegex)
    }
    return LinkifyCompat.addLinks(
        spannableString,
        emailPattern,
        SCHEME_EMAIL,
        null,
        DEFAULT_TRANSFORM_FILTER
    )
}

private inline fun SpannableString.withRestoredSpans(block: (SpannableString) -> Unit): SpannableString {
    val currentSpans = getSpans(0, length, Object::class.java).asSequence()
    val copyString = SpannableString(this)
    block(copyString)
    currentSpans.filterNotNull().forEach { copyString.setSpan(it, getSpanStart(it), getSpanEnd(it), 0) }
    clearSpans()
    return copyString
}

private inline fun SpannableString.changeString(block: (String) -> String): SpannableString {
    val currentSpans = getSpans(0, length, Object::class.java).asSequence()
    val result = block(toString())
    val copyString = SpannableString(result)
    currentSpans.filterNotNull().forEach { copyString.setSpan(it, getSpanStart(it), getSpanEnd(it), 0) }
    clearSpans()
    return copyString
}