package com.liveperson.infra.ui.view.ui;

import android.app.ActionBar.LayoutParams;
import android.content.Context;
import android.graphics.Rect;
import android.graphics.drawable.BitmapDrawable;
import android.os.Handler;
import android.os.Message;
import android.view.Gravity;
import android.view.LayoutInflater;
import android.view.View;
import android.widget.PopupWindow;
import android.widget.TextView;

import com.liveperson.infra.log.LPLog;
import com.liveperson.infra.ui.R;

/**
 * A popup window to display tooltip.
 * Use the showToolTip methods to display a tooltip above the required anchor view
 */
public class TooltipWindow {

	public static final String TAG = "TooltipWindow";


	private static final int MSG_DISMISS_TOOLTIP = 100;
	private Context mContext;
	private PopupWindow mTooltipPopupWindow;
	private View mTooltipContentView;
	private LayoutInflater inflater;
	private int mTooltipDelayMs;

	private boolean mEnabled;

	/**
	 * Constructor
	 * @param ctx
	 */
	public TooltipWindow(Context ctx) {
		this.mContext = ctx;
		mTooltipPopupWindow = new PopupWindow(ctx);

		inflater = (LayoutInflater) ctx.getSystemService(Context.LAYOUT_INFLATER_SERVICE);
		mTooltipContentView = inflater.inflate(R.layout.lpinfra_ui_tooltip_layout, null);
		mTooltipDelayMs = ctx.getResources().getInteger(R.integer.lp_mic_tooltip_display_time_ms);
		mEnabled = true;
	}

	/**
	 * Show the tooltip relative to the given anchor
	 * @param anchor
	 * @param tooltipText
	 */
	public void showToolTip(View anchor, String tooltipText) {
		showToolTip(anchor, tooltipText, false);
	}

	/**
	 * Show the tooltip relative to the given anchor
	 * @param anchor
	 * @param tooltipText
	 * @param indefinitely true to display tooltip without timeout
	 */
	public void showToolTip(View anchor, String tooltipText, boolean indefinitely) {

		if (!mEnabled) {
			LPLog.INSTANCE.d(TAG, "showToolTip: tooltips disabled. Not showing.");
			return;
		}

		int[] anchorScreenPos = new int[2];

		// SEt the tooltip text
		TextView tooltipTextView = mTooltipContentView.findViewById(R.id.lpui_tooltip_text);
		if(tooltipTextView != null) {
			tooltipTextView.setText(tooltipText);
		}

		mTooltipPopupWindow.setHeight(LayoutParams.WRAP_CONTENT);
		mTooltipPopupWindow.setWidth(LayoutParams.WRAP_CONTENT);

		mTooltipPopupWindow.setOutsideTouchable(true);
		mTooltipPopupWindow.setTouchable(true);
		mTooltipPopupWindow.setBackgroundDrawable(new BitmapDrawable());

		mTooltipPopupWindow.setContentView(mTooltipContentView);

		// Get location of anchor view on screen
		anchor.getLocationOnScreen(anchorScreenPos);

		// Get rect for anchor view
		Rect anchorRect = new Rect(anchorScreenPos[0], anchorScreenPos[1],
				anchorScreenPos[0]+ anchor.getWidth(), anchorScreenPos[1] + anchor.getHeight());

		// Call view measure to calculate how big the tooltip view should be.
		mTooltipContentView.measure(LayoutParams.WRAP_CONTENT, LayoutParams.WRAP_CONTENT);

		int contentViewHeight = mTooltipContentView.getMeasuredHeight();
		int contentViewWidth = mTooltipContentView.getMeasuredWidth();


		// Calculate the X and Y position of the tooltip
		int tooltipPositionX = anchorRect.centerX() - (contentViewWidth / 2);
		int tooltipPositionY = anchorRect.top - ((int) (contentViewHeight * 1.5));

		mTooltipPopupWindow.showAtLocation(anchor, Gravity.NO_GRAVITY, tooltipPositionX, tooltipPositionY);

		// Set to dismiss mTooltipPopupWindow after X milliseconds
		if(!indefinitely) {
			handler.sendEmptyMessageDelayed(MSG_DISMISS_TOOLTIP, mTooltipDelayMs);
		}
	}

	/**
	 * Enable disable tooltip showing
	 * @param enabled
	 */
	public void enable(boolean enabled){
		LPLog.INSTANCE.d(TAG, "enable: " + enabled);

		mEnabled = enabled;
	}

	public boolean isTooltipShown() {
		return (mTooltipPopupWindow != null && mTooltipPopupWindow.isShowing());
	}

	public void dismissTooltip() {
		if (mTooltipPopupWindow != null && mTooltipPopupWindow.isShowing())
			mTooltipPopupWindow.dismiss();
	}

	private Handler handler = new Handler(new Handler.Callback() {
		@Override
		public boolean handleMessage(Message message) {
			switch (message.what) {
				case MSG_DISMISS_TOOLTIP:
					if (mTooltipPopupWindow != null && mTooltipPopupWindow.isShowing())
						mTooltipPopupWindow.dismiss();
					break;
			}
			return true;
		}
	});

}
