/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package com.longmaosoft.sdk.core;

import com.alibaba.fastjson.JSONObject;
import com.longmaosoft.sdk.core.auth.LongMaoCredentials;
import com.longmaosoft.sdk.core.auth.Signer;
import com.longmaosoft.sdk.core.exception.ClientException;
import com.longmaosoft.sdk.core.exception.ServerException;
import com.longmaosoft.sdk.core.http.HttpRequest;
import com.longmaosoft.sdk.core.utils.ParameterUtils;

import java.io.File;
import java.io.IOException;
import java.net.URLEncoder;
import java.util.HashMap;
import java.util.Map;

public abstract class LongMaoRequest<T extends LongMaoResponse> extends HttpRequest {

    private final Map<String, String> queryParameters = new HashMap<String, String>();
    private final Map<String, Object> postParameters = new HashMap<String, Object>();
    private File file;

    public LongMaoRequest(String method, String version) {
        this.queryParameters.put("method", method);
        this.queryParameters.put("version", version);
        this.queryParameters.put("timestamp", ParameterUtils.getTime());
        this.queryParameters.put("format", "JSON");
    }

    protected T request(String domain, LongMaoCredentials credentials) throws IOException {
        Map<String,String> imutableMap = new HashMap<>(this.queryParameters);
        imutableMap.put("access_key_id", credentials.getAccessKeyId());

        if(isSign()) {
            imutableMap.put("sign", Signer.sign(imutableMap, credentials.getAccessKeySecret()));
        }

        StringBuilder queryParams = new StringBuilder();
        for(Map.Entry<String,String> entry : imutableMap.entrySet()) {
            queryParams.append("&").append(entry.getKey()).append("=").append(URLEncoder.encode(entry.getValue(), "UTF-8"));
        }
        String api = domain + "?" + queryParams.substring(1);

        putRequestParams();
        String str = null;
        if(file != null) {
            str = this.request(api, postParameters, file);
        }else{
            str = this.request(api, JSONObject.toJSONString(postParameters));
        }

        JSONObject data = JSONObject.parseObject(str);
        String code = data.getString("code");
        if(!"200".equals(code)) {
            if(code.equals("500")) {
                throw new ServerException(code, data.getString("message"), null);
            }else{
                throw new ClientException(code, data.getString("message"), null);
            }
        }

        return parserResponseData(data);
    }

    protected void putPostParameter(String name, Object value) {
        if(value.getClass().isAssignableFrom(File.class)) {
            file = (File) value;
        }else{
            this.postParameters.put(name, value);
        }
    }

    protected abstract Boolean isSign();

    protected abstract void putRequestParams();

    protected abstract T parserResponseData(JSONObject data);

}
