/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package com.longmaosoft.sdk.core.http;

import com.longmaosoft.sdk.core.exception.ClientException;
import org.apache.http.Consts;
import org.apache.http.StatusLine;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.RequestBuilder;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.StringEntity;
import org.apache.http.entity.mime.HttpMultipartMode;
import org.apache.http.entity.mime.MultipartEntityBuilder;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClients;
import org.apache.http.util.EntityUtils;

import java.io.File;
import java.io.IOException;
import java.util.HashMap;
import java.util.Map;

public class HttpRequest {

    protected Map<String, String> headers = new HashMap<String, String>();
    protected Integer connectTimeout = 10000;
    protected Integer readTimeout = 10000;

    protected void setConnectTimeout(Integer connectTimeout) {
        this.connectTimeout = connectTimeout;
    }

    protected void setReadTimeout(Integer readTimeout) {
        this.readTimeout = readTimeout;
    }

    protected void addHeader(String name, String value) {
        headers.put(name, value);
    }

    protected String request(String api, String json) throws IOException {
        CloseableHttpClient httpClient = HttpClients.createDefault();
        try{
            ///////////////////////////////////////////////////////////////////////////
            // 构建请求对象
            ///////////////////////////////////////////////////////////////////////////
            RequestBuilder requestBuilder = RequestBuilder.post(api);
            //设置请求编码
            requestBuilder.setCharset(Consts.UTF_8);
            ///////////////////////////////////////////////////////////////////////////
            // 设置请求超时时间
            ///////////////////////////////////////////////////////////////////////////
            RequestConfig requestConfig = RequestConfig.custom()
                    .setSocketTimeout(readTimeout)
                    .setConnectTimeout(connectTimeout)
                    .build();
            requestBuilder.setConfig(requestConfig);
            ///////////////////////////////////////////////////////////////////////////
            // 设置请求参数
            ///////////////////////////////////////////////////////////////////////////
            ContentType contentType = ContentType.create("application/json", "UTF-8");
            StringEntity entity = new StringEntity(json, contentType);
            requestBuilder.setEntity(entity);
            ///////////////////////////////////////////////////////////////////////////
            // 设置请求头
            ///////////////////////////////////////////////////////////////////////////
            if(headers != null) {
                for(Map.Entry<String,String> item : headers.entrySet()) {
                    requestBuilder.addHeader(item.getKey(), item.getValue());
                }
            }
            ///////////////////////////////////////////////////////////////////////////
            // 获取请求结果
            ///////////////////////////////////////////////////////////////////////////
            CloseableHttpResponse response = httpClient.execute(requestBuilder.build());
            StatusLine statusLine = response.getStatusLine();
            if(statusLine.getStatusCode() != 200) {
                throw new ClientException(String.valueOf(statusLine.getStatusCode()), statusLine.getReasonPhrase(), null);
            }

            try{
                return EntityUtils.toString(response.getEntity(), "UTF-8");
            }finally {
                response.close();
            }
        }finally {
            httpClient.close();
        }
    }

    protected String request(String api, Map<String,Object> params, File file) throws IOException {
        CloseableHttpClient httpClient = HttpClients.createDefault();
        try{
            ///////////////////////////////////////////////////////////////////////////
            // 构建请求对象
            ///////////////////////////////////////////////////////////////////////////
            RequestBuilder requestBuilder = RequestBuilder.post(api);
            //设置请求编码
            requestBuilder.setCharset(Consts.UTF_8);
            ///////////////////////////////////////////////////////////////////////////
            // 设置请求超时时间
            ///////////////////////////////////////////////////////////////////////////
            RequestConfig requestConfig = RequestConfig.custom()
                    .setSocketTimeout(readTimeout)
                    .setConnectTimeout(connectTimeout)
                    .build();
            requestBuilder.setConfig(requestConfig);
            ///////////////////////////////////////////////////////////////////////////
            // 设置请求头
            ///////////////////////////////////////////////////////////////////////////
            if(headers != null) {
                for(Map.Entry<String,String> item : headers.entrySet()) {
                    requestBuilder.addHeader(item.getKey(), item.getValue());
                }
            }
            ///////////////////////////////////////////////////////////////////////////
            // 设置请求参数
            ///////////////////////////////////////////////////////////////////////////
            MultipartEntityBuilder builder = MultipartEntityBuilder.create()
                    .setCharset(Consts.UTF_8)
                    .setMode(HttpMultipartMode.BROWSER_COMPATIBLE)//加上此行代码解决返回中文乱码问题
                    .addBinaryBody("file", file);

            for (Map.Entry<String, Object> e : params.entrySet()) {
                builder.addTextBody(e.getKey(), e.getValue().toString());
            }

            requestBuilder.setEntity(builder.build());
            ///////////////////////////////////////////////////////////////////////////
            // 获取请求结果
            ///////////////////////////////////////////////////////////////////////////
            CloseableHttpResponse response = httpClient.execute(requestBuilder.build());
            StatusLine statusLine = response.getStatusLine();
            if(statusLine.getStatusCode() != 200) {
                throw new ClientException(String.valueOf(statusLine.getStatusCode()), statusLine.getReasonPhrase(), null);
            }

            try{
                return EntityUtils.toString(response.getEntity(), "UTF-8");
            }finally {
                response.close();
            }
        }finally {
            httpClient.close();
        }
    }


}
