package com.laputapp.ui;

import android.os.Bundle;
import android.support.v4.view.ViewCompat;
import android.support.v4.widget.SwipeRefreshLayout;
import android.support.v4.widget.TextViewCompat;
import android.support.v7.widget.LinearLayoutManager;
import android.support.v7.widget.RecyclerView;
import android.widget.LinearLayout;
import android.widget.TextView;
import android.widget.ViewSwitcher;

import com.laputapp.R;
import com.laputapp.http.Callbacks;
import com.laputapp.http.DataLoader;
import com.laputapp.http.Response;
import com.laputapp.ui.adapter.RecyclerViewAdapter;
import com.laputapp.utilities.Lists;
import com.laputapp.utilities.UiUtilities;
import com.laputapp.widget.DelegatedSwipeRefreshLayout;
import com.laputapp.widget.ViewDelegate;
import java.util.ArrayList;
import retrofit.RetrofitError;

public abstract class RecyclerBaseActivity<T> extends com.laputapp.ui.BaseActivity implements
        SwipeRefreshLayout.OnRefreshListener, ViewDelegate, DataLoader.Loader<T>,
        DataLoader.KeyExtractor<T>,
        Callbacks.RequestCallback<ArrayList<T>> {

    private static final long DEFAULT_PAGE = 1;
    private static final long DEFAULT_PAGE_SIZE = 10;

    ViewSwitcher mSwitcherLoading;
    ViewSwitcher mSwitcherContent;
    RecyclerView mRecyclerView;
    DelegatedSwipeRefreshLayout mSwipeRefreshLayout;

    TextView mEmptyView;
    LinearLayout mBtnContainerView;

    private RecyclerViewAdapter mAdapter;

    private boolean mDataHasLoaded = false;

    protected ArrayList<T> mItems = Lists.newArrayList();
    private DataLoader<T> mDataLoader;

    protected abstract RecyclerViewAdapter createRecyclerViewAdapter();

    protected long createDefaultPage() {
        return DEFAULT_PAGE;
    }

    protected long createDefaultPageSize() {
        return DEFAULT_PAGE_SIZE;
    }

    protected int[] getColorSchemeResources() {
        return null;
    }


    @Override
    public void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        setContentView(R.layout.fragment_recycler_view);
        mDataLoader = new DataLoader<>(this, this, createDefaultPage(), createDefaultPageSize());
        mDataLoader.addRequestCallback(this);

        mEmptyView = UiUtilities.getView(this, android.R.id.empty);
        mSwitcherLoading = UiUtilities.getView(this, R.id.view_loading_switcher);
        mSwitcherContent = UiUtilities.getView(this, R.id.view_content_switcher);
        mRecyclerView = UiUtilities.getView(this, R.id.recycler_view);
        mSwipeRefreshLayout = UiUtilities.getView(this, R.id.swipe_refresh_layout);
        mBtnContainerView = UiUtilities.getView(this, R.id.view_btns);
    }

    @Override
    protected void onPostCreate(Bundle savedInstanceState) {
        super.onPostCreate(savedInstanceState);
        setupRefreshLayout();
        setupRecyclerView();
        updateDataLoader();
    }

    private void updateDataLoader() {
        if (!mItems.isEmpty()) {
            updateRecyclerView();
        }

        if (!mDataHasLoaded) {
            mDataLoader.refresh();
        }
    }

    @Override
    public boolean canChildScrollUp() {
        return ViewCompat.canScrollVertically(mRecyclerView, -1);
    }

    /**
     * If override to return false, {@link #startRefreshing()} & {@link #stopRefreshing()} shouldn't
     * be called.
     *
     * @return
     */
    public boolean canRefreshManually() {
        return true;
    }

    public void setupRefreshLayout() {
        if (!canRefreshManually()) {
            mSwipeRefreshLayout.setEnabled(false);
            return;
        }

        int[] colors = getColorSchemeResources();
        if (colors != null) {
            mSwipeRefreshLayout.setColorSchemeColors(colors);
        }

        mSwipeRefreshLayout.setOnRefreshListener(this);
        mSwipeRefreshLayout.setViewDelegate(this);
    }

    protected void setupRecyclerView() {
        mAdapter = createRecyclerViewAdapter();
        mRecyclerView.setHasFixedSize(true);
        mRecyclerView.setAdapter(mAdapter);
        mRecyclerView.setLayoutManager(new LinearLayoutManager(this));
    }

    @Override
    public void onDestroy() {
        super.onDestroy();
        mDataLoader.removeRequestCallback();
        mDataLoader = null;
        mAdapter = null;
    }

    protected void startRefreshing() {
        if (isFinishing()) return;
        ViewCompat.postOnAnimation(mSwipeRefreshLayout, new Runnable() {
            @Override
            public void run() {
                mSwipeRefreshLayout.setRefreshing(true);
            }
        });
    }

    protected void stopRefreshing() {
        if (isFinishing()) return;
        mSwipeRefreshLayout.setRefreshing(false);
    }

    protected RecyclerBaseActivity showLoadingView() {
        if (isFinishing()) return this;
        if (mSwitcherLoading.getCurrentView().getId() == com.laputapp.R.id.view_content_switcher) {
            mSwitcherLoading.showPrevious();
        }
        return this;
    }

    protected RecyclerBaseActivity showContentView() {
        if (isFinishing()) return this;
        if (mSwitcherLoading.getCurrentView().getId() == com.laputapp.R.id.view_loading) {
            mSwitcherLoading.showNext();
        }
        if (mSwitcherContent.getCurrentView().getId() != com.laputapp.R.id.view_recycler) {
            mSwitcherContent.showPrevious();
        }
        return this;
    }

    protected RecyclerBaseActivity showEmptyView() {
        if (isFinishing()) return this;
        if (mSwitcherLoading.getCurrentView().getId() == com.laputapp.R.id.view_loading) {
            mSwitcherLoading.showNext();
        }
        if (mSwitcherContent.getCurrentView().getId() == com.laputapp.R.id.view_recycler) {
            mSwitcherContent.showNext();
        }
        return this;
    }

    @Override
    public void onRefresh() {
        mDataLoader.refresh();
    }

    protected void updateRecyclerView() {
        if (isFinishing()) return;

        mAdapter.updateData(mItems);
        if (mItems.isEmpty()) {
            showEmptyView();
        } else {
            showContentView();
        }
    }

    public DataLoader<T> getDataLoader() {
        return mDataLoader;
    }

    public RecyclerView getRecyclerView() {
        return mRecyclerView;
    }

    @Override
    public void beforeRefresh() {
    }

    @Override
    public void beforeLoadMore() {
    }

    @Override
    public T register(T resource) {
        return resource;
    }

    @Override
    public void onRequestComplete(Response<ArrayList<T>> response) {
        mItems = mDataLoader.getResources();
        mDataHasLoaded = !mItems.isEmpty();
    }

    @Override
    public void onRequestFailure(Response<ArrayList<T>> response) {
        mDataHasLoaded = false;
    }

    @Override
    public void onRequestFailure(RetrofitError error) {
        mDataHasLoaded = false;
    }

    @Override
    public void onRequestNetworkError() {
        mDataHasLoaded = false;
    }

    @Override
    public void onFinish() {
        stopRefreshing();
        updateRecyclerView();
    }

    @Override
    public boolean canCache() {
        return false;
    }

    @Override
    public void handleLocalCache(ArrayList<T> data) {

    }

    @Override
    public void onCacheLoaded(ArrayList<T> localData) {
        if (localData == null || localData.isEmpty()) return;
        for (T data : localData) {
            register(data);
        }
        mItems.addAll(localData);
        if (!mItems.isEmpty()) {
            updateRecyclerView();
        }
    }

    public ArrayList<T> getItems() {
        return mItems;
    }

    public RecyclerBaseActivity setEmptyText(int resId) {
        setEmptyText(getString(resId));
        return this;
    }

    public RecyclerBaseActivity setEmptyText(String empty) {
        mEmptyView.setText(empty);
        return this;
    }

    public RecyclerBaseActivity setEmptyTextColor(int color) {
        mEmptyView.setTextColor(color);
        return this;
    }

    public RecyclerBaseActivity setEmptyDrawablePadding(int pad) {
        mEmptyView.setCompoundDrawablePadding(pad);
        return this;
    }

    public RecyclerBaseActivity setEmptyTopDrawable(int top) {
        return setEmptyTopDrawable(0, top, 0, 0);
    }

    public RecyclerBaseActivity setEmptyTopDrawable(
            int start, int top, int end, int bottom) {
        TextViewCompat.setCompoundDrawablesRelativeWithIntrinsicBounds(mEmptyView, start, top, end, bottom);
        return this;
    }

    public LinearLayout getBtnContainerView() {
        return mBtnContainerView;
    }

}