/**
 * Created by YuGang Yang on April 29, 2015.
 * Copyright 2007-2015 Laputapp.com. All rights reserved.
 */
package com.laputapp.ui;

import android.os.Bundle;
import android.support.v4.view.ViewCompat;
import android.support.v4.widget.SwipeRefreshLayout.OnRefreshListener;
import android.support.v4.widget.TextViewCompat;
import android.support.v7.widget.LinearLayoutManager;
import android.support.v7.widget.RecyclerView;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.widget.LinearLayout;
import android.widget.TextView;
import android.widget.ViewSwitcher;
import com.laputapp.R;
import com.laputapp.http.Callbacks;
import com.laputapp.http.DataLoader;
import com.laputapp.http.Response;
import com.laputapp.ui.adapter.RecyclerAdapter;
import com.laputapp.utilities.Lists;
import com.laputapp.utilities.UiUtilities;
import com.laputapp.widget.DelegatedSwipeRefreshLayout;
import com.laputapp.widget.ViewDelegate;
import java.util.ArrayList;
import retrofit.RetrofitError;


public abstract class RecyclerFragment<T>  extends BaseFragment implements
    OnRefreshListener, ViewDelegate, DataLoader.Loader<T>,
    DataLoader.KeyExtractor<T>,
    Callbacks.RequestCallback<ArrayList<T>> {

  private static final long DEFAULT_PAGE = 1;
  private static final long DEFAULT_PAGE_SIZE = 10;

  ViewSwitcher mSwitcherLoading;
  ViewSwitcher mSwitcherContent;
  RecyclerView mRecyclerView;
  DelegatedSwipeRefreshLayout mSwipeRefreshLayout;

  TextView mEmptyView;
  LinearLayout mBtnContainerView;

  private RecyclerAdapter<T> mAdapter;

  private boolean mDataHasLoaded = false;

  protected ArrayList<T> mItems = Lists.newArrayList();
  private DataLoader<T> mDataLoader;

  protected abstract RecyclerAdapter<T> createRecyclerViewAdapter();

  protected long createDefaultPage() {
    return DEFAULT_PAGE;
  }

  protected long createDefaultPageSize() {
    return DEFAULT_PAGE_SIZE;
  }

  protected int[] getColorSchemeResources() {
    return null;
  }


  @Override public void onCreate(Bundle savedInstanceState) {
    super.onCreate(savedInstanceState);
    mAdapter = createRecyclerViewAdapter();
    mDataLoader = new DataLoader<>(this, this, createDefaultPage(), createDefaultPageSize());
    mDataLoader.addRequestCallback(this);
  }

  @Override public View onCreateView(LayoutInflater inflater, ViewGroup container,
      Bundle savedInstanceState) {
    return inflater.inflate(R.layout.fragment_recycler_view, container, false);
  }

  @Override public void onViewCreated(View view, Bundle savedInstanceState) {
    super.onViewCreated(view, savedInstanceState);
    mEmptyView = UiUtilities.getView(view, android.R.id.empty);
    mSwitcherLoading = UiUtilities.getView(view, R.id.view_loading_switcher);
    mSwitcherContent = UiUtilities.getView(view, R.id.view_content_switcher);
    mRecyclerView = UiUtilities.getView(view, R.id.recycler_view);
    mSwipeRefreshLayout = UiUtilities.getView(view, R.id.swipe_refresh_layout);
    mBtnContainerView = UiUtilities.getView(view, R.id.view_btns);
    setupRefreshLayout();
    setupRecyclerView();
  }

  @Override public void onActivityCreated(Bundle savedInstanceState) {
    super.onActivityCreated(savedInstanceState);

    if (!mItems.isEmpty()) {
      updateRecyclerView();
    }

    if (!mDataHasLoaded) {
      mDataLoader.refresh();
    }
  }

  public void setupRefreshLayout() {
    int[] colors = getColorSchemeResources();
    if (colors != null) {
      mSwipeRefreshLayout.setColorSchemeColors(colors);
    }

    mSwipeRefreshLayout.setOnRefreshListener(this);
    mSwipeRefreshLayout.setViewDelegate(this);
  }

  private void setupRecyclerView() {
    mRecyclerView.setHasFixedSize(true);
    mRecyclerView.setAdapter(mAdapter);
    mRecyclerView.setLayoutManager(new LinearLayoutManager(getActivity()));
  }

  @Override public void onDestroy() {
    super.onDestroy();
    mDataLoader.removeRequestCallback();
    mDataLoader = null;
    mAdapter = null;
  }

  protected void startRefreshing() {
    if (!isVisible()) return;
    ViewCompat.postOnAnimation(mSwipeRefreshLayout, new Runnable() {
      @Override public void run() {
        mSwipeRefreshLayout.setRefreshing(true);
      }
    });
  }

  protected void stopRefreshing() {
    if (!isVisible()) return;
    mSwipeRefreshLayout.setRefreshing(false);
  }

  protected RecyclerFragment showLoadingView() {
    if (!isVisible()) return this;
    if (mSwitcherLoading.getCurrentView().getId() == R.id.view_content_switcher) {
      mSwitcherLoading.showPrevious();
    }
    return this;
  }

  protected RecyclerFragment showContentView() {
    if (!isVisible()) return this;
    if (mSwitcherLoading.getCurrentView().getId() == R.id.view_loading) {
      mSwitcherLoading.showNext();
    }
    if (mSwitcherContent.getCurrentView().getId() != R.id.view_recycler) {
      mSwitcherContent.showPrevious();
    }
    return this;
  }

  protected RecyclerFragment showEmptyView() {
    if (!isVisible()) return this;
    if (mSwitcherLoading.getCurrentView().getId() == R.id.view_loading) {
      mSwitcherLoading.showNext();
    }
    if (mSwitcherContent.getCurrentView().getId() == R.id.view_recycler) {
      mSwitcherContent.showNext();
    }
    return this;
  }

  @Override public void onRefresh() {
    mDataLoader.refresh();
  }

  @Override public boolean canChildScrollUp() {
    return ViewCompat.canScrollVertically(mRecyclerView, -1);
  }

  protected void updateRecyclerView() {
    if (!isVisible()) return;

    mAdapter.updateData(mItems);
    if (mItems.isEmpty()) {
      showEmptyView();
    } else {
      showContentView();
    }
  }

  public DataLoader<T> getDataLoader() {
    return mDataLoader;
  }

  public DelegatedSwipeRefreshLayout getRefreshLayout() {
    return mSwipeRefreshLayout;
  }

  public RecyclerView getRecyclerView() {
    return mRecyclerView;
  }

  @Override public void beforeRefresh() {
  }

  @Override public void beforeLoadMore() {
  }

  @Override public T register(T resource) {
    return resource;
  }

  @Override public void onRequestComplete(Response<ArrayList<T>> response) {
    mItems = mDataLoader.getResources();
    mDataHasLoaded = !mItems.isEmpty();
  }

  @Override public void onRequestFailure(Response<ArrayList<T>> response) {
    mDataHasLoaded = false;
  }

  @Override public void onRequestFailure(RetrofitError error) {
    mDataHasLoaded = false;
  }

  @Override public void onRequestNetworkError() {
    mDataHasLoaded = false;
  }

  @Override public void onFinish() {
    stopRefreshing();
    updateRecyclerView();
  }

  @Override public boolean canCache() {
    return false;
  }

  @Override public void handleLocalCache(ArrayList<T> data) {

  }

  @Override public void onCacheLoaded(ArrayList<T> localData) {
    if (localData == null || localData.isEmpty()) return;
    for (T data : localData) {
      register(data);
    }
    mItems.addAll(localData);
    if (!mItems.isEmpty()) {
      updateRecyclerView();
    }
  }

  public ArrayList<T> getItems() {
    return mItems;
  }

  public RecyclerFragment setEmptyText(int resId) {
    setEmptyText(getString(resId));
    return this;
  }

  public RecyclerFragment setEmptyText(String empty) {
    mEmptyView.setText(empty);
    return this;
  }

  public RecyclerFragment setEmptyTextColor(int color) {
    mEmptyView.setTextColor(color);
    return this;
  }

  public RecyclerFragment setEmptyDrawablePadding(int pad) {
    mEmptyView.setCompoundDrawablePadding(pad);
    return this;
  }

  public RecyclerFragment setEmptyTopDrawable(int top) {
    return setEmptyTopDrawable(0, top, 0, 0);
  }

  public RecyclerFragment setEmptyTopDrawable(
      int start, int top, int end, int bottom) {
    TextViewCompat.setCompoundDrawablesRelativeWithIntrinsicBounds(mEmptyView, start, top, end, bottom);
    return this;
  }

  public LinearLayout getBtnContainerView() {
    return mBtnContainerView;
  }
}
