package com.laputapp.ui.adapter;

import android.content.Context;
import android.support.annotation.NonNull;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;

import java.util.ArrayList;
import java.util.List;

/**
 * An adapter that used with any {@link ViewGroup}.
 */
public abstract class GeneralAdapter<T> {

    public static final int NO_POSITION = -1;

    private ViewGroup mParent;
    private List<View> mChildren;

    private List<T> mData;

    public GeneralAdapter(@NonNull ViewGroup parent) {
        mParent = parent;
        mChildren = new ArrayList<>();
    }

    public int getCount() {
        return mData != null ? mData.size() : 0;
    }

    public T getItem(int position) {
        return mData.get(position);
    }

    @SuppressWarnings("unchecked")
    public View getView(int position, View convertView, ViewGroup parent) {
        GeneralAdapter.ViewHolder<T> viewHolder;
        if (convertView == null) {
            convertView = LayoutInflater.from(parent.getContext())
                    .inflate(getItemLayoutResId(position), parent, false);
            viewHolder = onCreateViewHolder(convertView);
            convertView.setTag(viewHolder);
            mChildren.add(convertView);
        } else {
            viewHolder = (GeneralAdapter.ViewHolder<T>) convertView.getTag();
        }
        viewHolder.bind(getItem(position), position);
        return convertView;
    }

    public abstract GeneralAdapter.ViewHolder<T> onCreateViewHolder(View view);

    public void updateData(List<T> data) {
        mData = data;
        notifyDataSetChanged();
    }

    public void notifyDataSetChanged() {
        mParent.removeAllViews();
        for (int i = 0; i < getCount(); i++) {
            View child = getView(i, i < mChildren.size() ? mChildren.get(i) : null, mParent);
            mParent.addView(child, i);
        }
    }

    public abstract int getItemLayoutResId(int position);

    public static class ViewHolder<T> {

        public final View itemView;
        private int mPosition = NO_POSITION;

        public ViewHolder(View itemView) {
            this.itemView = itemView;
        }

        public Context getContext() {
            return itemView.getContext();
        }

        public int getPosition() {
            return mPosition;
        }

        public void bind(T item, int position) {
            mPosition = position;
            if (item instanceof PositionConsumer) {
                ((PositionConsumer) item).setPosition(position);
            }
        }
    }
}