/*
 * Copyright (C) 2005-2020 ManyDesigns srl.  All rights reserved.
 * http://www.manydesigns.com/
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 3 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package com.manydesigns.elements.forms;

import com.manydesigns.elements.FormElement;
import com.manydesigns.elements.KeyValueAccessor;
import com.manydesigns.elements.Mode;
import com.manydesigns.elements.composites.AbstractCompositeElement;
import com.manydesigns.elements.fields.MultipartRequestField;
import com.manydesigns.elements.xml.XhtmlBuffer;
import org.jetbrains.annotations.NotNull;

import java.util.HashMap;
import java.util.Map;

/*
* @author Paolo Predonzani     - paolo.predonzani@manydesigns.com
* @author Angelo Lupo          - angelo.lupo@manydesigns.com
* @author Giampiero Granatella - giampiero.granatella@manydesigns.com
* @author Alessio Stalla       - alessio.stalla@manydesigns.com
*/
public class FieldSet extends AbstractCompositeElement<FormElement> {
    public static final String copyright =
            "Copyright (C) 2005-2020 ManyDesigns srl";

    protected final int nColumns;
    protected final Mode mode;

    protected String name;

    public FieldSet(String name, int nColumns, Mode mode) {
        this.name = name;
        this.nColumns = nColumns;
        this.mode = mode;
        if(nColumns > 4 && nColumns != 6) {
            throw new IllegalArgumentException("nColumns = " + nColumns + " but only 1, 2, 3, 4, 6 columns are supported");
        }
    }

    public void toXhtml(@NotNull XhtmlBuffer xb) {
        if (mode.isHidden()) {
            for (FormElement current : this) {
                current.toXhtml(xb);
            }
        } else {
            xb.openElement("fieldset");
            xb.addAttribute("class", "mde-columns-" + nColumns);
            if (name != null) {
                xb.writeLegend(name, null);
            }

            int currentColumn = 0;

            for (FormElement current : this) {
                int colSpan = Math.min(current.getColSpan(), nColumns);
                if (current.isForceNewRow() || currentColumn + colSpan > nColumns) {
                    if(currentColumn > 0) {
                        closeCurrentRow(xb, currentColumn);
                        currentColumn = 0;
                    }
                }

                if (currentColumn == 0) {
                    xb.openElement("div");
                    xb.addAttribute("class", "row");
                }

                xb.openElement("div");
                xb.addAttribute("class", "col-md-" + (colSpan * (12 / nColumns)) + " mde-colspan-" + colSpan);
                current.toXhtml(xb);
                xb.closeElement("div");

                currentColumn = currentColumn + colSpan;

                if (currentColumn >= nColumns) {
                    closeCurrentRow(xb, currentColumn);
                    currentColumn = 0;
                }
            }

            if(currentColumn > 0) {
                closeCurrentRow(xb, currentColumn);
            }
            xb.closeElement("fieldset");
        }
    }

    protected void closeCurrentRow(XhtmlBuffer xb, int currentColumn) {
        if (currentColumn < nColumns) {
            xb.openElement("div");
            xb.addAttribute("class", "col-md-" + ((nColumns - currentColumn) * (12 / nColumns)));
            xb.closeElement("div");
        }
        xb.closeElement("div");
    }

    public String getName() {
        return name;
    }

    public void setName(String name) {
        this.name = name;
    }

    public Mode getMode() {
        return mode;
    }

    public boolean isRequiredFieldsPresent() {
        for (FormElement current : this) {
            if (current.hasRequiredFields()) {
                return true;
            }
        }
        return false;
    }

    public boolean isMultipartRequest() {
        for (FormElement current : this) {
            if (current instanceof MultipartRequestField) {
                MultipartRequestField field = (MultipartRequestField) current;
                if(!field.getMode().isView(field.isInsertable(), field.isUpdatable())) {
                    return true;
                }
            }
        }
        return false;
    }

    public void readFrom(KeyValueAccessor accessor) {
        for (FormElement current : this) {
            current.readFrom(accessor);
        }
    }

    public void writeTo(KeyValueAccessor accessor) {
        for (FormElement current : this) {
            current.writeTo(accessor);
        }
    }

    public Map<String, Object> getValue() {
        Map<String, Object> value = new HashMap<>();
        for (FormElement current : this) {
            value.put(current.getName(), current.getValue());
        }
        return value;
    }

    public int getNColumns() {
        return nColumns;
    }
}
