/*
 * Copyright (C) 2005-2020 ManyDesigns srl.  All rights reserved.
 * http://www.manydesigns.com/
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 3 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package com.manydesigns.portofino.database.platforms;

import com.manydesigns.portofino.model.database.ConnectionProvider;
import com.manydesigns.portofino.model.database.platforms.AbstractDatabasePlatform;
import org.hibernate.dialect.H2Dialect;

import java.sql.Connection;
import java.sql.Statement;

/*
* @author Paolo Predonzani     - paolo.predonzani@manydesigns.com
* @author Angelo Lupo          - angelo.lupo@manydesigns.com
* @author Giampiero Granatella - giampiero.granatella@manydesigns.com
* @author Alessio Stalla       - alessio.stalla@manydesigns.com
*/
public class H2DatabasePlatform extends AbstractDatabasePlatform {
    public static final String copyright =
            "Copyright (C) 2005-2020 ManyDesigns srl";

    public final static String DESCRIPTION = "H2";
    public final static String STANDARD_DRIVER_CLASS_NAME =
            "org.h2.Driver";

    //**************************************************************************
    // Constructors
    //**************************************************************************

    public H2DatabasePlatform() {
        super(H2Dialect.class.getName(), "jdbc:h2:<database or connection spec>");
    }

    //**************************************************************************
    // Implementation of DatabaseAbstraction
    //**************************************************************************

    public String getDescription() {
        return DESCRIPTION;
    }

    public String getStandardDriverClassName() {
        return STANDARD_DRIVER_CLASS_NAME;
    }

    public boolean isApplicable(ConnectionProvider connectionProvider) {
        return "H2".equals(connectionProvider.getDatabaseProductName());
    }

    @Override
    public void shutdown(ConnectionProvider connectionProvider) {
        super.shutdown(connectionProvider);
        try(Connection connection = connectionProvider.acquireConnection();
            Statement statement = connection.createStatement()) {
            statement.execute("SHUTDOWN");
        } catch (Exception e) {
            logger.warn("Could not shutdown connection provider " + connectionProvider.getDatabase().getDatabaseName(), e);
        }
    }
}
