/*
 * The MIT License
 *
 * Copyright 2014 Tim Boudreau.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */
package com.mastfrog.acteur.errors;

import io.netty.handler.codec.http.HttpResponseStatus;
import static java.util.Collections.emptyMap;
import static java.util.Collections.unmodifiableMap;
import java.util.LinkedHashMap;
import java.util.Map;

/**
 * Exception that can be thrown from a Provider and result in a specific HTTP
 * error response. Please do not use where you can manage this without
 * exceptions - this is a last resort for when you don't have access to the
 * Acteur to set its state.
 *
 * @author Tim Boudreau
 */
public class ResponseException extends RuntimeException {

    private final HttpResponseStatus status;
    private Map<CharSequence, CharSequence> headers;

    public ResponseException(HttpResponseStatus status, String msg) {
        super(msg);
        this.status = status;
    }

    public HttpResponseStatus status() {
        return status;
    }

    public synchronized ResponseException withHeader(CharSequence name, CharSequence value) {
        if (headers == null) {
            headers = new LinkedHashMap<>();
        }
        headers.put(name, value);
        return this;
    }

    public Map<CharSequence, CharSequence> headers() {
        if (headers == null) {
            return emptyMap();
        }
        return unmodifiableMap(headers);
    }

}
