/*
 * The MIT License
 *
 * Copyright 2013 Tim Boudreau.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */
package com.mastfrog.acteur.server;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.inject.util.Providers;
import com.mastfrog.acteur.ContentConverter;
import com.mastfrog.acteur.HttpEvent;
import com.mastfrog.acteur.headers.HeaderValueType;
import com.mastfrog.acteur.headers.Headers;
import com.mastfrog.acteur.headers.Method;
import com.mastfrog.mime.MimeType;
import com.mastfrog.url.Path;
import com.mastfrog.url.Protocol;
import com.mastfrog.url.Protocols;
import com.mastfrog.url.URL;
import com.mastfrog.util.codec.Codec;
import com.mastfrog.util.collections.CollectionUtils;
import static com.mastfrog.util.preconditions.Checks.nonNegative;
import static com.mastfrog.util.preconditions.Checks.notNull;
import com.mastfrog.util.strings.Strings;
import io.netty.buffer.ByteBuf;
import io.netty.buffer.ByteBufHolder;
import io.netty.buffer.Unpooled;
import io.netty.channel.Channel;
import io.netty.channel.ChannelHandlerContext;
import static io.netty.handler.codec.http.HttpHeaderNames.CONNECTION;
import static io.netty.handler.codec.http.HttpHeaderNames.HOST;
import static io.netty.handler.codec.http.HttpHeaderValues.KEEP_ALIVE;
import io.netty.handler.codec.http.HttpRequest;
import io.netty.handler.codec.http.QueryStringDecoder;
import io.netty.util.AsciiString;
import static io.netty.util.CharsetUtil.UTF_8;
import java.io.IOException;
import java.net.InetSocketAddress;
import java.net.SocketAddress;
import java.nio.charset.Charset;
import static java.util.Collections.unmodifiableMap;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.TreeMap;
import java.util.TreeSet;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 *
 * @author Tim Boudreau
 */
final class EventImpl implements HttpEvent {

    private static final Pattern PORT_PATTERN = Pattern.compile(".*?:(\\d+)");
    private static final AsciiString HTTPS = AsciiString.of("https");
    private final HttpRequest req;
    private final Path path;
    private final SocketAddress address;
    private final PathFactory paths;
    private boolean neverKeepAlive = false;
    private final ChannelHandlerContext channel;
    private ContentConverter converter;
    private boolean ssl;
    private boolean early;
    private Map<String, String> paramsMap;

    EventImpl(HttpRequest req, PathFactory paths) {
        this.req = req;
        this.path = paths.toPath(req.uri());
        this.paths = paths;
        address = new InetSocketAddress("timboudreau.com", 8985); //XXX for tests
        this.channel = null;
        Codec codec = new ServerModule.CodecImpl(Providers.of(new ObjectMapper()));
        this.converter = new ContentConverter(codec, Providers.of(UTF_8), null);
    }

    EventImpl(HttpRequest req, SocketAddress addr, ChannelHandlerContext channel, PathFactory paths, ContentConverter converter, boolean ssl) {
        this.req = req;
        this.path = paths.toPath(req.uri());
        address = addr;
        this.channel = channel;
        this.converter = converter;
        this.ssl = ssl;
        this.paths = paths;
    }

    @Override
    public boolean isPreContent() {
        return early;
    }

    /**
     * Returns a best-effort at reconstructing the inbound URL, following the
     * following algorithm:
     * <ul>
     * <li>If the application has external url generation configured via
     * PathFactory, prefer the output of that</li>
     * <li>If not, try to honor non-standard but common headers such as
     * <code>X-Forwarded-Proto, X-URI-Scheme, Forwarded, X-Forwarded-Host</code></li>
     * </ul>
     *
     * Applications which respond to multiple virtual host names may need a
     * custom implementation of PathFactory bound to do this correctly.
     *
     * @return A URL string
     */
    public String getRequestURL(boolean preferHeaders) {
        HttpEvent evt = this;
        String uri = evt.request().uri();
        if (uri.startsWith("http://") || uri.startsWith("https://")) {
            return uri;
        }
        URL takeFrom = paths.constructURL("/");
        CharSequence proto = DefaultPathFactory.findProtocol(evt);
        if (proto == null) {
            proto = takeFrom.getProtocol().toString();
        }
        String host = evt.header("X-Forwarded-Host");
        if (host == null) {
            host = evt.header(HOST);
        }
        int port = -1;
        if (host != null && host.indexOf(':') > 0) {
            port = findPort(host);
            host = host.substring(0, host.indexOf(':'));
            if (port != -1 && Protocols.forName(proto.toString()).getDefaultPort().intValue() == port) {
                port = -1;
            } else if (port != -1 && Protocols.forName(proto.toString()).getDefaultPort().intValue() != port) {
                host = host + ":" + port;
            }
        }
        if (!preferHeaders) {
            String configuredHost = takeFrom.getHost().toString();
            if (!configuredHost.equals("localhost")) {
                host = configuredHost;
                port = takeFrom.getPort().intValue();
                proto = takeFrom.getProtocol().toString();
                Protocol protocol = com.mastfrog.url.Protocols.forName(proto.toString());
                if (port != -1 && port != protocol.getDefaultPort().intValue()) {
                    host = host + ":" + takeFrom.getPort().intValue();
                }
            }
        }
        if (uri.length() == 0 || uri.charAt(0) != '/') {
            host += '/';
        }
        return proto + "://" + host + uri;
    }

    private static int findPort(String what) {
        Matcher m = PORT_PATTERN.matcher(what);
        if (m.find()) {
            return Integer.parseInt(m.group(1));
        }
        return -1;
    }

    EventImpl early() {
        early = true;
        return this;
    }

    public boolean isSsl() {
        boolean result = ssl;
        if (!result) {
            CharSequence cs = header(Headers.X_FORWARDED_PROTO);
            if (cs != null) {
                result = HTTPS.contentEqualsIgnoreCase(HTTPS);
            }
        }
        return result;
    }

    @Override
    public String toString() {
        return req.method() + "\t" + req.uri();
    }

    public void setNeverKeepAlive(boolean val) {
        neverKeepAlive = val;
    }

    @Override
    public Channel channel() {
        return channel.channel();
    }

    @Override
    public ChannelHandlerContext ctx() {
        return channel;
    }

    @Override
    public ByteBuf content() {
        return req instanceof ByteBufHolder ? ((ByteBufHolder) req).content()
                : Unpooled.EMPTY_BUFFER;
    }

    @Override
    public <T> T jsonContent(Class<T> type) throws Exception {
        MimeType mimeType = header(Headers.CONTENT_TYPE);
        if (mimeType == null) {
            mimeType = MimeType.ANY_TYPE;
        }
        return converter.toObject(content(), mimeType, type);
    }

    @Override
    public String stringContent() throws IOException {
        ByteBuf content = content();
        if (content == null) {
            return "";
        }
        MimeType type = header(Headers.CONTENT_TYPE);
        if (type == null) {
            type = MimeType.PLAIN_TEXT_UTF_8;
        }
        return converter.toString(content,
                type.charset().orElse(UTF_8));
    }

    @Override
    public HttpRequest request() {
        return req;
    }

    @Override
    public Method method() {
        try {
            return Method.get(req);
        } catch (IllegalArgumentException e) {
            return Method.UNKNOWN;
        }
    }

    @Override
    public SocketAddress remoteAddress() {
        return address;
    }

    @Override
    public String header(CharSequence nm) {
        return req.headers().get(notNull("nm", nm));
    }

    @Override
    public String urlParameter(String param) {
        return urlParametersAsMap().get(param);
    }

    @Override
    public Path path() {
        return path;
    }

    @Override
    public <T> List<T> headers(HeaderValueType<T> headerType) {
        List<CharSequence> headers = CollectionUtils.<CharSequence>generalize(request().headers().getAll(headerType.name()));
        return CollectionUtils.convertedList(headers, headerType, CharSequence.class, headerType.type());
    }

    @Override
    public Map<CharSequence, CharSequence> headersAsMap() {
        Map<CharSequence, CharSequence> headers = CollectionUtils.caseInsensitiveStringMap();
        for (Map.Entry<String, String> e : request().headers().entries()) {
            headers.put(e.getKey(), e.getValue());
        }
        return headers;
    }

    @Override
    public <T> T header(HeaderValueType<T> value) {
        String header = header(value.name());
        if (header != null) {
            return value.toValue(header);
        }
        return null;
    }

    @Override
    public synchronized Map<String, String> urlParametersAsMap() {
        if (paramsMap == null) {
            QueryStringDecoder queryStringDecoder = new QueryStringDecoder(req.uri());
            Map<String, List<String>> params = queryStringDecoder.parameters();
            Map<String, String> result = new TreeMap<>();
            for (Map.Entry<String, List<String>> e : params.entrySet()) {
                if (e.getValue().isEmpty()) {
                    continue;
                }
                result.put(e.getKey(), e.getValue().get(0));
            }
            paramsMap = unmodifiableMap(result);
        }
        return paramsMap;
    }

    @Override
    @SuppressWarnings("unchecked")
    public <T> T urlParametersAs(Class<T> type) {
        return converter.toObject(urlParametersAsMap(), type);
    }

    @Override
    public boolean requestsConnectionStayOpen() {
        if (neverKeepAlive) {
            return false;
        }
        boolean hasKeepAlive = req.headers()
                .contains(CONNECTION, KEEP_ALIVE, true);
        return hasKeepAlive;
    }

    @Override
    @SuppressWarnings("deprecation")
    public Optional<Integer> intUrlParameter(String name) {
        String val = urlParameter(name);
        if (val != null) {
            int ival = Integer.parseInt(val);
            return Optional.of(ival);
        }
        return Optional.empty();
    }

    @Override
    @SuppressWarnings("deprecation")
    public Optional<Long> longUrlParameter(String name) {
        String val = urlParameter(name);
        if (val != null) {
            long lval = Long.parseLong(val);
            return Optional.of(lval);
        }
        return Optional.empty();
    }

    public <T> java.util.Optional<T> httpHeader(HeaderValueType<T> header) {
        return java.util.Optional.ofNullable(header(notNull("header", header)));
    }

    @Override
    public java.util.Optional<CharSequence> httpHeader(CharSequence name) {
        return java.util.Optional.ofNullable(this.header(name));
    }

    @Override
    public java.util.Optional<CharSequence> uriAnchor() {
        String u = req.uri();
        int ix = u.indexOf('#');
        if (ix >= 0) {
            return java.util.Optional.<CharSequence>of(u.substring(ix + 1));
        }
        return java.util.Optional.empty();
    }

    @Override
    public java.util.Optional<CharSequence> uriPathElement(int index) {
        Path p = path();
        if (p.size() >= nonNegative("index", index)) {
            return java.util.Optional.empty();
        }
        return java.util.Optional.<CharSequence>of(p.getElement(index).toString());
    }

    @Override
    public java.util.Optional<CharSequence> uriQueryParameter(CharSequence name, boolean decode) {
        String val = urlParametersAsMap().get(notNull("name", name).toString());
        return java.util.Optional.<CharSequence>ofNullable(val);
    }

    @Override
    public String httpMethod() {
        return method().toString();
    }

    @Override
    public String requestUri(boolean preferHeaders) {
        return getRequestURL(preferHeaders);
    }

    @Override
    public Set<? extends CharSequence> httpHeaderNames() {
        Set<CharSequence> result = new TreeSet<>(Strings.charSequenceComparator(true));
        Iterator<Map.Entry<CharSequence, CharSequence>> it = req.headers().iteratorCharSequence();
        while (it.hasNext()) {
            result.add(it.next().getKey());
        }
        return result;
    }

}
