package com.meizu.cloud.pushsdk;

import android.content.Context;
import android.content.Intent;

import com.meizu.cloud.pushinternal.DebugLogger;
import com.meizu.cloud.pushsdk.constants.PushConstants;
import com.meizu.cloud.pushsdk.handler.AbstractAppLogicListener;
import com.meizu.cloud.pushsdk.handler.MessageHandler;
import com.meizu.cloud.pushsdk.handler.impl.MessageV2Handler;
import com.meizu.cloud.pushsdk.handler.impl.MessageV3Handler;
import com.meizu.cloud.pushsdk.handler.impl.notification.NotificationClickMessageHandler;
import com.meizu.cloud.pushsdk.handler.impl.notification.NotificationDeleteMessageHandler;
import com.meizu.cloud.pushsdk.handler.impl.platform.PushSwitchStatusHandler;
import com.meizu.cloud.pushsdk.handler.impl.RegisterMessageHandler;
import com.meizu.cloud.pushsdk.handler.impl.platform.ReceiveNotifyMessageHandler;
import com.meizu.cloud.pushsdk.handler.impl.platform.RegisterStatusHandler;
import com.meizu.cloud.pushsdk.handler.impl.ThroughMessageHandler;
import com.meizu.cloud.pushsdk.handler.impl.UnRegisterMessageHandler;
import com.meizu.cloud.pushsdk.handler.impl.platform.SubScribeAliasStatusHandler;
import com.meizu.cloud.pushsdk.handler.impl.platform.SubScribeTagsStatusHandler;
import com.meizu.cloud.pushsdk.handler.impl.platform.UnRegisterStatusHandler;
import com.meizu.cloud.pushsdk.handler.impl.schedule.ScheduleNotificationHandler;
import com.meizu.cloud.pushsdk.notification.PushNotificationBuilder;
import com.meizu.cloud.pushsdk.platform.message.PushSwitchStatus;
import com.meizu.cloud.pushsdk.platform.message.RegisterStatus;
import com.meizu.cloud.pushsdk.platform.message.SubAliasStatus;
import com.meizu.cloud.pushsdk.platform.message.SubTagsStatus;
import com.meizu.cloud.pushsdk.platform.message.UnRegisterStatus;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 *
 *
 *  +-------------------+             +------------------+                                                    +--------+
 *  | onReceiveMessage  |----------->>|ParseIntent start |                                                    |  End   |
 *  +-------------------+             +------------------+                                                    +--------+
 *                                            ||                                                                  /|\
 *                                            \/                                                                   |
 *                             +---------------------------------+                                                 |
 *                             |      RegisterMessageHandler     |-------->  send pushId to app--------------------|
 *                             +---------------------------------+                                                 |
 *                                            ||                                                                   |
 *                                            \/                                                                   |
 *                             +---------------------------------+                                                /|\
 *                             |     UnRegisterMessageHandler    |---------> send unPush status to app-------------|
 *                             +---------------------------------+                                                 |
 *                                            ||                                                                   |
 *                                            \/                                                                   |
 *                             +---------------------------------+                                                /|\
 *                             |        MessageV3Handler         |---------> show v3 Notification------------------|
 *                             +---------------------------------+                                                 |
 *                                            ||                                                                   |
 *                                            \/                                                                   |
 *                             +---------------------------------+                                                /|\
 *                             |          MessageV2Handler       |---------> show V2 Notification------------------|
 *                             +---------------------------------+                                                 |
 *                                            ||                                                                   |
 *                                            \/                                                                   |
 *                             +---------------------------------+                                                /|\
 *                             |      ThroughMessageHandler      |---------> send message to app-------------------|
 *                             +---------------------------------+                                                 |
 *                                            ||                                                                   |
 *                                            \/                                                                   |
 *                             +---------------------------------+                                                /|\
 *                             | NotificationClickMessageHandler |---------> process notification message----------|
 *                             +---------------------------------+
 *                                            ||                                                                   |
 *                                            \/                                                                   |
 *                             +---------------------------------+                                                /|\
 *                             | NotificationDeleteMessageHandler |--------->process delete notification message---|
 *                             +---------------------------------+
 *
 ******/
public class PushMessageProxy {
    private static final String TAG = "PushMessageProxy";
    private Context context;
    private Map<Integer,MessageHandler> managerHashMap = new HashMap<>();
    static volatile PushMessageProxy singleton = null;
    private Map<String,AbstractAppLogicListener> messageListenerMap = null;

    public PushMessageProxy(Context context){
        this(context,null);
    }

    public PushMessageProxy(Context context,List<MessageHandler> messageManagerList) {
        this(context,messageManagerList,null);
    }


    public PushMessageProxy(Context context,List<MessageHandler> messageManagerList,AbstractAppLogicListener receiverLisener) {
        if (context == null) {
            throw new IllegalArgumentException("Context must not be null.");
        }
        this.context = context.getApplicationContext();
        messageListenerMap = new HashMap<>();
        AbstractAppLogicListener abstractAppLogicListener = new DefaultPushMessageListener();
        if(messageManagerList == null){
            addHandler(new MessageV3Handler(context, abstractAppLogicListener));
            addHandler(new MessageV2Handler(context, abstractAppLogicListener));
            addHandler(new ThroughMessageHandler(context, abstractAppLogicListener));
            addHandler(new NotificationClickMessageHandler(context,abstractAppLogicListener));
            addHandler(new RegisterMessageHandler(context,abstractAppLogicListener));
            addHandler(new UnRegisterMessageHandler(context,abstractAppLogicListener));
            addHandler(new NotificationDeleteMessageHandler(context,abstractAppLogicListener));
            addHandler(new PushSwitchStatusHandler(context,abstractAppLogicListener));
            addHandler(new RegisterStatusHandler(context,abstractAppLogicListener));
            addHandler(new UnRegisterStatusHandler(context,abstractAppLogicListener));
            addHandler(new SubScribeAliasStatusHandler(context,abstractAppLogicListener));
            addHandler(new SubScribeTagsStatusHandler(context,abstractAppLogicListener));
            addHandler(new ScheduleNotificationHandler(context,abstractAppLogicListener));
            addHandler(new ReceiveNotifyMessageHandler(context,abstractAppLogicListener));
        } else {
            addHandler(messageManagerList);
        }

    }

    public static PushMessageProxy with(Context context) {
        if (singleton == null) {
            synchronized (PushMessageProxy.class) {
                if (singleton == null) {
                    DebugLogger.i(TAG, "PushMessageProxy init");
                    singleton = new PushMessageProxy(context);
                }
            }
        }
        return singleton;
    }


    public PushMessageProxy receiverListener(String tag,AbstractAppLogicListener abstractAppLogicListener){
        messageListenerMap.put(tag, abstractAppLogicListener);
        return this;
    }

    public PushMessageProxy unReceiverListener(String tag){
        messageListenerMap.put(tag, null);
        return this;
    }

    /**
     * add single Message manager
     * */
    public PushMessageProxy addHandler(MessageHandler messageManager){
        managerHashMap.put(messageManager.getProcessorType(),messageManager);
        return this;
    }

    /**
     * add manager list
     * */
    public PushMessageProxy addHandler(List<MessageHandler> messageManagerList){
        if(messageManagerList == null){
            throw new IllegalArgumentException("messageManagerList must not be null.");
        }
        for (MessageHandler messageManager : messageManagerList){
            addHandler(messageManager);
        }
        return this;
    }


    /**
     * process Message receiver intent
     *
     * @param intent
     * */
    public void processMessage(Intent intent){
        DebugLogger.e(TAG,"is onMainThread "+isOnMainThread());
        try {
            String method = intent.getStringExtra(PushConstants.MZ_PUSH_MESSAGE_METHOD);
            DebugLogger.i(TAG, "receive action " + intent.getAction() +" method "+method);
            if(intent != null){
                for (Map.Entry<Integer, MessageHandler> iterator : managerHashMap.entrySet()){
                    MessageHandler messageManager = iterator.getValue();
                    boolean mathResult = messageManager.sendMessage(intent);
                    if(mathResult){
                        break;
                    }
                }
            }
        } catch (Exception e){
            DebugLogger.e(TAG,"processMessage error "+e.getMessage());
        }

    }

    /**
     * check current thread
     * */
    protected boolean isOnMainThread(){
        return Thread.currentThread() == context.getMainLooper().getThread();
    }

    public class DefaultPushMessageListener extends AbstractAppLogicListener{

        @Override
        public void onMessage(Context context, Intent intent) {
            for(Map.Entry<String, AbstractAppLogicListener> entry : messageListenerMap.entrySet()){
                AbstractAppLogicListener abstractAppLogicListener = entry.getValue();
                if(abstractAppLogicListener != null){
                    abstractAppLogicListener.onMessage(context,intent);

                }
            }
        }

        @Override
        public void onRegister(Context context, String pushId) {
            for(Map.Entry<String, AbstractAppLogicListener> entry : messageListenerMap.entrySet()){
                AbstractAppLogicListener abstractAppLogicListener = entry.getValue();
                if(abstractAppLogicListener != null){
                    abstractAppLogicListener.onRegister(context, pushId);

                }
            }
        }

        @Override
        public void onMessage(Context context, String message) {
            for(Map.Entry<String, AbstractAppLogicListener> entry : messageListenerMap.entrySet()){
                AbstractAppLogicListener abstractAppLogicListener = entry.getValue();
                if(abstractAppLogicListener != null){
                    abstractAppLogicListener.onMessage(context, message);
                }
            }
        }

        @Override
        public void onUnRegister(Context context, boolean success) {
            for(Map.Entry<String, AbstractAppLogicListener> entry : messageListenerMap.entrySet()){
                AbstractAppLogicListener abstractAppLogicListener = entry.getValue();
                if(abstractAppLogicListener != null){
                    abstractAppLogicListener.onUnRegister(context, success);
                }
            }
        }

        @Override
        public void onUpdateNotificationBuilder(PushNotificationBuilder pushNotificationBuilder) {
            for(Map.Entry<String, AbstractAppLogicListener> entry : messageListenerMap.entrySet()){
                AbstractAppLogicListener abstractAppLogicListener = entry.getValue();
                if(abstractAppLogicListener != null){
                    abstractAppLogicListener.onUpdateNotificationBuilder(pushNotificationBuilder);
                }
            }
        }

        @Override
        public void onPushStatus(Context context,PushSwitchStatus pushSwitchStatus) {
            for(Map.Entry<String, AbstractAppLogicListener> entry : messageListenerMap.entrySet()){
                AbstractAppLogicListener abstractAppLogicListener = entry.getValue();
                if(abstractAppLogicListener != null){
                    abstractAppLogicListener.onPushStatus(context,pushSwitchStatus);
                }
            }
        }

        @Override
        public void onRegisterStatus(Context context,RegisterStatus registerStatus) {
            for(Map.Entry<String, AbstractAppLogicListener> entry : messageListenerMap.entrySet()){
                AbstractAppLogicListener abstractAppLogicListener = entry.getValue();
                if(abstractAppLogicListener != null){
                    abstractAppLogicListener.onRegisterStatus(context,registerStatus);
                }
            }
        }

        @Override
        public void onUnRegisterStatus(Context context,UnRegisterStatus unRegisterStatus) {
            for(Map.Entry<String, AbstractAppLogicListener> entry : messageListenerMap.entrySet()){
                AbstractAppLogicListener abstractAppLogicListener = entry.getValue();
                if(abstractAppLogicListener != null){
                    abstractAppLogicListener.onUnRegisterStatus(context,unRegisterStatus);
                }
            }
        }

        @Override
        public void onSubTagsStatus(Context context,SubTagsStatus subTagsStatus) {
            for(Map.Entry<String, AbstractAppLogicListener> entry : messageListenerMap.entrySet()){
                AbstractAppLogicListener abstractAppLogicListener = entry.getValue();
                if(abstractAppLogicListener != null){
                    abstractAppLogicListener.onSubTagsStatus(context,subTagsStatus);
                }
            }
        }

        @Override
        public void onSubAliasStatus(Context context,SubAliasStatus subAliasStatus) {
            for(Map.Entry<String, AbstractAppLogicListener> entry : messageListenerMap.entrySet()){
                AbstractAppLogicListener abstractAppLogicListener = entry.getValue();
                if(abstractAppLogicListener != null){
                    abstractAppLogicListener.onSubAliasStatus(context,subAliasStatus);
                }
            }
        }

        @Override
        public void onNotificationClicked(Context context, String title, String content, String selfDefineContentString) {
            for(Map.Entry<String, AbstractAppLogicListener> entry : messageListenerMap.entrySet()){
                AbstractAppLogicListener abstractAppLogicListener = entry.getValue();
                if(abstractAppLogicListener != null){
                    abstractAppLogicListener.onNotificationClicked(context,title,content,selfDefineContentString);
                }
            }
        }

        @Override
        public void onNotificationArrived(Context context, String title, String content, String selfDefineContentString) {
            for(Map.Entry<String, AbstractAppLogicListener> entry : messageListenerMap.entrySet()){
                AbstractAppLogicListener abstractAppLogicListener = entry.getValue();
                if(abstractAppLogicListener != null){
                    abstractAppLogicListener.onNotificationArrived(context,title,content,selfDefineContentString);
                }
            }
        }

        @Override
        public void onNotificationDeleted(Context context, String title, String content, String selfDefineContentString) {
            for(Map.Entry<String, AbstractAppLogicListener> entry : messageListenerMap.entrySet()){
                AbstractAppLogicListener abstractAppLogicListener = entry.getValue();
                if(abstractAppLogicListener != null){
                    abstractAppLogicListener.onNotificationDeleted(context,title,content,selfDefineContentString);
                }
            }
        }

        @Override
        public void onNotifyMessageArrived(Context context, String message) {
            for(Map.Entry<String, AbstractAppLogicListener> entry : messageListenerMap.entrySet()){
                AbstractAppLogicListener abstractAppLogicListener = entry.getValue();
                if(abstractAppLogicListener != null){
                    abstractAppLogicListener.onNotifyMessageArrived(context,message);
                }
            }
        }
    }

}
