package com.meizu.cloud.pushsdk.common.util;

import android.text.TextUtils;

import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.util.Arrays;
import java.util.HashMap;

/**
 * Created by zbin on 2016/1/14 0014.
 */
public abstract class Reflector {
    private final static String sTag = "Reflector";

    protected Object[] mArgs;
    protected Class<?>[] mTypes;
    protected String mMethodName;

    private static HashMap<String, Class<?>> sClassCache;
    private static HashMap<Class<?>, HashMap<String, Method>> sMethodCache;
    private static HashMap<String, Field> mFiledCache;

    static {
        sClassCache = new HashMap<>();
        sMethodCache = new HashMap<>();
        mFiledCache = new HashMap<>();
    }

    /**
     * 通过待反射对象获取Reflector对象
     * @param object 待反射的对象
     * @return
     */
    public static Reflector from(Object object) {
        return new Obj(object);
    }

    /**
     * 通过待反射类获取Reflector对象
     * @param clz 待反射的类
     * @return
     */
    public static Reflector from(Class<?> clz) {
        return new Clz(clz);
    }

    /**
     * 通过待反射类名获取Reflector对象
     * @param clzName 待反射类名
     * @return
     */
    public static Reflector forName(String clzName) {
        Class<?> clz = sClassCache.get(clzName);
        if(clz == null) {
            try {
                clz = Class.forName(clzName);
                sClassCache.put(clzName, clz);
            } catch (ClassNotFoundException ignore) {
                Logger.trace(sTag, ignore);
            }
        }
        return from(clz);
    }

    private static Class<?>[] getParamTypes(Object[] args) {
        Class<?>[] paramTypes = null;
        if (args != null) {
            paramTypes = new Class<?>[args.length];
            for (int i = 0; i < args.length; ++i) {
                paramTypes[i] = args[i].getClass();
            }
        }
        return paramTypes;
    }

    public abstract <T> Result<T> invoke();

    protected abstract Class<?> getObjClass();

    protected abstract Object getObj();

    /**
     * 通过反射设置反射对象的属性值，设置完成之后会内部会再次读取一次，然后将其返回
     * @param filedName 属性名称
     * @param value 该属性的值
     * @param <T> 属性的类型
     * @return 设置完成之后再次读取的值
     */
    public <T> Result<T> setFieldValue(String filedName, T value) {
        Result<T> result = new Result<>();
        try {
            Field field = mFiledCache.get(filedName);
            if(field == null) {
                field = getObjClass().getDeclaredField(filedName);
                mFiledCache.put(filedName, field);
            }
            if(field != null) {
                field.setAccessible(true);
                field.set(getObj(), value);
                result = getFieldValue(filedName);
            }
        } catch (Exception ignore) {
            Logger.trace(sTag, ignore);
        }

        Logger.e(sTag, "set " + filedName + " value " + value + " result:" + result);

        return result;
    }

    /**
     * 通过反射读取反射对象的属性值
     * @param filedName 属性名称
     * @param <T> 属性值的类型
     * @return 反射结果
     */
    public <T> Result<T> getFieldValue(String filedName) {
        Result<T> result = new Result<>();
        try {
            Field field = mFiledCache.get(filedName);
            if (field == null) {
                field = getObjClass().getDeclaredField(filedName);
                mFiledCache.put(filedName, field);
            }
            if (field != null) {
                field.setAccessible(true);
                result.value = (T) field.get(getObj());
                result.ok = true;
            }
        } catch (Exception ignore) {
            Logger.trace(sTag, ignore);
        }

        Logger.d(sTag, "get " + filedName + " value result:" + result);

        return result;
    }

    private static Method getMethod(Class<?> clz, String name, Class... types) throws NoSuchMethodException {
        Method method;
        try {
            method = clz.getMethod(name, types);
        } catch (NoSuchMethodException e) {
            try {
                method = clz.getDeclaredMethod(name, types);
            } catch (NoSuchMethodException e1) {
                method = similarMethod(clz, name, types);
            }
        }
        return method;
    }

    /**
     * 查找最相近的方法
     * @param clz 类对象
     * @param name 方法名
     * @param types 参数类型
     * @return 方法对象
     * @throws NoSuchMethodException
     */
    private static Method similarMethod(Class<?> clz, String name, Class<?>[] types) throws NoSuchMethodException {
        for (Method method : clz.getMethods()) {
            if (isSimilarSignature(method, name, types)) {
                return method;
            }
        }

        for (Method method : clz.getDeclaredMethods()) {
            if (isSimilarSignature(method, name, types)) {
                return method;
            }
        }

        throw new NoSuchMethodException("No similar method " + name +
                " with params " + Arrays.toString(types) + " could be found on type " + clz);
    }

    private static boolean isSimilarSignature(Method possiblyMatchingMethod,
                                              String desiredMethodName, Class<?>[] desiredParamTypes) {
        return possiblyMatchingMethod.getName().equals(desiredMethodName) &&
                match(possiblyMatchingMethod.getParameterTypes(), desiredParamTypes);
    }

    private static boolean match(Class<?>[] declaredTypes, Class<?>[] actualTypes) {
        if (declaredTypes.length == actualTypes.length) {
            for (int i = 0; i < actualTypes.length; i++) {
                if (actualTypes[i] == NULL.class)
                    continue;

                if (wrapper(declaredTypes[i]).isAssignableFrom(wrapper(actualTypes[i])))
                    continue;

                return false;
            }

            return true;
        } else {
            return false;
        }
    }

    /**
     * 设置参数类型，如果setArgs中的对象类型与反射方法的参数类型完全一样（不存在父类对象上传递子类对象），可以不需要调用此方法
     * @param types 待反射方法的参数类型
     * @return
     */
    public Reflector setTypes(Class<?>[] types) {
        mTypes = types;
        return this;
    }

    /**
     * 设置待反射方法的参数
     * @param args 参数
     * @return
     */
    public Reflector setArgs(Object[] args) {
        mArgs = args;
        return this;
    }

    /**
     * 设置待反射方法的方法名
     * @param methodName 方法名
     * @return
     */
    public Reflector setMethodName(String methodName) {
        mMethodName = methodName;
        return this;
    }

    private static class Obj extends Reflector {
        private Object mObject;

        public Obj(Object object) {
            mObject = object;
        }

        @Override
        protected Class<?> getObjClass() {
            return mObject.getClass();
        }

        @Override
        protected Object getObj() {
            return mObject;
        }

        @Override
        public <T> Result<T> invoke() {
            Result<T> result = new Result<>();
            if (mObject != null &&
                    !TextUtils.isEmpty(mMethodName)) {
                try {
                    if ((mArgs != null && mArgs.length > 0) &&
                            mTypes == null) {
                        mTypes = getParamTypes(mArgs);
                    }
                    HashMap<String, Method> methodMap = sMethodCache.get(mObject.getClass());
                    if(methodMap == null) {
                        methodMap = new HashMap<>();
                        sMethodCache.put(mObject.getClass(), methodMap);
                    }
                    Method method = methodMap.get(mMethodName);
                    if(method == null) {
                        method = getMethod(mObject.getClass(), mMethodName, mTypes);
                        if (!method.isAccessible()) {
                            method.setAccessible(true);
                        }
                        methodMap.put(mMethodName, method);
                    }
                    result.value = (T) method.invoke(mObject, mArgs);
                    result.ok = true;
                } catch (Exception ignore) {
                    Logger.trace(sTag, ignore);
                }
            }
            Logger.d(sTag, "[Obj.invoke]:" +
                    ", mMethodName='" + mMethodName +
                    ", \nmTypes=" + Arrays.toString(mTypes) +
                    ", \nmArgs=" + Arrays.toString(mArgs) +
                    ", \nresult=" + result);
            return result;
        }
    }

    private static class Clz extends Reflector {
        private Class<?> mClz;

        public Clz(Class<?> clz) {
            mClz = clz;
        }

        @Override
        protected Class<?> getObjClass() {
            return mClz;
        }

        @Override
        protected Object getObj() {
            return mClz;
        }

        @Override
        public <T> Result<T> invoke() {
            Result<T> result = new Result<>();
            if (mClz != null &&
                    !TextUtils.isEmpty(mMethodName)) {
                if ((mArgs != null && mArgs.length > 0) &&
                        mTypes == null) {
                    mTypes = getParamTypes(mArgs);
                }
                try {
                    HashMap<String, Method> methodMap = sMethodCache.get(mClz);
                    if(methodMap == null) {
                        methodMap = new HashMap<>();
                        sMethodCache.put(mClz, methodMap);
                    }
                    Method method = methodMap.get(mMethodName);
                    if(method == null) {
                        method = getMethod(mClz, mMethodName, mTypes);
                        if (!method.isAccessible()) {
                            method.setAccessible(true);
                        }
                        methodMap.put(mMethodName, method);
                    }
                    result.value = (T) method.invoke(mClz, mArgs);
                    result.ok = true;
                } catch (Exception ignore) {
                    Logger.trace(sTag, ignore);
                }
            }
            Logger.d(sTag, "[Clz.invoke]:" +
                    ", mMethodName='" + mMethodName +
                    ", \nmTypes=" + Arrays.toString(mTypes) +
                    ", \nmArgs=" + Arrays.toString(mArgs) +
                    ", \nresult=" + result);
            return result;
        }
    }

    public static class Result<T> {
        public boolean ok = false;
        public T value;

        @Override
        public String toString() {
            return "Result{" +
                    "ok=" + ok +
                    ", value=" + value +
                    '}';
        }
    }

    public static Class<?> wrapper(Class<?> type) {
        if (type == null) {
            return null;
        }
        else if (type.isPrimitive()) {
            if (boolean.class == type) {
                return Boolean.class;
            }
            else if (int.class == type) {
                return Integer.class;
            }
            else if (long.class == type) {
                return Long.class;
            }
            else if (short.class == type) {
                return Short.class;
            }
            else if (byte.class == type) {
                return Byte.class;
            }
            else if (double.class == type) {
                return Double.class;
            }
            else if (float.class == type) {
                return Float.class;
            }
            else if (char.class == type) {
                return Character.class;
            }
            else if (void.class == type) {
                return Void.class;
            }
        }

        return type;
    }

    private static class NULL {}
}
