package com.meizu.cloud.pushsdk.notification;

import android.app.KeyguardManager;
import android.app.Notification;
import android.app.NotificationManager;
import android.app.PendingIntent;
import android.content.Context;
import android.content.Intent;
import android.content.pm.PackageManager;
import android.graphics.Bitmap;
import android.graphics.drawable.BitmapDrawable;
import android.net.Uri;
import android.os.Build;
import android.os.Handler;
import android.os.PowerManager;
import android.text.TextUtils;
import com.meizu.cloud.pushinternal.DebugLogger;
import com.meizu.cloud.pushsdk.constants.PushConstants;
import com.meizu.cloud.pushsdk.handler.MessageV3;
import com.meizu.cloud.pushsdk.networking.AndroidNetworking;
import com.meizu.cloud.pushsdk.networking.common.ANRequest;
import com.meizu.cloud.pushsdk.networking.common.ANResponse;
import com.meizu.cloud.pushsdk.notification.model.AdvanceSetting;
import com.meizu.cloud.pushsdk.util.MinSdkChecker;
import com.meizu.cloud.pushsdk.util.MzSystemUtils;
import com.meizu.cloud.pushsdk.util.PushPreferencesUtils;

/**
 * 支持Notification自定义
 */
public abstract class AbstractPushNotification implements PushNotification {
    protected static final String TAG = "AbstractPushNotification";
    protected Context context;
    protected PushNotificationBuilder pushNotificationBuilder;
    private NotificationManager notificationManager;
    protected Handler handler;
    protected AbstractPushNotification(Context context, PushNotificationBuilder pushNotificationBuilder) {
        this.pushNotificationBuilder = pushNotificationBuilder;
        this.context = context;
        handler = new Handler(context.getMainLooper());
        this.notificationManager = (NotificationManager)context.getSystemService(Context.NOTIFICATION_SERVICE);
    }

    /**
     * builder for expandable content such text and picture
     * this content can build with NotificationCompat builder
     * @param builder
     *             the notification builder
     * @param messageV3
     *              message which construct notification
     * */
    protected void buildExpandableContent(Notification.Builder builder, MessageV3 messageV3){}

    /**
     * for build notification contentview
     * @param notification
     * @param messageV3
     * */
    protected void buildContentView(Notification notification,MessageV3 messageV3){}

    /**
     * for build notification bigContentView
     * @param messageV3
     * @param notification
     * */
    protected void buildBigContentView(Notification notification,MessageV3 messageV3){}


    /**
     * notification builder set app icon
     * @param builder
     * @param messageV3
     * */
    protected void appIconSettingBuilder(Notification.Builder builder, MessageV3 messageV3){}

    /**
     * construct Notification
     * notification builder
     *                build notification expandable Content
     * notification contentView
     *                build notification contentView and bigContentView
     * */
    protected Notification construtNotificationFinal(MessageV3 messageV3,PendingIntent clickIntent, PendingIntent deleteIntent){
        Notification.Builder builder = new Notification.Builder(context);
        basicSettingBuilder(builder, messageV3, clickIntent, deleteIntent);
        advanceSettingBuilder(builder, messageV3);

        appIconSettingBuilder(builder, messageV3);
        buildExpandableContent(builder, messageV3);
        Notification notification;
        if (MinSdkChecker.isSupportNotificationBuild()) {
            notification = builder.build();
        } else {
            notification = builder.getNotification();
        }
        buildContentView(notification, messageV3);
        buildBigContentView(notification, messageV3);
        return notification;
    }

    /**
     * create click intent for click callback
     * @param messageV3
     * */
    protected PendingIntent createClickIntent(MessageV3 messageV3){
        Intent serviceClickIntent = new Intent();
        serviceClickIntent.setData(Uri.parse("custom://" + System.currentTimeMillis()));
        serviceClickIntent.putExtra(PushConstants.MZ_PUSH_PRIVATE_MESSAGE, messageV3);
        serviceClickIntent.putExtra(PushConstants.MZ_PUSH_MESSAGE_METHOD, PushConstants.MZ_PUSH_MESSAGE_METHOD_ACTION_PRIVATE);
        String messageReceiver = MzSystemUtils.findReceiver(context, PushConstants.MZ_PUSH_ON_MESSAGE_ACTION, context.getPackageName());
        serviceClickIntent.setClassName(context.getPackageName(), messageReceiver);
        serviceClickIntent.setAction(PushConstants.MZ_PUSH_ON_MESSAGE_ACTION);
        //不同弹出的通知栏Extra数据不能覆盖
        PendingIntent clickPendingIntent = PendingIntent.getBroadcast(context, 0, serviceClickIntent, PendingIntent.FLAG_ONE_SHOT);
        return clickPendingIntent;
    }

    protected PendingIntent createDeleteIntent(MessageV3 messageV3){
        Intent serviceDeleteIntent = new Intent();
        //区分intent传递不同的notification,防止由于数据相同导致intent覆盖
        serviceDeleteIntent.setData(Uri.parse("custom://" + System.currentTimeMillis()));
        serviceDeleteIntent.putExtra(PushConstants.MZ_PUSH_PRIVATE_MESSAGE, messageV3);
        serviceDeleteIntent.putExtra(PushConstants.MZ_PUSH_MESSAGE_METHOD, PushConstants.MZ_PUSH_MESSAGE_METHOD_ACTION_NOTIFICATION_DELETE);
        String messageReceiver = MzSystemUtils.findReceiver(context, PushConstants.MZ_PUSH_ON_MESSAGE_ACTION, context.getPackageName());
        serviceDeleteIntent.setClassName(context.getPackageName(), messageReceiver);
        serviceDeleteIntent.setAction(PushConstants.MZ_PUSH_ON_MESSAGE_ACTION);
        PendingIntent deletePendingIntent = PendingIntent.getBroadcast(context, 0, serviceDeleteIntent, PendingIntent.FLAG_ONE_SHOT);
        return deletePendingIntent;
    }

    /**
     * basic setting include
     * title: the notification title
     * content: the notification main content
     * autoCancel: notification disappear when click it
     * smallIcon: the icon show in the status bar,the default icon is the application icon
     * clickIntent: action when click notification
     * deleteIntent: action when delete notification
     * */
    protected void basicSettingBuilder(Notification.Builder builder, MessageV3 messageV3, PendingIntent clickIntent, PendingIntent deleteIntent){
        builder.setContentTitle(messageV3.getTitle());
        builder.setContentText(messageV3.getContent());
        builder.setTicker(messageV3.getContent());
        builder.setAutoCancel(true);
        //set visible when screen lock
        if(MinSdkChecker.isSupportSendNotification()){
            builder.setVisibility(Notification.VISIBILITY_PUBLIC);
        }
        builder.setSmallIcon((pushNotificationBuilder != null && pushNotificationBuilder.getmStatusbarIcon() != 0) ?
                pushNotificationBuilder.getmStatusbarIcon() : context.getApplicationInfo().icon);
        builder.setContentIntent(clickIntent);
        builder.setDeleteIntent(deleteIntent);
    }

    /**
     * default: light,vibrate,sound and all
     * onGoing: cannot delete notification if you do not click the notification
     * HeadUpNotification: show global notification
     * */
    protected void advanceSettingBuilder(Notification.Builder builder, MessageV3 messageV3){
        AdvanceSetting advanceSetting = messageV3.getmAdvanceSetting();
        if(advanceSetting != null){
            if(advanceSetting.getNotifyType() != null){
                boolean vibrate = advanceSetting.getNotifyType().isVibrate();
                boolean lights = advanceSetting.getNotifyType().isLights();
                boolean sound = advanceSetting.getNotifyType().isSound();
                if(vibrate || lights || sound){
                    int defaultType = 0;
                    if(vibrate){
                        defaultType |= Notification.DEFAULT_VIBRATE;
                    }
                    if(lights){
                        defaultType |= Notification.DEFAULT_LIGHTS;
                        //defaultType |= Notification.FLAG_SHOW_LIGHTS;
                        //builder.setLights(0xff00ff00,300,1000);
                    }
                    if(sound){
                        defaultType |= Notification.DEFAULT_SOUND;
                    }
                    DebugLogger.e(TAG,"current notification type is "+defaultType);
                    builder.setDefaults(defaultType);
                }

            }
            builder.setOngoing(!advanceSetting.isClearNotification());
            if(advanceSetting.isHeadUpNotification()){
                if(MinSdkChecker.isSupportNotificationBuild()){
                    builder.setPriority(Notification.PRIORITY_MAX);
                }
                //setFullScreenIntent when screen off it will auto click
                //builder.setFullScreenIntent(clickIntent, true);
                //boolean isScreenOn = isScreenOnAndUnlock();
                //DebugLogger.e(TAG,"current screen is "+isScreenOn);
                //builder.setOngoing(!isScreenOn && !advanceSetting.isClearNotification());
            }
        }
    }

    public Bitmap getBitmapFromURL(String src) {
        ANRequest request = AndroidNetworking.get(src).build();
        ANResponse<Bitmap> response = request.executeForBitmap();
        if(response.isSuccess() && response.getResult() != null){
            DebugLogger.i(TAG,"ANRequest On other Thread down load largeIcon "+src+ "image "+(response.getResult() != null? "success":"fail"));
            return response.getResult();
        } else {
            DebugLogger.i(TAG, "ANRequest On other Thread down load largeIcon " + src + "image fail");
            return null;
        }
    }

    /**
     * 获取当前应用图标
     * */
    public Bitmap getAppIcon(Context context,String packageName) {
        Bitmap appIcon;
        try {
            PackageManager packageManager = context.getPackageManager();
            BitmapDrawable bitmapDrawable = (BitmapDrawable) packageManager.getApplicationIcon(packageName);
            appIcon = bitmapDrawable.getBitmap();
        } catch (PackageManager.NameNotFoundException e) {
            DebugLogger.i(TAG, "getappicon error " + e.getMessage());
            BitmapDrawable bitmapDrawable = (BitmapDrawable)context.getApplicationInfo().loadIcon(context.getPackageManager());
            appIcon = bitmapDrawable.getBitmap();
        }
        return appIcon;
    }

    /**
     * check current thread
     * */
    protected boolean isOnMainThread(){
        return Thread.currentThread() == context.getMainLooper().getThread();
    }

    /**
     * check screen status and screen lock status
     * */
    protected boolean isScreenOnAndUnlock(){
        PowerManager powerManager = (PowerManager) context.getSystemService(Context.POWER_SERVICE);
        if(Build.VERSION.SDK_INT < 20 ? powerManager.isScreenOn():powerManager.isInteractive()){
            KeyguardManager keyguardManager = (KeyguardManager) context.getSystemService(Context.KEYGUARD_SERVICE);
            if(!keyguardManager.inKeyguardRestrictedInputMode()){
                return true;
            }
        }
        return false;
    }


    @Override
    public void show(MessageV3 messageV3) {
        Notification notification = construtNotificationFinal(messageV3, createClickIntent(messageV3), createDeleteIntent(messageV3));
        int notifyId = (int) System.currentTimeMillis();
        if(messageV3.isDiscard()){
            if(PushPreferencesUtils.getDiscardNotificationId(context, messageV3.getPackageName()) == 0){
                PushPreferencesUtils.putDiscardNotificationIdByPackageName(context,messageV3.getPackageName(),notifyId);
                DebugLogger.i(TAG,"no notification show so put notification id "+notifyId);
            }
            if(!TextUtils.isEmpty(messageV3.getTaskId())){
                if(PushPreferencesUtils.getDiscardNotificationTaskId(context,messageV3.getPackageName()) == 0){
                    PushPreferencesUtils.putDiscardNotificationTaskId(context,messageV3.getPackageName(),Integer.valueOf(messageV3.getTaskId()));
                } else {
                    if(Integer.valueOf(messageV3.getTaskId()) < PushPreferencesUtils.getDiscardNotificationTaskId(context,messageV3.getPackageName())){
                        DebugLogger.i(TAG, "current package " + messageV3.getPackageName() + " taskid " + messageV3.getTaskId() + " dont show notification");
                        return;
                    } else {
                        PushPreferencesUtils.putDiscardNotificationTaskId(context,messageV3.getPackageName(),Integer.valueOf(messageV3.getTaskId()));
                        notifyId = PushPreferencesUtils.getDiscardNotificationId(context,messageV3.getPackageName());
                    }
                }
            }
            DebugLogger.i(TAG, "current package " + messageV3.getPackageName() + " notificationId=" + notifyId + " taskId=" + messageV3.getTaskId());
        }
        notificationManager.notify(notifyId, notification);
        dismissFloatNotification(notificationManager,notifyId,messageV3);
    }

    /**
     * show head_notification and setOngoing dismiss float notification and show normal notification
     * */
    protected void dismissFloatNotification(final NotificationManager notificationManager, final int notificationId,MessageV3 messageV3){
        AdvanceSetting advanceSetting = messageV3.getmAdvanceSetting();
        if(advanceSetting != null){
            boolean isFloatNotification = advanceSetting.isHeadUpNotification();
            boolean isClearNotification = advanceSetting.isClearNotification();
            if(isFloatNotification && !isClearNotification){
                messageV3.getmAdvanceSetting().setHeadUpNotification(false);
                messageV3.getmAdvanceSetting().getNotifyType().setSound(false);
                messageV3.getmAdvanceSetting().getNotifyType().setVibrate(false);
                final Notification notification = construtNotificationFinal(messageV3,createClickIntent(messageV3),createDeleteIntent(messageV3));
                //refresh notification for clear head up notification
                handler.postDelayed(new Runnable() {
                    @Override
                    public void run() {
                        notificationManager.notify(notificationId,notification);
                    }
                },5000);
            }
        }
    }
}
