package com.meizu.cloud.pushsdk.notification.util;

import android.os.SystemClock;


import com.meizu.cloud.pushinternal.DebugLogger;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.Enumeration;
import java.util.zip.ZipEntry;
import java.util.zip.ZipException;
import java.util.zip.ZipFile;

/**
 * Extract the zip file asynchronously.
 */
public class ZipExtractTask {
    private static final String LOG_TAG = "ZipExtractTask";
    private final File mInput;
    private final File mOutput;
    private String mEextractedDir;

    /**
     * This method can be use inner ActiveView, it can update ActiveView after extract finished.
     *
     * @param inFile The zip file to be extracted with absolute path. ex: /storage/emulated/0/Android/data/com.example.meizu.bannerviewtest/files/banner/output.zip
     * @param outputDir The output directory with absolute path. ex: /storage/emulated/0/Android/data/com.example.meizu.bannerviewtest/files/banner
     */
    public ZipExtractTask(String inFile, String outputDir) {
        mInput = new File(inFile);
        mOutput = new File(outputDir);
        mEextractedDir =  mOutput.getAbsolutePath();
        DebugLogger.i(LOG_TAG, "Extract mInput file = " + mInput.toString());
        DebugLogger.i(LOG_TAG, "Extract mOutput file = " + mOutput.toString());
    }

    private void deleteZipFile() {
        // delete *.zip file after extracted.
        if (mInput != null && mInput.exists()) {
            if (mInput.delete()) {
                DebugLogger.i(LOG_TAG, "Delete file:" + mInput.toString() + " after extracted.");
            } else {
                DebugLogger.i(LOG_TAG, "Can't delete file:" + mInput.toString() + " after extracted.");
            }
        }
    }

    public boolean doUnzipSync() {
        long extractedSize = unzip();
        return extractedSize > 0 ? true : false;
    }


    /**
     * This method unzip an decrypt zip file with java.util.zip.
     */
    private long unzip() {
        long start = SystemClock.currentThreadTimeMillis();

        long extractedSize = 0L;
        boolean deleteTempDir = false;
        Enumeration<ZipEntry> entries;
        ZipFile zip = null;
        String zipDirName = null;
        try {
            zip = new ZipFile(mInput);
            entries = (Enumeration<ZipEntry>) zip.entries();
            while (entries.hasMoreElements()) {
                ZipEntry entry = entries.nextElement();
                if (entry.isDirectory()) {
                    continue;
                }

                String fileName = entry.getName();
                if (zipDirName == null && fileName != null) {
                    zipDirName = fileName.split("/")[0];
                    DebugLogger.i(LOG_TAG, "Extract temp directory=" + mOutput + "/" + zipDirName);
                }
                File destination = new File(mOutput, fileName);
                if (!destination.getParentFile().exists()) {
                    if (destination.getParentFile().mkdirs()) {
                        DebugLogger.i(LOG_TAG, "Make Destination directory=" + destination.getParentFile().getAbsolutePath());
                    } else {
                        DebugLogger.i(LOG_TAG, "Can't make destination directory=" + destination.getParentFile().getAbsolutePath());
                    }
                }

                FileOutputStream outStream = new FileOutputStream(destination);
                extractedSize += copy(zip.getInputStream(entry), outStream);
                outStream.close();
            }

            String tempDir = mOutput + "/" + zipDirName;
            if (!mEextractedDir.equals(tempDir)) {
                FileUtil.copyFolder(tempDir, mEextractedDir);
                deleteTempDir = true;
            }

        } catch (ZipException e) {
            DebugLogger.e(LOG_TAG, "ZipException :" + e.toString());

        } catch (IOException e) {
            DebugLogger.e(LOG_TAG, "Extracted IOException:" + e.toString());
        } finally {
            try {
                if (zip != null) {
                    zip.close();
                }
            } catch (IOException e) {
                DebugLogger.e(LOG_TAG, "Extracted IOException:" + e.toString());
            }
        }

        long end = SystemClock.currentThreadTimeMillis();

        DebugLogger.i(LOG_TAG, "Extract file " + mInput + ", UseTime =" + String.valueOf(end - start));

        if (deleteTempDir) {
            FileUtil.deleteDirectory(mOutput + "/" + zipDirName);
        }

        deleteZipFile();
        return extractedSize;
    }

    private int copy(InputStream input, OutputStream output){
        byte[] buffer = new byte[1024 * 8];
        BufferedInputStream in = new BufferedInputStream(input, 1024 * 8);
        BufferedOutputStream out  = new BufferedOutputStream(output, 1024 * 8);
        int count =0;
        int n;
        try {
            while ((n = in.read(buffer, 0, 1024 * 8)) != -1) {
                out.write(buffer, 0, n);
                count += n;
            }
            out.flush();
        } catch (IOException e) {
            DebugLogger.e(LOG_TAG, "Extracted IOException:" + e.toString());
        } finally {
            try {
                out.close();
            } catch (IOException e) {
                DebugLogger.e(LOG_TAG, "out.close() IOException e=" + e.toString());
            }
            try {
                in.close();
            } catch (IOException e) {
                DebugLogger.e(LOG_TAG, "in.close() IOException e=" + e.toString());
            }
        }
        return count;
    }
}
