package com.meizu.cloud.pushsdk.pushtracer.emitter;

import android.content.Context;
import android.net.Uri;

import com.meizu.cloud.pushsdk.networking.http.HttpURLConnectionCall;
import com.meizu.cloud.pushsdk.networking.http.MediaType;
import com.meizu.cloud.pushsdk.networking.http.Request;
import com.meizu.cloud.pushsdk.networking.http.RequestBody;
import com.meizu.cloud.pushsdk.pushtracer.constant.Parameters;
import com.meizu.cloud.pushsdk.pushtracer.constant.TrackerConstants;
import com.meizu.cloud.pushsdk.pushtracer.dataload.DataLoad;
import com.meizu.cloud.pushsdk.pushtracer.dataload.SelfDescribingJson;
import com.meizu.cloud.pushsdk.pushtracer.storage.EventStore;
import com.meizu.cloud.pushsdk.pushtracer.utils.Logger;
import com.meizu.cloud.pushsdk.pushtracer.utils.Util;
import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.Map;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import javax.net.ssl.HostnameVerifier;
import javax.net.ssl.SSLSocketFactory;



/**
 * Emitter for build requests
 */
public abstract class Emitter {
    protected int POST_WRAPPER_BYTES = 88;
    protected int POST_STM_BYTES = 22;

    private final String TAG = Emitter.class.getSimpleName();
    protected final MediaType JSON = MediaType.parse(TrackerConstants.POST_CONTENT_TYPE);
    protected Context context;
    protected Uri.Builder uriBuilder;
    protected RequestCallback requestCallback;
    protected HttpMethod httpMethod;
    protected BufferOption bufferOption;
    protected RequestSecurity requestSecurity;
    protected SSLSocketFactory sslSocketFactory;
    protected HostnameVerifier hostnameVerifier;
    protected String uri;
    protected int emitterTick;
    protected int emptyLimit;
    protected int sendLimit;
    protected long byteLimitGet;
    protected long byteLimitPost;
    protected TimeUnit timeUnit;

    protected AtomicBoolean isRunning = new AtomicBoolean(false);

    /**
     * Builder for the Emitter.
     */
    @SuppressWarnings("unchecked")
    public static class EmitterBuilder {

        protected static Class<? extends Emitter> defaultEmitterClass;


        private Class<? extends Emitter> emitterClass;
        protected final String uri; // Required
        protected final Context context; // Required
        protected RequestCallback requestCallback = null; // Optional
        protected HttpMethod httpMethod = HttpMethod.POST; // Optional
        protected BufferOption bufferOption = BufferOption.Single; // Optional Single emits
        protected RequestSecurity requestSecurity = RequestSecurity.HTTPS; // Optional
        protected int emitterTick = 5; // Optional
        protected int sendLimit = 250; // Optional
        protected int emptyLimit = 5; // Optional
        protected long byteLimitGet = 40000; // Optional
        protected long byteLimitPost = 40000; // Optional
        protected TimeUnit timeUnit = TimeUnit.SECONDS;
        protected SSLSocketFactory sslSocketFactory; // Optional
        protected HostnameVerifier hostnameVerifier; // Optional

        /**
         * @param uri The uri of the collector
         * @param context the android context
         */
        public EmitterBuilder(String uri, Context context) {
            this(uri, context, defaultEmitterClass);
        }

        /**
         *
         * @param uri The collector uri to send events to
         * @param context The android context
         * @param emitterClass The emitter class to use
         */
        public EmitterBuilder(String uri, Context context, Class<? extends Emitter> emitterClass) {
            this.uri = uri;
            this.context = context;
            this.emitterClass = emitterClass;
        }

        /**
         * @param httpMethod The method by which requests are emitted
         * @return itself
         */
        public EmitterBuilder method(HttpMethod httpMethod) {
            this.httpMethod = httpMethod;
            return this;
        }

        /**
         * @param option the buffer option for the emitter
         * @return itself
         */
        public EmitterBuilder option(BufferOption option) {
            this.bufferOption = option;
            return this;
        }

        /**
         * @param requestSecurity the security chosen for requests
         * @return itself
         */
        public EmitterBuilder security(RequestSecurity requestSecurity) {
            this.requestSecurity = requestSecurity;
            return this;
        }

        /**
         * @param sslSocketFactory the sslSocketFactory chosen for requests
         * @return itself
         * */
        public EmitterBuilder sslSocketFactory(SSLSocketFactory sslSocketFactory){
            this.sslSocketFactory = sslSocketFactory;
            return this;
        }

        /**
         * @param hostnameVerifier the hostnameVerifier chosen for requests
         * @return itself
         * */
        public EmitterBuilder hostnameVerifier(HostnameVerifier hostnameVerifier){
            this.hostnameVerifier = hostnameVerifier;
            return this;
        }

        /**
         * @param requestCallback Request callback function
         * @return itself
         */
        public EmitterBuilder callback(RequestCallback requestCallback) {
            this.requestCallback = requestCallback;
            return this;
        }

        /**
         * @param emitterTick The tick count between emitter attempts
         * @return itself
         */
        public EmitterBuilder tick(int emitterTick) {
            this.emitterTick = emitterTick;
            return this;
        }

        /**
         * @param sendLimit The maximum amount of events to grab for an emit attempt
         * @return itself
         */
        public EmitterBuilder sendLimit(int sendLimit) {
            this.sendLimit = sendLimit;
            return this;
        }

        /**
         * @param emptyLimit The amount of emitter ticks that are performed before we shut down
         *                   due to the database being empty.
         * @return itself
         */
        public EmitterBuilder emptyLimit(int emptyLimit) {
            this.emptyLimit = emptyLimit;
            return this;
        }

        /**
         * @param byteLimitGet The maximum amount of bytes allowed to be sent in a payload
         *                     in a GET request.
         * @return itself
         */
        public EmitterBuilder byteLimitGet(long byteLimitGet) {
            this.byteLimitGet = byteLimitGet;
            return this;
        }

        /**
         * @param byteLimitPost The maximum amount of bytes allowed to be sent in a payload
         *                      in a POST request.
         * @return itself
         */
        public EmitterBuilder byteLimitPost(long byteLimitPost) {
            this.byteLimitPost = byteLimitPost;
            return this;
        }

        /**
         * @param timeUnit a valid TimeUnit
         * @return itself
         */
        public EmitterBuilder timeUnit(TimeUnit timeUnit) {
            this.timeUnit = timeUnit;
            return this;
        }


    }

    /**
     * Creates an emitter object
     *
     * @param builder The builder that constructs an emitter
     */
    public Emitter(EmitterBuilder builder) {
        this.httpMethod = builder.httpMethod;
        this.requestCallback = builder.requestCallback;
        this.context = builder.context;
        this.bufferOption = builder.bufferOption;
        this.requestSecurity = builder.requestSecurity;
        this.sslSocketFactory = builder.sslSocketFactory;
        this.hostnameVerifier = builder.hostnameVerifier;
        this.emitterTick = builder.emitterTick;
        this.emptyLimit = builder.emptyLimit;
        this.sendLimit = builder.sendLimit;
        this.byteLimitGet = builder.byteLimitGet;
        this.byteLimitPost = builder.byteLimitPost;
        this.uri = builder.uri;
        this.timeUnit = builder.timeUnit;
        buildEmitterUri();
        //buildHttpsSecurity();

        Logger.i(TAG, "Emitter created successfully!");
    }

    /**
     * Sets the Emitter URI
     */
    private void buildEmitterUri() {
        Logger.e(TAG,"security "+requestSecurity);
        if (this.requestSecurity == RequestSecurity.HTTP) {
            this.uriBuilder = Uri.parse("http://" + this.uri).buildUpon();
        } else {
            this.uriBuilder = Uri.parse("https://" + this.uri).buildUpon();
        }
        if (this.httpMethod == HttpMethod.GET) {
            uriBuilder.appendPath("i");
        }
        else {
            uriBuilder.appendEncodedPath("push_data_report/mobile");
        }
    }

    private void buildHttpsSecurity(){
        if(requestSecurity == RequestSecurity.HTTPS){
            if(sslSocketFactory == null){
                Logger.e(TAG,"Https Ensure you have set SSLSocketFactory");
            } else {
                //warning set sslSocketFactory
                //client = client.newBuilder().sslSocketFactory(sslSocketFactory).build();
            }
            if(hostnameVerifier == null){
                Logger.e(TAG,"Https Ensure you have set HostnameVerifier");
            } else {
                //client = client.newBuilder().hostnameVerifier(hostnameVerifier).build();
            }
        }
    }

    /**
     * @param payload the payload to be added to
     *                the EventStore
     */
    public abstract void add(DataLoad payload);

    /**
     * Shuts the emitter down!
     */
    public abstract void shutdown();

    /**
     * Sends everything in the database to the endpoint.
     */
    public abstract void flush();

    /**
     * Performs a synchronous sending of a list of
     * ReadyRequests.
     *
     * @param requests the requests to send
     * @return the request results.
     */
    protected LinkedList<RequestResult> performSyncEmit(LinkedList<ReadyRequest> requests) {
        LinkedList<RequestResult> results = new LinkedList<>();
        for (ReadyRequest request : requests) {
            int code = requestSender(request.getRequest());
            if (request.isOversize()) {
                results.add(new RequestResult(true, request.getEventIds()));
            } else {
                results.add(new RequestResult(isSuccessfulSend(code), request.getEventIds()));
            }
        }
        return results;
    }

    /**
     * The function responsible for actually sending
     * the request to the collector.
     *
     * @param request The request to be sent
     * @return a RequestResult
     */
    protected int requestSender(Request request) {
        try {
            Logger.d(TAG, "Sending request: %s", request);
            return new HttpURLConnectionCall(request).execute().code();
        } catch (IOException e) {
            Logger.e(TAG, "Request sending failed: %s", e.toString());
            return -1;
        }
    }

    /**
     * Returns a list of ReadyRequests which can
     * all be sent regardless of if it is GET or POST.
     * - Checks if the event is over-sized.
     * - Stores all of the relevant event ids.
     *
     * @param events a list of EmittableEvents pulled
     *               from the database.
     * @return a list of ready to send requests
     */
    protected LinkedList<ReadyRequest> buildRequests(EmittableEvents events) {

        int dataLoadCount = events.getEvents().size();
        LinkedList<Long> eventIds = events.getEventIds();
        LinkedList<ReadyRequest> requests = new LinkedList<>();

        if (httpMethod == HttpMethod.GET) {
            for (int i = 0; i < dataLoadCount; i++) {

                // Get the eventId for this request
                LinkedList<Long> reqEventId = new LinkedList<>();
                reqEventId.add(eventIds.get(i));

                // Build and store the request
                DataLoad dataLoad = events.getEvents().get(i);
                boolean oversize = dataLoad.getByteSize() + POST_STM_BYTES > byteLimitGet;
                Request request = requestBuilderGet(dataLoad);
                requests.add(new ReadyRequest(oversize, request, reqEventId));
            }
        } else {
            for (int i = 0; i < dataLoadCount; i += bufferOption.getCode()) {

                LinkedList<Long> reqEventIds = new LinkedList<>();
                ArrayList<DataLoad> postDataLoadMaps = new ArrayList<>();
                long totalByteSize = 0;

                for (int j = i; j < (i + bufferOption.getCode()) && j < dataLoadCount; j++) {
                    DataLoad dataLoad = events.getEvents().get(j);
                    long payloadByteSize = dataLoad.getByteSize() + POST_STM_BYTES;

                    if ((payloadByteSize + POST_WRAPPER_BYTES) > byteLimitPost) {
                        ArrayList<DataLoad> singlePayloadMap = new ArrayList<>();
                        LinkedList<Long> reqEventId = new LinkedList<>();

                        // Build and store the request
                        singlePayloadMap.add(dataLoad);
                        reqEventId.add(eventIds.get(j));
                        Request request = requestBuilderPost(singlePayloadMap);
                        requests.add(new ReadyRequest(true, request, reqEventId));
                    }
                    else if ((totalByteSize + payloadByteSize + POST_WRAPPER_BYTES +
                            (postDataLoadMaps.size() -1)) > byteLimitPost) {
                        Request request = requestBuilderPost(postDataLoadMaps);
                        requests.add(new ReadyRequest(false, request, reqEventIds));

                        // Clear collections and build a new POST
                        postDataLoadMaps = new ArrayList<>();
                        reqEventIds = new LinkedList<>();

                        // Build and store the request
                        postDataLoadMaps.add(dataLoad);
                        reqEventIds.add(eventIds.get(j));
                        totalByteSize = payloadByteSize;
                    }
                    else {
                        totalByteSize += payloadByteSize;
                        postDataLoadMaps.add(dataLoad);
                        reqEventIds.add(eventIds.get(j));
                    }
                }

                // Check if all payloads have been processed
                if (!postDataLoadMaps.isEmpty()) {
                    Request request = requestBuilderPost(postDataLoadMaps);
                    requests.add(new ReadyRequest(false, request, reqEventIds));
                }
            }
        }
        return requests;
    }

    // Request Builders

    /**
     * Builds an OkHttp GET request which is ready
     * to be executed.
     * @param dataLoad The payload to be sent in the
     *                request.
     * @return an OkHttp request object
     */
    @SuppressWarnings("unchecked")
    private Request requestBuilderGet(DataLoad dataLoad) {
        addStmToEvent(dataLoad, "");

        // Clear the previous query
        uriBuilder.clearQuery();

        // Build the request query
        HashMap hashMap = (HashMap) dataLoad.getMap();

        for (String key : (Iterable<String>) hashMap.keySet()) {
            String value = (String) hashMap.get(key);
            uriBuilder.appendQueryParameter(key, value);
        }

        // Build the request
        String reqUrl = uriBuilder.build().toString();
        return new Request.Builder()
                .url(reqUrl)
                .get()
                .build();
    }

    /**
     * Builds an OkHttp POST request which is ready
     * to be executed.
     * @param dataLoads The payloads to be sent in the
     *                 request.
     * @return an OkHttp request object
     */
    private Request requestBuilderPost(ArrayList<DataLoad> dataLoads) {
        ArrayList<Map> finalPayloads = new ArrayList<>();
        StringBuffer dataloadbuffer = new StringBuffer();
        for (DataLoad payload : dataLoads) {
            dataloadbuffer.append(payload.toString());
            //addStmToEvent(payload, stm);
            finalPayloads.add(payload.getMap());
        }
        //SelfDescribingJson postPayload = new SelfDescribingJson("push_group_data", finalPayloads);
        //Logger.d(TAG,"SelfDescribingJson "+postPayload);
        String reqUrl = uriBuilder.build().toString();
        String postStr = dataloadbuffer.toString();
        Logger.d(TAG,"post final String "+ postStr);
        RequestBody reqBody = RequestBody.create(JSON, postStr);
        return new Request.Builder()
                .url(reqUrl)
                .post(reqBody)
                .build();
    }

    /**
     * Adds the Sending Time (stm) field
     * to each event payload.
     *
     * @param dataLoad The payload to append the field to
     * @param timestamp An optional timestamp String
     */
    private void addStmToEvent(DataLoad dataLoad, String timestamp) {
        dataLoad.add(Parameters.SENT_TIMESTAMP,
                timestamp.equals("") ? Util.getTimestamp() : timestamp);
    }

    // Setters, Getters and Checkers

    /**
     * @return the emitter event store
     */
    public abstract EventStore getEventStore();

    /**
     * @return the emitter status
     */
    public abstract boolean getEmitterStatus();

    /**
     * Returns truth on if the request
     * was sent successfully.
     *
     * @param code the response code
     * @return the truth as to the success
     */
    protected boolean isSuccessfulSend(int code) {
        return code >= 200 && code < 300;
    }

    /**
     * Sets whether the buffer should send events instantly or after the buffer has reached
     * it's limit. By default, this is set to BufferOption Default.
     *
     * @param option Set the BufferOption enum to Instant send events upon creation.
     */
    public void setBufferOption(BufferOption option) {
        if (!isRunning.get()) {
            this.bufferOption = option;
        }
    }

    /**
     * Sets the HttpMethod for the Emitter
     *
     * @param method the HttpMethod
     */
    public void setHttpMethod(HttpMethod method) {
        if (!isRunning.get()) {
            this.httpMethod = method;
            buildEmitterUri();
        }
    }

    /**
     * Sets the RequestSecurity for the Emitter
     *
     * @param security the RequestSecurity
     */
    public void setRequestSecurity(RequestSecurity security) {
        if (!isRunning.get()) {
            this.requestSecurity = security;
            buildEmitterUri();
        }
    }

    /**
     * Updates the URI for the Emitter
     *
     * @param uri new Emitter URI
     */
    public void setEmitterUri(String uri) {
        if (!isRunning.get()) {
            this.uri = uri;
            buildEmitterUri();
        }
    }

    /**
     * @return the emitter uri
     */
    public String getEmitterUri() {
        return this.uriBuilder.clearQuery().build().toString();
    }

    /**
     * @return the request callback method
     */
    public RequestCallback getRequestCallback() {
        return this.requestCallback;
    }

    /**
     * @return the Emitters request method
     */
    public HttpMethod getHttpMethod() {
        return this.httpMethod;
    }

    /**
     * @return the buffer option selected for the emitter
     */
    public BufferOption getBufferOption() {
        return this.bufferOption;
    }

    /**
     * @return the request security selected for the emitter
     */
    public RequestSecurity getRequestSecurity() {
        return this.requestSecurity;
    }

    /**
     * @return the emitter tick
     */
    public int getEmitterTick() {
        return this.emitterTick;
    }

    /**
     * @return the amount of times the event store can be empty
     *         before it is shutdown.
     */
    public int getEmptyLimit() {
        return this.emptyLimit;
    }

    /**
     * @return the emitter send limit
     */
    public int getSendLimit() {
        return this.sendLimit;
    }

    /**
     * @return the GET byte limit
     */
    public long getByteLimitGet() {
        return this.byteLimitGet;
    }

    /**
     * @return the POST byte limit
     */
    public long getByteLimitPost() {
        return this.byteLimitPost;
    }

}
