package com.meizu.cloud.pushsdk.pushtracer.storage;

import android.content.ContentValues;
import android.content.Context;
import android.database.Cursor;
import android.database.DatabaseUtils;
import android.database.sqlite.SQLiteDatabase;


import com.meizu.cloud.pushsdk.pushtracer.dataload.DataLoad;
import com.meizu.cloud.pushsdk.pushtracer.dataload.TrackerDataload;
import com.meizu.cloud.pushsdk.pushtracer.emitter.EmittableEvents;
import com.meizu.cloud.pushsdk.pushtracer.utils.Logger;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

/**
 * Created by liaojinlong on 16-4-14.
 */
public class EventStore {
    private String TAG = EventStore.class.getSimpleName();

    private SQLiteDatabase database;
    private EventStoreHelper dbHelper;
    private String[] allColumns = {
            EventStoreHelper.COLUMN_ID,
            EventStoreHelper.COLUMN_EVENT_DATA,
            EventStoreHelper.COLUMN_DATE_CREATED
    };
    private long lastInsertedRowId = -1;

    private int sendLimit;

    /**
     * Creates a new Event Store
     *
     * @param context The android context object
     * @param sendLimit The maximum amount of events that can be sent
     *                  concurrently
     */
    public EventStore(Context context, int sendLimit) {
        dbHelper = EventStoreHelper.getInstance(context);
        open();
        this.sendLimit = sendLimit;

        Logger.d(TAG, "DB Path: " + database.getPath());
    }

    /**
     * Adds an event to the database.
     *
     * @param dataLoad the payload to be added
     */
    public void add(DataLoad dataLoad) {
        insertEvent(dataLoad);
    }

    /**
     * Opens a new writable database if it
     * is currently closed.
     */
    public void open() {
        if (!isDatabaseOpen()) {
            database = dbHelper.getWritableDatabase();
            database.enableWriteAheadLogging();
        }
    }

    /**
     * Closes the database
     */
    public void close() {
        dbHelper.close();
    }

    /**
     * Inserts a payload into the database
     *
     * @param dataLoad The event dataLoad to
     *                be stored
     * @return a boolean stating if the insert
     * was a success or not
     */
    @SuppressWarnings("unchecked")
    public long insertEvent(DataLoad dataLoad) {
        if (isDatabaseOpen()) {
            byte[] bytes = EventStore.serialize(dataLoad.getMap());
            ContentValues values = new ContentValues(2);
            values.put(EventStoreHelper.COLUMN_EVENT_DATA, bytes);
            lastInsertedRowId = database.insert(EventStoreHelper.TABLE_EVENTS, null, values);
        }
        Logger.d(TAG, "Added event to database: "+lastInsertedRowId);
        return lastInsertedRowId;
    }

    /**
     * Removes an event from the database
     *
     * @param id the row id of the event
     * @return a boolean of success to remove
     */
    public boolean removeEvent(long id) {
        int retval = -1;
        if (isDatabaseOpen()) {
            retval = database.delete(EventStoreHelper.TABLE_EVENTS,
                    EventStoreHelper.COLUMN_ID + "=" + id, null);
        }
        Logger.d(TAG, "Removed event from database: "+ id);
        return retval == 1;
    }

    /**
     * Empties the database of all events
     *
     * @return a boolean of success to remove
     */
    public boolean removeAllEvents() {
        int retval = -1;
        if (isDatabaseOpen()) {
            retval = database.delete(EventStoreHelper.TABLE_EVENTS, null, null);
        }
        Logger.d(TAG, "Removing all events from database.");
        return retval == 0;
    }

    /**
     * Converts an event map to a byte
     * array for storage.
     *
     * @param map the map containing all
     *            the event parameters
     * @return the byte array or null
     */
    private static byte[] serialize(Map<String, String> map) {
        try {
            ByteArrayOutputStream mem_out = new ByteArrayOutputStream();
            ObjectOutputStream out = new ObjectOutputStream(mem_out);
            out.writeObject(map);
            out.close();
            mem_out.close();
            return mem_out.toByteArray();
        } catch (IOException e) {
            e.printStackTrace();
        }
        return null;
    }

    /**
     * Converts a byte array back into an
     * event map for sending.
     *
     * @param bytes the bytes to be converted
     * @return the Map or null
     */
    @SuppressWarnings("unchecked")
    private static Map<String, String> deserializer(byte[] bytes) {
        try {
            ByteArrayInputStream mem_in = new ByteArrayInputStream(bytes);
            ObjectInputStream in = new ObjectInputStream(mem_in);
            Map<String, String> map = (HashMap<String, String>) in.readObject();
            in.close();
            mem_in.close();
            return map;
        } catch (ClassNotFoundException | IOException e) {
            e.printStackTrace();
        }
        return null;
    }

    /**
     * Returns the events that validate a
     * specific query.
     *
     * @param query the query to be passed against
     *              the database
     * @param orderBy what to order the query by
     * @return the list of events that satisfied
     * the query
     */
    public List<Map<String, Object>> queryDatabase(String query, String orderBy) {
        List<Map<String, Object>> res = new ArrayList<>();
        if (isDatabaseOpen()) {
            Cursor cursor = database.query(EventStoreHelper.TABLE_EVENTS, allColumns, query,
                    null, null, null, orderBy);

            cursor.moveToFirst();
            while (!cursor.isAfterLast()) {
                Map<String, Object> eventMetadata = new HashMap<>();
                eventMetadata.put(EventStoreHelper.METADATA_ID, cursor.getLong(0));
                eventMetadata.put(EventStoreHelper.METADATA_EVENT_DATA,
                        EventStore.deserializer(cursor.getBlob(1)));
                eventMetadata.put(EventStoreHelper.METADATA_DATE_CREATED, cursor.getString(2));
                cursor.moveToNext();
                res.add(eventMetadata);
            }
            cursor.close();
        }
        return res;
    }

    // Getters

    /**
     * Returns amount of events currently
     * in the database.
     *
     * @return the count of events in the
     * database
     */
    public long getSize() {
        return DatabaseUtils.queryNumEntries(database, EventStoreHelper.TABLE_EVENTS);
    }

    /**
     * Returns the last rowId to be
     * inserted.
     *
     * @return the last inserted rowId
     */
    public long getLastInsertedRowId() {
        return lastInsertedRowId;
    }

    /**
     * Returns an EmittableEvents object which
     * contains events and eventIds within a
     * defined range of the database.
     *
     * @return an EmittableEvents object containing
     * eventIds and event payloads.
     */
    @SuppressWarnings("unchecked")
    public EmittableEvents getEmittableEvents() {

        LinkedList<Long> eventIds = new LinkedList<>();
        ArrayList<DataLoad> events = new ArrayList<>();

        // FIFO Pattern for sending events
        for (Map<String, Object> eventMetadata : getDescEventsInRange(this.sendLimit)) {

            // Create a TrackerPayload for each event
            TrackerDataload payload = new TrackerDataload();
            Map<String, Object> eventData = (Map<String, Object>)
                    eventMetadata.get(EventStoreHelper.METADATA_EVENT_DATA);
            payload.addMap(eventData);

            // Store the eventId
            Long eventId = (Long) eventMetadata.get(EventStoreHelper.METADATA_ID);
            eventIds.add(eventId);

            // Add the payload to the list
            events.add(payload);
        }
        return new EmittableEvents(events, eventIds);
    }

    /**
     * Returns a Map containing the event
     * payload values, the table row ID and
     * the date it was created.
     *
     * @param id the row id of the event to get
     * @return event metadata
     */
    public Map<String, Object> getEvent(long id) {
        List<Map<String, Object>> res =
                queryDatabase(EventStoreHelper.COLUMN_ID + "=" + id, null);
        if (!res.isEmpty()) {
            return res.get(0);
        }
        else {
            return null;
        }
    }

    /**
     * Returns a list of all the events in the
     * database.
     *
     * @return the events in the database
     */
    public List<Map<String, Object>> getAllEvents() {
        return queryDatabase(null, null);
    }

    /**
     * Returns a descending range of events
     * from the top of the database.
     *
     * @param range amount of rows to take
     * @return a list of event metadata
     */
    public List<Map<String, Object>> getDescEventsInRange(int range) {
        return queryDatabase(null, "id ASC LIMIT " + range);
    }

    /**
     * Returns truth on if database is open.
     *
     * @return a boolean for database status
     */
    public boolean isDatabaseOpen() {
        return database != null && database.isOpen();
    }
}
