package com.meizu.cloud.pushsdk.pushtracer.tracker;


import android.content.Context;

import com.meizu.cloud.pushinternal.BuildConfig;
import com.meizu.cloud.pushsdk.pushtracer.constant.Parameters;
import com.meizu.cloud.pushsdk.pushtracer.constant.TrackerConstants;
import com.meizu.cloud.pushsdk.pushtracer.dataload.SelfDescribingJson;
import com.meizu.cloud.pushsdk.pushtracer.dataload.TrackerDataload;
import com.meizu.cloud.pushsdk.pushtracer.emitter.Emitter;
import com.meizu.cloud.pushsdk.pushtracer.event.PushEvent;
import com.meizu.cloud.pushsdk.pushtracer.utils.LogLevel;
import com.meizu.cloud.pushsdk.pushtracer.utils.Logger;


import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;

/**
 * Builds a Tracker object which is used to
 * send events to a Snowplow Collector.
 */
public abstract class Tracker {

    private final static String TAG = Tracker.class.getSimpleName();
    protected final String trackerVersion = BuildConfig.VERSION_NAME;
    protected Emitter emitter;
    protected Subject subject;
    protected Session trackerSession;
    protected String namespace;
    protected String appId;
    protected boolean base64Encoded;
    protected LogLevel level;
    protected boolean sessionContext;
    protected long sessionCheckInterval;
    protected int threadCount;
    protected TimeUnit timeUnit;

    protected AtomicBoolean dataCollection = new AtomicBoolean(true);

    /**
     * Builder for the Tracker
     */
    @SuppressWarnings("unchecked")
    public static class TrackerBuilder {

        protected static Class<? extends Tracker> defaultTrackerClass;


        private Class<? extends Tracker> trackerClass;
        protected final Emitter emitter; // Required
        protected final String namespace; // Required
        protected final String appId; // Required
        protected final Context context; // Required
        protected Subject subject = null; // Optional
        protected boolean base64Encoded = false; // Optional
        protected LogLevel logLevel = LogLevel.OFF; // Optional
        protected boolean sessionContext = false; // Optional
        protected long foregroundTimeout = 600; // Optional - 10 minutes
        protected long backgroundTimeout = 300; // Optional - 5 minutes
        protected long sessionCheckInterval = 15; // Optional - 15 seconds
        protected int threadCount = 10; // Optional
        protected TimeUnit timeUnit = TimeUnit.SECONDS; // Optional

        /**
         * @param emitter Emitter to which events will be sent
         * @param namespace Identifier for the Tracker instance
         * @param appId Application ID
         * @param context The Android application context
         */
        public TrackerBuilder(Emitter emitter, String namespace, String appId, Context context) {
            this(emitter, namespace, appId, context, defaultTrackerClass);
        }

        /**
         * @param emitter Emitter to which events will be sent
         * @param namespace Identifier for the Tracker instance
         * @param appId Application ID
         * @param context The Android application context
         * @param trackerClass Default tracker class
         */
        public TrackerBuilder(Emitter emitter, String namespace, String appId, Context context,
                              Class<? extends Tracker> trackerClass) {
            this.emitter = emitter;
            this.namespace = namespace;
            this.appId = appId;
            this.context = context;
            this.trackerClass = trackerClass;
        }

        /**
         * @param subject Subject to be tracked
         * @return itself
         */
        public TrackerBuilder subject(Subject subject) {
            this.subject = subject;
            return this;
        }

        /**
         * @param base64 Whether JSONs in the payload should be base-64 encoded
         * @return itself
         */
        public TrackerBuilder base64(Boolean base64) {
            this.base64Encoded = base64;
            return this;
        }


        /**
         * @param log The log level for the Tracker class
         * @return itself
         */
        public TrackerBuilder level(LogLevel log) {
            this.logLevel = log;
            return this;
        }

        /**
         * @param sessionContext whether to add a session context
         * @return itself
         */
        public TrackerBuilder sessionContext(boolean sessionContext) {
            this.sessionContext = sessionContext;
            return this;
        }

        /**
         * @param timeout The session foreground timeout
         * @return itself
         */
        public TrackerBuilder foregroundTimeout(long timeout) {
            this.foregroundTimeout = timeout;
            return this;
        }

        /**
         * @param timeout The session background timeout
         * @return itself
         */
        public TrackerBuilder backgroundTimeout(long timeout) {
            this.backgroundTimeout = timeout;
            return this;
        }

        /**
         * @param sessionCheckInterval The session check interval
         * @return itself
         */
        public TrackerBuilder sessionCheckInterval(long sessionCheckInterval) {
            this.sessionCheckInterval = sessionCheckInterval;
            return this;
        }

        /**
         * @param threadCount the amount of threads to use for concurrency
         * @return itself
         */
        public TrackerBuilder threadCount(int threadCount) {
            this.threadCount = threadCount;
            return this;
        }

        /**
         * @param timeUnit a valid TimeUnit
         * @return itself
         */
        public TrackerBuilder timeUnit(TimeUnit timeUnit) {
            this.timeUnit = timeUnit;
            return this;
        }

    }

    /**
     * Creates a new Snowplow Tracker.
     *
     * @param builder The builder that constructs a tracker
     */
    public Tracker(TrackerBuilder builder) {
        this.emitter = builder.emitter;
        this.appId = builder.appId;
        this.base64Encoded = builder.base64Encoded;
        this.namespace = builder.namespace;
        this.subject = builder.subject;
        this.level = builder.logLevel;
        this.sessionContext = builder.sessionContext;
        this.sessionCheckInterval = builder.sessionCheckInterval;
        this.threadCount = builder.threadCount < 2 ? 2 : builder.threadCount;
        this.timeUnit = builder.timeUnit;

        // If session context is True
        if (this.sessionContext) {
            this.trackerSession = new Session(
                    builder.foregroundTimeout,
                    builder.backgroundTimeout,
                    builder.timeUnit,
                    builder.context);
        }

        Logger.updateLogLevel(builder.logLevel);
        Logger.i(TAG, "Tracker created successfully.");
    }

    /**
     * Builds and adds a finalized payload by adding in extra
     * information to the payload:
     * - The event contexts
     * - The Tracker Subject
     * - The Tracker parameters
     *
     * @param payload Payload the raw event payload to be
     *                decorated.
     * @param context The raw context list
     */
    private void addEventPayload(TrackerDataload payload, List<SelfDescribingJson> context) {

        // If there is a subject present for the Tracker add it
        if (this.subject != null) {
            payload.addMap(new HashMap<String, Object>(this.subject.getSubject()));
        }

        // Build the final context and add it
        SelfDescribingJson envelope = getFinalContext(context);
        payload.add(Parameters.CONTEXT,envelope.getMap());
        // Add this payload to the emitter
        Logger.i(TAG, "Adding new payload to event storage: %s", payload);
        this.emitter.add(payload);
    }

    /**
     * Builds the final event context.
     *
     * @param context the base event context
     * @return the final event context json with
     *         many contexts inside
     */
    private SelfDescribingJson getFinalContext(List<SelfDescribingJson> context) {

        // Add session context
        if (this.sessionContext) {
            context.add(this.trackerSession.getSessionContext());
        }

        // Add subject context's
        if (this.subject != null) {
            if (!this.subject.getSubjectLocation().isEmpty()) {
                SelfDescribingJson locationPayload = new SelfDescribingJson(
                        TrackerConstants.GEOLOCATION_SCHEMA, this.subject.getSubjectLocation());
                context.add(locationPayload);
            }
            if (!this.subject.getSubjectMobile().isEmpty()) {
                SelfDescribingJson mobilePayload = new SelfDescribingJson(
                        TrackerConstants.MOBILE_SCHEMA, this.subject.getSubjectMobile());
                context.add(mobilePayload);
            }
        }

        // Convert List of SelfDescribingJson into a List of Map
        List<Map> contextMaps = new LinkedList<>();
        for (SelfDescribingJson selfDescribingJson : context) {
            contextMaps.add(selfDescribingJson.getMap());
        }

        // Return the contexts as a new SelfDescribingJson
        return new SelfDescribingJson(TrackerConstants.SCHEMA_CONTEXTS, contextMaps);
    }


    /**
     * Tracks a Push Event.
     * override this method in classic tracker ,or it run main thread
     * @param event the Structured event.
     */
    public void track(PushEvent event) {
        if (!dataCollection.get()) {
            return;
        }

        List<SelfDescribingJson> context = event.getSelfDescribingJson();
        TrackerDataload payload = event.getDataLoad();
        addEventPayload(payload, context);
    }


    /**
     * Starts the session checker on a
     * polling interval.
     */
    public abstract void resumeSessionChecking();

    /**
     * Shuts the session checker down.
     */
    public abstract void pauseSessionChecking();

    /**
     * Stops event collection and ends all
     * concurrent processes.
     */
    public void pauseEventTracking() {
        if (dataCollection.compareAndSet(true, false)) {
            pauseSessionChecking();
            getEmitter().shutdown();
        }
    }

    /**
     * Starts event collection processes
     * again.
     */
    public void resumeEventTracking() {
        if (dataCollection.compareAndSet(false, true)) {
            resumeSessionChecking();
            getEmitter().flush();
        }
    }

    public void restartEventTracking(){
        if (!dataCollection.get()) {
            return;
        }
        getEmitter().flush();
    }

    // Get & Set Functions

    /**
     * @param subject a valid subject object
     */
    public void setSubject(Subject subject) {
        this.subject = subject;
    }

    /**
     * @param emitter a valid emitter object
     */
    public void setEmitter(Emitter emitter) {
        // Need to shutdown prior emitter before updating
        getEmitter().shutdown();

        // Set the new emitter
        this.emitter = emitter;
    }


    /**
     * @return the tracker version that was set
     */
    public String getTrackerVersion() {
        return this.trackerVersion;
    }

    /**
     * @return the trackers subject object
     */
    public Subject getSubject() {
        return this.subject;
    }

    /**
     * @return the emitter associated with the tracker
     */
    public Emitter getEmitter() {
        return this.emitter;
    }

    /**
     * @return the trackers namespace
     */
    public String getNamespace() {
        return this.namespace;
    }

    /**
     * @return the trackers set Application ID
     */
    public String getAppId() {
        return this.appId;
    }

    /**
     * @return the base64 setting of the tracker
     */
    public boolean getBase64Encoded() {
        return this.base64Encoded;
    }

    /**
     * @return the trackers logging level
     */
    public LogLevel getLogLevel() {
        return this.level;
    }

    /**
     * @return the trackers session object
     */
    public Session getSession() {
        return this.trackerSession;
    }

    /**
     * @return the state of data collection
     */
    public boolean getDataCollection() {
        return this.dataCollection.get();
    }

    /**
     * @return the amount of threads to use
     */
    public int getThreadCount() { return this.threadCount; }
}
