package com.meizu.cloud.pushsdk.common.event;

import android.annotation.TargetApi;
import android.content.Context;
import android.os.Build;
import android.os.Handler;
import android.os.HandlerThread;
import android.os.Looper;


import com.meizu.cloud.pushsdk.common.base.CountableWithConfig;
import com.meizu.cloud.pushsdk.common.util.Logger;
import com.meizu.cloud.pushsdk.common.util.Utility;

import java.util.ArrayList;
import java.util.HashMap;


/**
 * Created by zhangbin on 16-3-24.
 */
public class EventCore extends CountableWithConfig<Context> {
    private static EventCore sInstance;
    private HandlerThread mHandlerThread;
    private Handler mHandler;
    private HashMap<Integer, ArrayList<Listener>> mListenerMap;
    private HashMap<Integer, Event> mStickyEventMap;
    private ArrayList<Serviceable> mServices;

    private EventCore() {
        super("EventCore");
        mListenerMap = new HashMap<>();
        mStickyEventMap = new HashMap<>();
        mServices = new ArrayList<>();
    }

    public synchronized static EventCore get() {
        if(sInstance == null) {
            sInstance = new EventCore();
        }
        return sInstance;
    }

    public Looper getLooper() {
        return mHandler.getLooper();
    }

    public Context getContext() {
        return mConfig;
    }

    public EventCore post(Runnable task) {
        mHandler.post(task);
        return this;
    }

    public EventCore postDelayed(Runnable task, long delayMillis) {
        mHandler.postDelayed(task, delayMillis);
        return this;
    }

    public EventCore post(final Event event) {
        if(mHandler.hasMessages(event.getId())) {
            mHandler.removeMessages(event.getId());
        }
        mHandler.postDelayed(new Runnable() {
            @Override
            public void run() {
                Logger.d(mTag, "dispatch: " + event);
                ArrayList<Listener> listeners = mListenerMap.get(event.getId());
                if (listeners != null) {
                    for (Listener listener : listeners) {
                        listener.onEvent(event);
                    }
                }
                if (event.isSticky()) {
                    mStickyEventMap.put(event.getId(), event);
                }
            }
        }, event.getDelay());
        return this;
    }

    public <T> T getEventData(int id) {
        try {
            if(mStickyEventMap.containsKey(id)) {
                return (T)mStickyEventMap.get(id).getData();
            }
        } catch (ClassCastException e) {}
        return null;
    }

    public EventCore registerListener(final int id, final Listener listener) {
        mHandler.post(new Runnable() {
            @Override
            public void run() {
                ArrayList<Listener> listeners = mListenerMap.get(id);
                if (listeners == null) {
                    listeners = new ArrayList<>();
                    mListenerMap.put(id, listeners);
                }
                listeners.add(listener);
                if (mStickyEventMap.containsKey(id)) {
                    listener.onEvent(mStickyEventMap.get(id));
                }
            }
        });
        return this;
    }

    public EventCore unregisterListener(final int id, final Listener listener) {
        mHandler.post(new Runnable() {
            @Override
            public void run() {
                ArrayList<Listener> listeners = mListenerMap.get(id);
                if(listeners != null) {
                    listeners.remove(listener);
                }
            }
        });
        return this;
    }

    public EventCore loadService(final Serviceable service) {
        mHandler.post(new Runnable() {
            @Override
            public void run() {
                if(!mServices.contains(service)) {
                    mServices.add(service);
                    service.onStart();
                }
            }
        });
        return this;
    }

    public EventCore unloadService(final Serviceable service) {
        mHandler.post(new Runnable() {
            @Override
            public void run() {
                if(mServices.remove(service)) {
                    service.onStop();
                }
            }
        });
        return this;
    }

    @Override
    protected synchronized void onInit() {
        if(mConfig == null) {
            throw new NullPointerException("context must not be null");
        }
        Logger.i(mTag, "onInit");
        mConfig = mConfig.getApplicationContext();
        mHandlerThread = new HandlerThread(mTag);
        mHandlerThread.start();
        mHandler = new Handler(mHandlerThread.getLooper());
    }

    @Override
    protected synchronized void onDestroy() {
        Logger.i(mTag, "onDestroy");
        if(mHandlerThread != null) {
            if(Utility.compatApi(18)) {
                quitAPI18();
            } else {
                mHandlerThread.quit();
            }
        }
        sInstance = null;
    }

    @TargetApi(Build.VERSION_CODES.JELLY_BEAN_MR2)
    private void quitAPI18() {
        mHandlerThread.quitSafely();
    }

}
