package com.meizu.cloud.pushsdk.common.util;

import android.os.Handler;
import android.os.Looper;
import android.os.Message;
import android.util.Log;

import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.LinkedList;

/**
 * Created by zbin on 2016/1/13 0013.
 */
public class Logger {
    private static Handler sHandler = new WriteHandler(Looper.getMainLooper());
    private static final int MSG_CALLBACK_LOG = 0x01;
    private static Callback sCallback;
    private static LinkedList<LogCache> sLogCaches = new LinkedList<>();
    private static Callback.Level sConsoleLevel = Callback.Level.DEBUG;
    private static Callback.Level sFileLevel = Callback.Level.DEBUG;
    private static Option sOption = new Option();

    /**
     * 设置日志的输出级别
     * @param out 输出类型
     * @param level 输出等级，>= level的日志可以输出
     */
    public static void setOutLevel(Out out, Callback.Level level) {
        if(out == Out.CONSOLE) {
            sConsoleLevel = level;
        } else if(out == Out.FILE) {
            sFileLevel = level;
        }
    }

    /**
     * 设置app日志回调
     * @param callback 回调接口
     */
    public static void setCallback(Callback callback) {
        sCallback = callback;
    }

    public static Option getOption() {
        return sOption;
    }

    public static void flush() {
        synchronized (sLogCaches) {
            sHandler.removeMessages(MSG_CALLBACK_LOG);
            sHandler.obtainMessage(MSG_CALLBACK_LOG).sendToTarget();
        }
    }

    /**
     * 缓存日志
     * @param level 日志等级
     * @param tag 日志标签
     * @param msg 日志内容
     */
    private static void cacheLog(Callback.Level level, String tag, String msg) {
        if(sCallback != null &&
                sFileLevel.ordinal() <= level.ordinal()) {
            synchronized (sLogCaches) {
                sLogCaches.addLast(new LogCache(level, tag, msg));
                if(sLogCaches.size() >= sOption.mCacheCounts ||
                        sOption.mCacheMillis <= 0) {
                    flush();
                } else {
                    if(!sHandler.hasMessages(MSG_CALLBACK_LOG)) {
                        Message message = sHandler.obtainMessage(MSG_CALLBACK_LOG);
                        sHandler.sendMessageDelayed(message, sOption.mCacheMillis);
                    }
                }
            }
        }
    }

    public static void d(String tag, String msg) {
        if(sConsoleLevel.ordinal() <= Callback.Level.DEBUG.ordinal()) {
            Log.d(tag, msg);
        }
        cacheLog(Callback.Level.DEBUG, tag, msg);
    }

    public static void i(String tag, String msg) {
        if(sConsoleLevel.ordinal() <= Callback.Level.INFO.ordinal()) {
            Log.i(tag, msg);
        }
        cacheLog(Callback.Level.INFO, tag, msg);
    }

    public static void w(String tag, String msg) {
        if(sConsoleLevel.ordinal() <= Callback.Level.WARN.ordinal()) {
            Log.w(tag, msg);
        }
        cacheLog(Callback.Level.WARN, tag, msg);
    }

    public static void e(String tag, String msg) {
        if(sConsoleLevel.ordinal() <= Callback.Level.ERROR.ordinal()) {
            Log.e(tag, msg);
        }
        cacheLog(Callback.Level.ERROR, tag, msg);
    }

    public static void trace(String tag, Throwable tr) {
        e(tag, Log.getStackTraceString(tr));
    }

    private static class LogCache {
        static SimpleDateFormat sDateFormat = new SimpleDateFormat("MM-dd HH:mm:ss ");
        static String sPid = String.valueOf(android.os.Process.myPid());
        Callback.Level mLevel;
        String mTag;
        String mMsg;

        LogCache(Callback.Level level, String tag, String msg) {
            mLevel = level;
            mTag = sDateFormat.format(new Date()) +
                    sPid + "-" + String.valueOf(Thread.currentThread().getId()) +
                    " " + tag;
            mMsg = msg;
        }
    }

    /**
     * 日志输出对象枚举
     */
    public enum Out {
        CONSOLE,    //logcat 控制台
        FILE    //app callback
    }

    public interface Callback {
        enum Level {
            DEBUG, INFO, WARN, ERROR, NULL
        }
        void print(Level level, String tag, String msg);
    }

    private static class WriteHandler extends Handler {
        WriteHandler(Looper looper) {
            super(looper);
        }

        @Override
        public void handleMessage(Message msg) {
            if(sCallback != null) {
                ThreadPool.execute(new ThreadPool.Task() {

                    @Override
                    public void doInBackground() {
                        LinkedList<LogCache> temp;
                        synchronized (sLogCaches) {
                            temp = new LinkedList<>(sLogCaches);
                            sLogCaches.clear();
                        }
                        for (LogCache cache : temp) {
                            sCallback.print(cache.mLevel, cache.mTag, cache.mMsg);
                        }
                    }
                });
            }
        }
    }

    public static class Option {
        int mCacheCounts = 100;
        int mCacheMillis = 120 * 1000;

        public Option setCacheCounts(int cacheCounts) {
            mCacheCounts = cacheCounts;
            return this;
        }

        public Option setCacheSeconds(int cacheMillis) {
            mCacheMillis = cacheMillis;
            return this;
        }
    }
}
