package com.meizu.cloud.pushsdk.handler.impl;

import android.content.Context;
import android.content.Intent;
import android.text.TextUtils;

import com.meizu.cloud.pushsdk.handler.AbstractAppLogicListener;
import com.meizu.cloud.pushsdk.handler.MessageHandler;
import com.meizu.cloud.pushsdk.constants.PushConstants;
import com.meizu.cloud.pushsdk.notification.PushNotification;
import com.meizu.cloud.pushinternal.DebugLogger;
import com.meizu.cloud.pushsdk.pushtracer.utils.Util;
import com.meizu.cloud.pushsdk.util.MzSystemUtils;
import com.meizu.cloud.pushsdk.util.PushPreferencesUtils;
import org.json.JSONObject;

import java.util.HashMap;
import java.util.Map;


/**
 * Created by liaojinlong on 16-1-7.
 */
public abstract class AbstractMessageHandler<T> implements MessageHandler {
    protected static final String TAG = "AbstractMessageHandler";
    public static final int MESSAGE_TYPE_PUSH_SERVICE_V2 = 1 << 1;
    public static final int MESSAGE_TYPE_PUSH_SERVICE_V3 = 1 << 2;
    public static final int MESSAGE_TYPE_THROUGH = 1 << 3;
    public static final int MESSAGE_TYPE_REGISTER = 1 << 4;
    public static final int MESSAGE_TYPE_UNREGISTER = 1 << 5;
    public static final int MESSAGE_TYPE_NOTIFICATION_CLICK = 1 << 6;
    public static final int MESSAGE_TYPE_NOTIFICATION_DELETE = 1 << 7;
    public static final int MESSAGE_TYPE_PUSH_SWITCH_STATUS = 1 << 8;
    public static final int MESSAGE_TYPE_PUSH_REGISTER_STATUS = 1 << 9;
    public static final int MESSAGE_TYPE_PUSH_UNREGISTER_STATUS = 1 << 10;
    public static final int MESSAGE_TYPE_PUSH_SUBTAGS_STATUS = 1 << 11;
    public static final int MESSAGE_TYPE_PUSH_SUBALIAS_STATUS = 1 << 12;
    public static final int MESSAGE_TYPE_SCHEDULE_NOTIFICATION = 1 << 13;
    public static final int MESSAGE_TYPE_RECEIVE_NOTIFY_MESSAGE = 1 << 14;

    private AbstractAppLogicListener abstractAppLogicListener;
    private Context context;
    private Map<Integer,String> messageHandlerMap;


    /**schedule status code**/
    public static final int SCHEDULE_OFF = 0;
    public static final int SCHEDULE_ON_EXPIRE = 1;
    public static final int SCHEDULE_ON_TIME = 2;
    public static final int SCHEDULE_ON_DELAY = 3;

    protected AbstractMessageHandler(Context context){
        this(context,null);
    }

    protected AbstractMessageHandler(Context context, AbstractAppLogicListener abstractAppLogicListener){
        if(context == null){
            throw new IllegalArgumentException("Context must not be null.");
        }
        this.context = context.getApplicationContext();
        this.abstractAppLogicListener = abstractAppLogicListener;
        messageHandlerMap = new HashMap<>();
        messageHandlerMap.put(MESSAGE_TYPE_PUSH_SERVICE_V2,"MESSAGE_TYPE_PUSH_SERVICE_V2");
        messageHandlerMap.put(MESSAGE_TYPE_PUSH_SERVICE_V3,"MESSAGE_TYPE_PUSH_SERVICE_V3");
        messageHandlerMap.put(MESSAGE_TYPE_REGISTER,"MESSAGE_TYPE_REGISTER");
        messageHandlerMap.put(MESSAGE_TYPE_UNREGISTER,"MESSAGE_TYPE_UNREGISTER");
        messageHandlerMap.put(MESSAGE_TYPE_THROUGH,"MESSAGE_TYPE_THROUGH");
        messageHandlerMap.put(MESSAGE_TYPE_NOTIFICATION_CLICK,"MESSAGE_TYPE_NOTIFICATION_CLICK");
        messageHandlerMap.put(MESSAGE_TYPE_NOTIFICATION_DELETE,"MESSAGE_TYPE_NOTIFICATION_DELETE");
        messageHandlerMap.put(MESSAGE_TYPE_PUSH_SWITCH_STATUS,"MESSAGE_TYPE_PUSH_SWITCH_STATUS");
        messageHandlerMap.put(MESSAGE_TYPE_PUSH_REGISTER_STATUS,"MESSAGE_TYPE_PUSH_REGISTER_STATUS");
        messageHandlerMap.put(MESSAGE_TYPE_PUSH_SUBTAGS_STATUS,"MESSAGE_TYPE_PUSH_SUBTAGS_STATUS");
        messageHandlerMap.put(MESSAGE_TYPE_PUSH_UNREGISTER_STATUS,"MESSAGE_TYPE_PUSH_UNREGISTER_STATUS");
        messageHandlerMap.put(MESSAGE_TYPE_PUSH_SUBALIAS_STATUS,"MESSAGE_TYPE_PUSH_SUBALIAS_STATUS");
        messageHandlerMap.put(MESSAGE_TYPE_SCHEDULE_NOTIFICATION,"MESSAGE_TYPE_SCHEDULE_NOTIFICATION");
        messageHandlerMap.put(MESSAGE_TYPE_RECEIVE_NOTIFY_MESSAGE,"MESSAGE_TYPE_RECEIVE_NOTIFY_MESSAGE");
    }


    protected abstract T getMessage(Intent intent);

    protected abstract void unsafeSend(T message, PushNotification pushNotification);

    protected  PushNotification onCreateNotification(T message){
        return null;
    };

    /**
     * record receive push event
     * */
    protected void onBeforeEvent(T message){}

    protected void onAfterEvent(T Message){}

    /**
     * schedule notification status
     * default is schedule off
     * */
    protected int scheduleNotificationStatus(T message){
        return SCHEDULE_OFF;
    }

    protected void scheduleShowNotification(T message){

    }
    /**
     * get deviceId for onEvent
     * */
    protected String getDeviceId(Intent intent){
       String deviceId = intent.getStringExtra(PushConstants.MZ_PUSH_MESSAGE_STATISTICS_IMEI_KEY);
       if(TextUtils.isEmpty(deviceId)){
           deviceId = PushPreferencesUtils.getDeviceId(context());
           if(TextUtils.isEmpty(deviceId)){
               //兼容5.6.0之前云服务的没有传递deviceID,导致deviceId为空
               deviceId = MzSystemUtils.getDeviceId(context());
               PushPreferencesUtils.putDeviceId(context(), deviceId);
               DebugLogger.e(TAG,"force get deviceId "+deviceId);
           }
       }
       return  deviceId;
    }

    protected String getTaskId(Intent intent){
        return intent.getStringExtra(PushConstants.EXTRA_APP_PUSH_TASK_ID);
    }

    protected String getSeqId(Intent intent){
        return intent.getStringExtra(PushConstants.EXTRA_APP_PUSH_SEQ_ID);
    }

    protected String getPushServiceDefaultPackageName(Intent intent){
        String defaultPackageName = intent.getStringExtra(PushConstants.EXTRA_APP_PUSH_SERVICE_DEFAULT_PACKAGE_NAME);
        if(TextUtils.isEmpty(defaultPackageName)){
            defaultPackageName = context().getPackageName();
        }
        return defaultPackageName;
    }

    protected String getPushTimestamp(Intent intent){
        String pushTimestamp = intent.getStringExtra(PushConstants.EXTRA_APP_PUSH_TASK_TIMES_TAMP);
        DebugLogger.e(TAG,"receive push timestamp from pushservice "+pushTimestamp);
        if(TextUtils.isEmpty(pushTimestamp)){
            pushTimestamp = String.valueOf(System.currentTimeMillis() / 1000);
        }
        return pushTimestamp;
    }

    @Override
    public boolean sendMessage(Intent intent) {
        boolean flag = false;
        if(messageMatch(intent)){
            DebugLogger.e(TAG,"current message Type "+getMessageHandlerType(getProcessorType()));
            T message = getMessage(intent);
            //获取消息立即埋点,避免其它错误影响数据写入
            DebugLogger.e(TAG,"current Handler message "+message);
            onBeforeEvent(message);
            boolean isCustomSend = false;
            switch (scheduleNotificationStatus(message)){
                case SCHEDULE_OFF:
                    DebugLogger.e(TAG,"schedule send message off, send message directly");
                    isCustomSend = true;
                    flag = true;
                    break;
                case SCHEDULE_ON_EXPIRE:
                    DebugLogger.e(TAG,"expire notification, dont show message");
                    flag = false;
                    break;
                case SCHEDULE_ON_TIME:
                    DebugLogger.e(TAG,"notification on time ,show message");
                    isCustomSend = true;
                    flag = true;
                    break;
                case SCHEDULE_ON_DELAY:
                    DebugLogger.e(TAG,"schedule notification");
                    scheduleShowNotification(message);
                    flag = true;
                    break;
            }
            if(flag && isCustomSend){
                unsafeSend(message, onCreateNotification(message));
                onAfterEvent(message);
                DebugLogger.e(TAG,"send message end ");
            }
        }
        return flag;
    }


    public AbstractAppLogicListener appLogicListener(){
        return abstractAppLogicListener;
    }

    public Context context(){
        return context;
    }

    /**
     * get intent method
     * @return intent
     * */
    public String getIntentMethod(Intent intent){
       return intent.getStringExtra(PushConstants.MZ_PUSH_MESSAGE_METHOD);
    }

    /**
     * check the notification json type
     * */
    public boolean isNotificationJson(String notificationJson){
        try {
            JSONObject pushMessageObj = new JSONObject(notificationJson);
            return context().getPackageName().equals(pushMessageObj.getString("appId"));
        } catch (Exception e) {
            //for mock RunTime exception
            DebugLogger.e(TAG,"parse notification error");
            return false;
        }
    }

    /**
     * get selfDefineContentString
     * */
    public String selfDefineContentString(String uri,Map<String,String> map){
        String selfJson = null;
        if(!TextUtils.isEmpty(uri)){
            selfJson = uri;
        } else if(map != null){
            selfJson = map.get("sk");
            if(TextUtils.isEmpty(selfJson)){
                selfJson = Util.mapToJSONObject(map).toString();
            }
        }
        DebugLogger.e(TAG,"self json "+selfJson);
        return selfJson;
    }

    private String getMessageHandlerType(int type){
        return messageHandlerMap.get(type);
    }

    protected boolean canReceiveMessage(int pushType,String destPackageName){
        boolean flag = true;
        if(pushType == 0){
            flag = PushPreferencesUtils.getNotificationMessageSwitchStatus(context(),destPackageName);
        } else if(pushType == 1){
            flag = PushPreferencesUtils.getThroughMessageSwitchStatus(context(),destPackageName);
        }
        DebugLogger.e(TAG,destPackageName + (pushType == 0 ?  " canNotificationMessage ":" canThroughMessage ") + flag);
        return flag;
    }


}
