/*
 *    Copyright (C) 2016 Amit Shekhar
 *    Copyright (C) 2011 Android Open Source Project
 *
 *    Licensed under the Apache License, Version 2.0 (the "License");
 *    you may not use this file except in compliance with the License.
 *    You may obtain a copy of the License at
 *
 *        http://www.apache.org/licenses/LICENSE-2.0
 *
 *    Unless required by applicable law or agreed to in writing, software
 *    distributed under the License is distributed on an "AS IS" BASIS,
 *    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *    See the License for the specific language governing permissions and
 *    limitations under the License.
 */

package com.meizu.cloud.pushsdk.networking.internal;

/**
 * Created by amitshekhar on 22/03/16.
 */
import com.meizu.cloud.pushsdk.networking.common.ANConstants;
import com.meizu.cloud.pushsdk.networking.common.ANRequest;
import com.meizu.cloud.pushsdk.networking.error.ANError;
import com.meizu.cloud.pushsdk.networking.http.*;
import com.meizu.cloud.pushsdk.networking.http.RequestBody;
import com.meizu.cloud.pushsdk.networking.utils.Utils;

import static com.meizu.cloud.pushsdk.networking.common.Method.DELETE;
import static com.meizu.cloud.pushsdk.networking.common.Method.GET;
import static com.meizu.cloud.pushsdk.networking.common.Method.HEAD;
import static com.meizu.cloud.pushsdk.networking.common.Method.PATCH;
import static com.meizu.cloud.pushsdk.networking.common.Method.POST;
import static com.meizu.cloud.pushsdk.networking.common.Method.PUT;
import java.io.File;
import java.io.IOException;


public final class InternalNetworking {

    private InternalNetworking() {

    }
    public static String sUserAgent = null;

    public static Response performSimpleRequest(ANRequest request) throws ANError {
        Request okHttpRequest;
        Response okHttpResponse;
        try {
            Request.Builder builder = new Request.Builder().url(request.getUrl());
            addHeadersToRequestBuilder(builder, request);
            RequestBody requestBody = null;
            switch (request.getMethod()) {
                case GET: {
                    builder = builder.get();
                    break;
                }
                case POST: {
                    requestBody = request.getRequestBody();
                    builder = builder.post(requestBody);
                    break;
                }
                case PUT: {
                    requestBody = request.getRequestBody();
                    builder = builder.put(requestBody);
                    break;
                }
                case DELETE: {
                    requestBody = request.getRequestBody();
                    builder = builder.delete(requestBody);
                    break;
                }
                case HEAD: {
                    builder = builder.head();
                    break;
                }
                case PATCH: {
                    requestBody = request.getRequestBody();
                    builder = builder.patch(requestBody);
                    break;
                }
            }
            okHttpRequest = builder.build();
            request.setCall(new HttpURLConnectionCall(okHttpRequest));
            okHttpResponse = request.getCall().execute();
        } catch (IOException ioe) {
            throw new ANError(ioe);
        }
        return okHttpResponse;
    }

    public static Response performDownloadRequest(final ANRequest request) throws ANError {
        Request okHttpRequest;
        Response okHttpResponse;
        try {
            Request.Builder builder = new Request.Builder().url(request.getUrl());
            addHeadersToRequestBuilder(builder, request);
            builder = builder.get();
            okHttpRequest = builder.build();
            request.setCall(new HttpURLConnectionCall(okHttpRequest));
            okHttpResponse = request.getCall().execute();
            Utils.saveFile(okHttpResponse, request.getDirPath(), request.getFileName());
        } catch (IOException ioe) {
            try {
                File destinationFile = new File(request.getDirPath() + File.separator + request.getFileName());
                if (destinationFile.exists()) {
                    destinationFile.delete();
                }
            } catch (Exception e) {
                e.printStackTrace();
            }
            throw new ANError(ioe);
        }
        return okHttpResponse;
    }


    public static Response performUploadRequest(ANRequest request) throws ANError {
        Request okHttpRequest;
        Response okHttpResponse;
        try {
            Request.Builder builder = new Request.Builder().url(request.getUrl());
            addHeadersToRequestBuilder(builder, request);
            final RequestBody requestBody = request.getMultiPartRequestBody();
            builder = builder.post(new RequestProgressBody(requestBody, request.getUploadProgressListener()));
            okHttpRequest = builder.build();
            request.setCall(new HttpURLConnectionCall(okHttpRequest));
            okHttpResponse = request.getCall().execute();
        } catch (IOException ioe) {
            throw new ANError(ioe);
        }
        return okHttpResponse;
    }

    public static void addHeadersToRequestBuilder(Request.Builder builder, ANRequest request) {
        if (request.getUserAgent() != null) {
            builder.addHeader(ANConstants.USER_AGENT, request.getUserAgent());
        } else if (sUserAgent != null) {
            request.setUserAgent(sUserAgent);
            builder.addHeader(ANConstants.USER_AGENT, sUserAgent);
        }
        Headers requestHeaders = request.getHeaders();
        if (requestHeaders != null) {
            builder.headers(requestHeaders);
            if (request.getUserAgent() != null && !requestHeaders.names().contains(ANConstants.USER_AGENT)) {
                builder.addHeader(ANConstants.USER_AGENT, request.getUserAgent());
            }
        }
    }

    public static void setUserAgent(String userAgent) {
        sUserAgent = userAgent;
    }

}
