package com.meizu.cloud.pushsdk.platform.api;

import android.content.Context;
import android.os.Bundle;
import android.os.Handler;
import android.os.Message;
import android.text.TextUtils;
import com.meizu.cloud.pushinternal.DebugLogger;
import com.meizu.cloud.pushsdk.networking.error.ANError;
import com.meizu.cloud.pushsdk.networking.http.Response;
import com.meizu.cloud.pushsdk.networking.interfaces.OkHttpResponseAndStringRequestListener;
import com.meizu.cloud.pushsdk.platform.PlatformMessageSender;
import com.meizu.cloud.pushsdk.platform.message.PushSwitchStatus;
import com.meizu.cloud.pushsdk.platform.message.RegisterStatus;
import com.meizu.cloud.pushsdk.platform.message.SubAliasStatus;
import com.meizu.cloud.pushsdk.platform.message.SubTagsStatus;
import com.meizu.cloud.pushsdk.platform.message.UnRegisterStatus;
import com.meizu.cloud.pushsdk.util.MzSystemUtils;
import com.meizu.cloud.pushsdk.util.PushPreferencesUtils;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;


/**
 * Created by liaojinlong on 16-11-25.
 */
public class PushPlatformManager {
    private static final String TAG = "PushPlatformManager";
    private static PushPlatformManager mInstance;
    private ScheduledExecutorService executorService;
    private Handler mainHandler;
    private Context mContext;
    private PushAPI pushAPI;
    private int deviceIdRetry = 0;

    public PushPlatformManager(Context context){
        this.mContext = context;
        this.pushAPI = new PushAPI(context);
        executorService = Executors.newSingleThreadScheduledExecutor();
        mainHandler = new Handler(mContext.getMainLooper()){
            @Override
            public void handleMessage(Message msg) {
                String deviceId = msg.getData().getString("deviceId");
                String appId = msg.getData().getString("appId");
                String appKey = msg.getData().getString("appKey");
                switch (msg.what){
                    case 0:
                        register(appId,appKey,deviceId);
                        break;
                    case 1:
                        unRegister(appId,appKey,deviceId);
                        break;
                    case 2:
                        String packageName = msg.getData().getString("packageName");
                        unRegisterAdvance(packageName,deviceId);
                }

            }
        };
    }


    public static PushPlatformManager getInstance(Context context){
        if(mInstance == null){
            synchronized (PushPlatformManager.class){
                if(mInstance == null){
                    mInstance = new PushPlatformManager(context);
                }
            }
        }
        return mInstance;
    }

    /**
     * pushId 订阅 PushManger 接口使用
     * @param appId
     * @param appKey
     * */
    public void register(String appId,String appKey){
        String pushId = PushPreferencesUtils.getPushId(mContext);
        int expireTime = PushPreferencesUtils.getPushIdExpireTime(mContext);
        if(!TextUtils.isEmpty(pushId) && System.currentTimeMillis()/1000 < expireTime){
            RegisterStatus registerStatus = new RegisterStatus();
            registerStatus.setCode("200");
            registerStatus.setMessage("already register PushId,dont register frequently");
            registerStatus.setPushId(pushId);
            registerStatus.setExpireTime((int) (expireTime - System.currentTimeMillis()/1000));
            PlatformMessageSender.sendRegisterStatus(mContext, registerStatus);
        } else {
            //remove pushId when it expired
            PushPreferencesUtils.putPushId(mContext,"");
            DebugLogger.i(TAG,"after "+deviceIdRetry * 10 + " seconds start register");
            executeAfterGetDeviceId(0, appId, appKey, deviceIdRetry * 10);
        }
    }

    public void register(String appId,String appKey,String deviceId){
        if(TextUtils.isEmpty(deviceId) && deviceIdRetry++ < 3){
            DebugLogger.e(TAG,"Get deviceId error so after "+deviceIdRetry * 10 +" seconds retry register");
            executeAfterGetDeviceId(0,appId,appKey,deviceIdRetry * 10);
        } else {
            DebugLogger.i(TAG,"device retry count "+deviceIdRetry);
            deviceIdRetry = 0;
            pushAPI.register(appId, appKey, deviceId, new OkHttpResponseAndStringRequestListener() {
                @Override
                public void onResponse(Response okHttpResponse, String response) {
                    DebugLogger.e(TAG, response.toString());
                    RegisterStatus registerStatus = new RegisterStatus(response.toString());
                    DebugLogger.e(TAG, "registerStatus " + registerStatus);
                    PlatformMessageSender.sendRegisterStatus(mContext, registerStatus);
                }

                @Override
                public void onError(ANError error) {
                    if (error.getResponse() != null) {
                        DebugLogger.e(TAG, "status code=" + error.getErrorCode() + " data=" + error.getResponse());
                    }
                    RegisterStatus registerStatus = new RegisterStatus();
                    registerStatus.setCode(String.valueOf(error.getErrorCode()));
                    registerStatus.setMessage(error.getErrorBody());
                    DebugLogger.e(TAG, "registerStatus " + registerStatus);
                    PlatformMessageSender.sendRegisterStatus(mContext, registerStatus);
                }
            });
        }

    }

    public void register0(String appId,String appKey,String deviceId){
        DebugLogger.e(TAG, "PushService default package start register");
        pushAPI.register(appId, appKey, deviceId, new OkHttpResponseAndStringRequestListener() {
            @Override
            public void onResponse(Response okHttpResponse, String response) {
                DebugLogger.e(TAG, "Push Service"+response.toString());
            }

            @Override
            public void onError(ANError anError) {
                DebugLogger.e(TAG, "error code " + anError.getMessage());
            }
        });
    }

    public void unRegister(String appId,String appKey){
        String pushId = PushPreferencesUtils.getPushId(mContext);
        if(TextUtils.isEmpty(pushId)){
            UnRegisterStatus unRegisterStatus = new UnRegisterStatus();
            unRegisterStatus.setCode("200");
            unRegisterStatus.setMessage("already unRegister PushId,dont unRegister frequently");
            unRegisterStatus.setIsUnRegisterSuccess(true);
            PlatformMessageSender.sendUnRegisterStatus(mContext, unRegisterStatus);
        } else {
            executeAfterGetDeviceId(1,appId,appKey,0);
        }
    }


    public void unRegister(String appId,String appKey,String deviceId){
        pushAPI.unRegister(appId, appKey, deviceId, new OkHttpResponseAndStringRequestListener() {
            @Override
            public void onResponse(Response okHttpResponse, String response) {
                DebugLogger.e(TAG, "unRegister" + response.toString());
                UnRegisterStatus unRegisterStatus = new UnRegisterStatus(response.toString());
                DebugLogger.e(TAG, "unRegisterStatus " + unRegisterStatus);
                PlatformMessageSender.sendUnRegisterStatus(mContext, unRegisterStatus);
            }

            @Override
            public void onError(ANError anError) {
                UnRegisterStatus unRegisterStatus = new UnRegisterStatus();
                unRegisterStatus.setCode(String.valueOf(anError.getErrorCode()));
                unRegisterStatus.setIsUnRegisterSuccess(false);
                unRegisterStatus.setMessage(anError.getErrorBody());
                DebugLogger.e(TAG, "unRegisterStatus " + unRegisterStatus);
                PlatformMessageSender.sendUnRegisterStatus(mContext, unRegisterStatus);
            }
        });
    }


    /**
     * 此接口用于卸载应用后监听,并反注册该包名
     * @param packageName
     * */
    public void unRegisterAdvance(final String packageName){
        executorService.execute(new Runnable() {
            @Override
            public void run() {
                String deviceId = MzSystemUtils.getDeviceId(mContext);
                Message message = mainHandler.obtainMessage();
                Bundle bundle = new Bundle();
                bundle.putString("deviceId", deviceId);
                bundle.putString("packageName", packageName);
                message.setData(bundle);
                message.what = 2;
                DebugLogger.e(TAG, "deviceId " + deviceId + "packageName " + packageName);
                mainHandler.sendMessage(message);
            }
        });
    }

    public void unRegisterAdvance(final String packageName,String deviceId){
        pushAPI.unRegister(packageName, deviceId, new OkHttpResponseAndStringRequestListener() {
            @Override
            public void onResponse(Response okHttpResponse, String response) {
                DebugLogger.e(TAG, "unregisetr advance pakcage " + packageName + " result " + response);
            }

            @Override
            public void onError(ANError anError) {
                DebugLogger.e(TAG, "unregisetr advance pakcage " + packageName + " error " + anError.getErrorBody());
            }
        });
    }



    public void checkPush(String appId,String appKey, String pushId){
        pushAPI.checkPush(appId, appKey, pushId, new OkHttpResponseAndStringRequestListener() {
            @Override
            public void onResponse(Response okHttpResponse, String response) {
                DebugLogger.e(TAG, "checkPush" + response.toString());
                PushSwitchStatus pushSwitchStatus = new PushSwitchStatus(response.toString());
                DebugLogger.e(TAG, "pushSwitchStatus " + pushSwitchStatus);
                PlatformMessageSender.sendPushStatus(mContext, pushSwitchStatus);
            }

            @Override
            public void onError(ANError anError) {
                PushSwitchStatus pushSwitchStatus = new PushSwitchStatus();
                pushSwitchStatus.setCode(String.valueOf(anError.getErrorCode()));
                pushSwitchStatus.setMessage(anError.getErrorBody());
                DebugLogger.e(TAG, "pushSwitchStatus " + pushSwitchStatus);
                PlatformMessageSender.sendPushStatus(mContext, pushSwitchStatus);
            }
        });
    }

    public void switchPush(String appId,String appKey, String pushId, int msgType,boolean switcher){
        if(msgType == 0){
            PushPreferencesUtils.setNotificationMessageSwitchStatus(mContext,mContext.getPackageName(),switcher);
        } else if(msgType == 1){
            PushPreferencesUtils.setThroughMessageSwitchStatus(mContext,mContext.getPackageName(),switcher);
        }
        pushAPI.switchPush(appId, appKey, pushId, msgType, switcher, new OkHttpResponseAndStringRequestListener() {
            @Override
            public void onResponse(Response okHttpResponse, String response) {
                DebugLogger.e(TAG, "switchPush" + response.toString());
                PushSwitchStatus pushSwitchStatus = new PushSwitchStatus(response.toString());
                DebugLogger.e(TAG, "switchPush" + pushSwitchStatus);
                PlatformMessageSender.sendPushStatus(mContext, pushSwitchStatus);
            }

            @Override
            public void onError(ANError anError) {
                PushSwitchStatus pushSwitchStatus = new PushSwitchStatus();
                pushSwitchStatus.setCode(String.valueOf(anError.getErrorCode()));
                pushSwitchStatus.setMessage(anError.getErrorBody());
                DebugLogger.e(TAG, "pushSwitchStatus " + pushSwitchStatus);
                PlatformMessageSender.sendPushStatus(mContext, pushSwitchStatus);
            }
        });
        PlatformMessageSender.switchPushMessageSetting(mContext, msgType, switcher, mContext.getPackageName());
    }

    public void subScribeTags(String appId,String appKey,String pushId,String tags){
        pushAPI.subScribeTags(appId, appKey, pushId, tags, new OkHttpResponseAndStringRequestListener() {
            @Override
            public void onResponse(Response okHttpResponse, String response) {
                DebugLogger.e(TAG, "subScribeTags" + response.toString());
                SubTagsStatus subTagsStatus = new SubTagsStatus(response.toString());
                DebugLogger.e(TAG, "subScribeTags " + subTagsStatus.getTagList());
                PlatformMessageSender.sendSubTags(mContext, subTagsStatus);
            }

            @Override
            public void onError(ANError anError) {
                SubTagsStatus subTagsStatus = new SubTagsStatus();
                subTagsStatus.setCode(String.valueOf(anError.getErrorCode()));
                subTagsStatus.setMessage(anError.getErrorBody());
                DebugLogger.e(TAG, "subScribeTags " + subTagsStatus);
                PlatformMessageSender.sendSubTags(mContext, subTagsStatus);
            }
        });
    }

    public void unSubScribeTags(String appId,String appKey,String pushId,String tags){
        pushAPI.unSubScribeTags(appId, appKey, pushId, tags, new OkHttpResponseAndStringRequestListener() {
            @Override
            public void onResponse(Response okHttpResponse, String response) {
                DebugLogger.e(TAG, "unSubScribeTags " + response.toString());
                SubTagsStatus subTagsStatus = new SubTagsStatus(response.toString());
                DebugLogger.e(TAG, "unSubScribeTags " + subTagsStatus);
                PlatformMessageSender.sendSubTags(mContext, subTagsStatus);
            }

            @Override
            public void onError(ANError anError) {
                SubTagsStatus subTagsStatus = new SubTagsStatus();
                subTagsStatus.setCode(String.valueOf(anError.getErrorCode()));
                subTagsStatus.setMessage(anError.getErrorBody());
                DebugLogger.e(TAG, "subScribeTags " + subTagsStatus);
                PlatformMessageSender.sendSubTags(mContext, subTagsStatus);
            }
        });
    }

    public void checkSubScribeTags(String appId,String appKey,String pushId){
        pushAPI.checkSubScribeTags(appId, appKey, pushId, new OkHttpResponseAndStringRequestListener() {
            @Override
            public void onResponse(Response okHttpResponse, String response) {
                DebugLogger.e(TAG, "checkSubScribeTags " + response.toString());
                SubTagsStatus subTagsStatus = new SubTagsStatus(response.toString());
                DebugLogger.e(TAG, "checkSubScribeTags " + subTagsStatus);
                PlatformMessageSender.sendSubTags(mContext, subTagsStatus);
            }

            @Override
            public void onError(ANError anError) {
                SubTagsStatus subTagsStatus = new SubTagsStatus();
                subTagsStatus.setCode(String.valueOf(anError.getErrorCode()));
                subTagsStatus.setMessage(anError.getErrorDetail() + anError.getErrorBody());
                DebugLogger.e(TAG, "subScribeTags " + subTagsStatus);
                PlatformMessageSender.sendSubTags(mContext, subTagsStatus);
            }
        });
    }

    public void subScribeAlias(String appId,String appKey,String pushId,String alias){
        pushAPI.subScribeAlias(appId, appKey, pushId, alias, new OkHttpResponseAndStringRequestListener() {
            @Override
            public void onResponse(Response okHttpResponse, String response) {
                DebugLogger.e(TAG, "subScribeAlias " + response.toString());
                SubAliasStatus subAliasStatus = new SubAliasStatus(response.toString());
                DebugLogger.e(TAG, "subScribeAlias " + subAliasStatus);
                PlatformMessageSender.sendSubAlias(mContext, subAliasStatus);
            }

            @Override
            public void onError(ANError anError) {
                SubAliasStatus subAliasStatus = new SubAliasStatus();
                subAliasStatus.setCode(String.valueOf(anError.getErrorCode()));
                subAliasStatus.setMessage(anError.getErrorDetail()+" "+anError.getErrorBody());
                DebugLogger.e(TAG, "subScribeAlias " + subAliasStatus);
                PlatformMessageSender.sendSubAlias(mContext, subAliasStatus);
            }
        });
    }

    public void unSubScribeAlias(String appId,String appKey,String pushId,String alias){
        pushAPI.unSubScribeAlias(appId, appKey, pushId, alias, new OkHttpResponseAndStringRequestListener() {
            @Override
            public void onResponse(Response okHttpResponse, String response) {
                DebugLogger.e(TAG, "unSubScribeAlias " + response.toString());
                SubAliasStatus subAliasStatus = new SubAliasStatus(response.toString());
                DebugLogger.e(TAG, "unSubScribeAlias " + subAliasStatus);
                PlatformMessageSender.sendSubAlias(mContext, subAliasStatus);
            }

            @Override
            public void onError(ANError anError) {
                SubAliasStatus subAliasStatus = new SubAliasStatus();
                subAliasStatus.setCode(String.valueOf(anError.getErrorCode()));
                subAliasStatus.setMessage(anError.getErrorBody());
                DebugLogger.e(TAG, "subScribeAlias " + subAliasStatus);
                PlatformMessageSender.sendSubAlias(mContext, subAliasStatus);
            }
        });
    }

    public void checkSubScribeAlias(String appId,String appKey,String pushId){
        pushAPI.checkSubScribeAlias(appId, appKey, pushId, new OkHttpResponseAndStringRequestListener() {
            @Override
            public void onResponse(Response okHttpResponse, String response) {
                DebugLogger.e(TAG, "checkSubScribeAlias " + response.toString());
                SubAliasStatus subAliasStatus = new SubAliasStatus(response.toString());
                DebugLogger.e(TAG, "checkSubScribeAlias " + subAliasStatus);
                PlatformMessageSender.sendSubAlias(mContext, subAliasStatus);
            }

            @Override
            public void onError(ANError anError) {
                SubAliasStatus subAliasStatus = new SubAliasStatus();
                subAliasStatus.setCode(String.valueOf(anError.getErrorCode()));
                subAliasStatus.setMessage(anError.getErrorBody());
                DebugLogger.e(TAG, "subScribeAlias " + subAliasStatus);
                PlatformMessageSender.sendSubAlias(mContext, subAliasStatus);
            }
        });
    }


    /**
     * @param type
     *           register type 0:register 1: unRegister
     * */
    private void executeAfterGetDeviceId(final int type,final String appId, final String appkey,long delay){
        executorService.schedule(new Runnable() {
            @Override
            public void run() {
                String deviceId = MzSystemUtils.getDeviceId(mContext);
                Message message = mainHandler.obtainMessage();
                Bundle bundle = new Bundle();
                bundle.putString("deviceId", deviceId);
                bundle.putString("appId", appId);
                bundle.putString("appKey", appkey);
                message.setData(bundle);
                message.what = type;
                DebugLogger.e(TAG, "deviceId " + deviceId);
                mainHandler.sendMessage(message);
                if (!TextUtils.isEmpty(deviceId)) {
                    DebugLogger.i(TAG, "put deviceId " + deviceId + " to preference");
                    PushPreferencesUtils.putDeviceId(mContext, deviceId);
                }
            }
        }, delay, TimeUnit.SECONDS);
    }

}
