package com.mercury.sqkon.db

import app.cash.sqldelight.Query
import app.cash.sqldelight.SuspendingTransacterImpl
import app.cash.sqldelight.db.QueryResult
import app.cash.sqldelight.db.SqlCursor
import app.cash.sqldelight.db.SqlDriver
import kotlin.Any
import kotlin.Long
import kotlin.String
import kotlin.collections.Collection
import kotlinx.datetime.Instant

public class MetadataQueries(
  driver: SqlDriver,
  private val metadataAdapter: Metadata.Adapter,
) : SuspendingTransacterImpl(driver) {
  public fun <T : Any> selectByEntityName(entity_name: String, mapper: (
    entity_name: String,
    lastReadAt: Instant?,
    lastWriteAt: Instant?,
  ) -> T): Query<T> = SelectByEntityNameQuery(entity_name) { cursor ->
    mapper(
      cursor.getString(0)!!,
      cursor.getLong(1)?.let { metadataAdapter.lastReadAtAdapter.decode(it) },
      cursor.getLong(2)?.let { metadataAdapter.lastWriteAtAdapter.decode(it) }
    )
  }

  public fun selectByEntityName(entity_name: String): Query<Metadata> =
      selectByEntityName(entity_name) { entity_name_, lastReadAt, lastWriteAt ->
    Metadata(
      entity_name_,
      lastReadAt,
      lastWriteAt
    )
  }

  public suspend fun upsertRead(entity_name: String, lastReadAt: Instant?) {
    driver.execute(-243_992_908, """
        |INSERT INTO metadata (entity_name, lastReadAt)
        |    VALUES (?, ?)
        |    ON CONFLICT(entity_name)
        |    DO
        |        UPDATE SET lastReadAt = ?
        |        WHERE entity_name = ?
        """.trimMargin(), 4) {
          val lastReadAt_ = lastReadAt?.let { metadataAdapter.lastReadAtAdapter.encode(it) }
          bindString(0, entity_name)
          bindLong(1, lastReadAt_)
          bindLong(2, lastReadAt_)
          bindString(3, entity_name)
        }.await()
    notifyQueries(-243_992_908) { emit ->
      emit("metadata")
    }
  }

  public suspend fun updateReadForEntities(
    readAt: Long?,
    entity_name: String,
    entity_keys: Collection<String>,
  ) {
    val entity_keysIndexes = createArguments(count = entity_keys.size)
    driver.execute(null, """
        |UPDATE entity SET read_at = ?
        |    WHERE entity_name = ? AND entity_key IN $entity_keysIndexes
        """.trimMargin(), 2 + entity_keys.size) {
          bindLong(0, readAt)
          bindString(1, entity_name)
          entity_keys.forEachIndexed { index, entity_keys_ ->
            bindString(index + 2, entity_keys_)
          }
        }.await()
    notifyQueries(-171_241_092) { emit ->
      emit("entity")
    }
  }

  public suspend fun upsertWrite(entity_name: String, lastWriteAt: Instant?) {
    driver.execute(1_031_167_617, """
        |INSERT INTO metadata (entity_name, lastWriteAt)
        |    VALUES (?, ?)
        |    ON CONFLICT(entity_name)
        |    DO
        |        UPDATE SET lastWriteAt = ?
        |        WHERE entity_name = ?
        """.trimMargin(), 4) {
          val lastWriteAt_ = lastWriteAt?.let { metadataAdapter.lastWriteAtAdapter.encode(it) }
          bindString(0, entity_name)
          bindLong(1, lastWriteAt_)
          bindLong(2, lastWriteAt_)
          bindString(3, entity_name)
        }.await()
    notifyQueries(1_031_167_617) { emit ->
      emit("metadata")
    }
  }

  public suspend fun purgeExpires(entity_name: String, expiresAfter: Long?) {
    driver.execute(-1_558_990_952, """
        |DELETE FROM entity
        |    WHERE entity_name = ? AND expires_at IS NOT NULL AND expires_at < ?
        """.trimMargin(), 2) {
          bindString(0, entity_name)
          bindLong(1, expiresAfter)
        }.await()
    notifyQueries(-1_558_990_952) { emit ->
      emit("entity")
    }
  }

  public suspend fun purgeStale(
    entity_name: String,
    writeInstant: Long,
    readInstant: Long?,
  ) {
    driver.execute(-78_211_843, """
        |DELETE FROM entity
        |    WHERE entity_name = ?
        |    AND write_at < ?
        |    AND (read_at IS NULL OR read_at < ?)
        """.trimMargin(), 3) {
          bindString(0, entity_name)
          bindLong(1, writeInstant)
          bindLong(2, readInstant)
        }.await()
    notifyQueries(-78_211_843) { emit ->
      emit("entity")
    }
  }

  private inner class SelectByEntityNameQuery<out T : Any>(
    public val entity_name: String,
    mapper: (SqlCursor) -> T,
  ) : Query<T>(mapper) {
    override fun addListener(listener: Query.Listener) {
      driver.addListener("metadata", listener = listener)
    }

    override fun removeListener(listener: Query.Listener) {
      driver.removeListener("metadata", listener = listener)
    }

    override fun <R> execute(mapper: (SqlCursor) -> QueryResult<R>): QueryResult<R> =
        driver.executeQuery(-1_552_990_160,
        """SELECT metadata.entity_name, metadata.lastReadAt, metadata.lastWriteAt FROM metadata WHERE entity_name = ?""",
        mapper, 1) {
      bindString(0, entity_name)
    }

    override fun toString(): String = "metadata.sq:selectByEntityName"
  }
}
