package com.mercury.sqkon.db.library

import app.cash.sqldelight.SuspendingTransacterImpl
import app.cash.sqldelight.db.AfterVersion
import app.cash.sqldelight.db.QueryResult
import app.cash.sqldelight.db.SqlDriver
import app.cash.sqldelight.db.SqlSchema
import com.mercury.sqkon.db.Metadata
import com.mercury.sqkon.db.MetadataQueries
import com.mercury.sqkon.db.SqkonDatabase
import kotlin.Long
import kotlin.Unit
import kotlin.reflect.KClass

internal val KClass<SqkonDatabase>.schema: SqlSchema<QueryResult.AsyncValue<Unit>>
  get() = SqkonDatabaseImpl.Schema

internal fun KClass<SqkonDatabase>.newInstance(driver: SqlDriver,
    metadataAdapter: Metadata.Adapter): SqkonDatabase = SqkonDatabaseImpl(driver, metadataAdapter)

private class SqkonDatabaseImpl(
  driver: SqlDriver,
  metadataAdapter: Metadata.Adapter,
) : SuspendingTransacterImpl(driver), SqkonDatabase {
  override val metadataQueries: MetadataQueries = MetadataQueries(driver, metadataAdapter)

  public object Schema : SqlSchema<QueryResult.AsyncValue<Unit>> {
    override val version: Long
      get() = 2

    override fun create(driver: SqlDriver): QueryResult.AsyncValue<Unit> = QueryResult.AsyncValue {
      driver.execute(null, """
          |CREATE TABLE entity (
          |    entity_name TEXT NOT NULL,
          |    entity_key TEXT NOT NULL,
          |    -- UTC timestamp in milliseconds
          |    added_at INTEGER NOT NULL DEFAULT CURRENT_TIMESTAMP,
          |    -- UTC timestamp in milliseconds
          |    updated_at INTEGER NOT NULL DEFAULT CURRENT_TIMESTAMP,
          |    -- UTC timestamp in milliseconds
          |    expires_at INTEGER,
          |    -- JSONB Blob use jsonb_ operators
          |    value BLOB NOT NULL,
          |    -- UTC timestamp in milliseconds
          |    read_at INTEGER,
          |    -- UTC timestamp in milliseconds
          |    write_at INTEGER,
          |    PRIMARY KEY (entity_name, entity_key)
          |)
          """.trimMargin(), 0).await()
      driver.execute(null, """
          |CREATE TABLE metadata (
          |    entity_name TEXT NOT NULL PRIMARY KEY,
          |    lastReadAt INTEGER,
          |    lastWriteAt INTEGER
          |)
          """.trimMargin(), 0).await()
      driver.execute(null, "CREATE INDEX idx_entity_read_at ON entity (read_at)", 0).await()
      driver.execute(null, "CREATE INDEX idx_entity_write_at ON entity (write_at)", 0).await()
      driver.execute(null, "CREATE INDEX idx_entity_expires_at ON entity (expires_at)", 0).await()
    }

    private fun migrateInternal(
      driver: SqlDriver,
      oldVersion: Long,
      newVersion: Long,
    ): QueryResult.AsyncValue<Unit> = QueryResult.AsyncValue {
      if (oldVersion <= 1 && newVersion > 1) {
        driver.execute(null, """
            |CREATE TABLE metadata (
            |    entity_name TEXT NOT NULL PRIMARY KEY,
            |    lastReadAt INTEGER,
            |    lastWriteAt INTEGER
            |)
            """.trimMargin(), 0).await()
        driver.execute(null, "ALTER TABLE entity ADD COLUMN read_at INTEGER", 0).await()
        driver.execute(null, "ALTER TABLE entity ADD COLUMN write_at INTEGER", 0).await()
        driver.execute(null, "UPDATE entity SET write_at = CURRENT_TIMESTAMP", 0).await()
        driver.execute(null, "CREATE INDEX idx_entity_read_at ON entity (read_at)", 0).await()
        driver.execute(null, "CREATE INDEX idx_entity_write_at ON entity (write_at)", 0).await()
        driver.execute(null, "CREATE INDEX idx_entity_expires_at ON entity (expires_at)", 0).await()
      }
    }

    override fun migrate(
      driver: SqlDriver,
      oldVersion: Long,
      newVersion: Long,
      vararg callbacks: AfterVersion,
    ): QueryResult.AsyncValue<Unit> = QueryResult.AsyncValue {
      var lastVersion = oldVersion

      callbacks.filter { it.afterVersion in oldVersion until newVersion }
      .sortedBy { it.afterVersion }
      .forEach { callback ->
        migrateInternal(driver, oldVersion = lastVersion, newVersion = callback.afterVersion +
          1).await()
        callback.block(driver)
        lastVersion = callback.afterVersion + 1
      }

      if (lastVersion < newVersion) {
        migrateInternal(driver, lastVersion, newVersion).await()
      }
    }
  }
}
