/*
 * Decompiled with CFR 0.152.
 */
package com.metaeffekt.mirror.download.advisor;

import com.metaeffekt.artifact.analysis.utils.FileUtils;
import com.metaeffekt.artifact.analysis.utils.StringUtils;
import com.metaeffekt.artifact.analysis.utils.TimeUtils;
import com.metaeffekt.mirror.Mirror;
import com.metaeffekt.mirror.Retry;
import com.metaeffekt.mirror.download.Download;
import com.metaeffekt.mirror.download.ResourceLocation;
import com.metaeffekt.mirror.download.documentation.DocRelevantMethods;
import com.metaeffekt.mirror.download.documentation.MirrorMetadata;
import java.io.File;
import java.io.IOException;
import java.net.URL;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.time.OffsetDateTime;
import java.time.ZoneId;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import org.json.JSONArray;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@MirrorMetadata(directoryName="msrc-security-guide", mavenPropertyName="msrcSecurityGuideDownload")
public class MsrcSecurityGuideDownload
extends Download {
    private static final Logger LOG = LoggerFactory.getLogger(MsrcSecurityGuideDownload.class);
    private static final String START_DATE = "2016-01-01T00:00:00+01:00";
    private static final int WRITE_TO_FILE_THRESHOLD = 20;

    public MsrcSecurityGuideDownload(File baseMirrorDirectory) {
        super(baseMirrorDirectory, MsrcSecurityGuideDownload.class);
    }

    @Override
    @DocRelevantMethods(value={"MsrcSecurityGuideDownload#downloadAllFromTo", "MsrcSecurityGuideDownload#downloadSecurityGuideByDateRangeAndOffset"})
    public void performDownload() {
        String startDate = this.determineStartDate();
        String endDate = this.getCurrentTimestampForRequest();
        this.downloadAllFromTo(startDate, endDate);
    }

    private void downloadAllFromTo(String startDate, String endDate) {
        int offset = 0;
        int totalEntriesCount = 0;
        int currentEntriesCount = 0;
        int currentRequestsSinceLastWriteToFile = 0;
        long startTime = System.currentTimeMillis();
        ConcurrentHashMap<String, List<JSONObject>> entriesToBeWrittenToFile = new ConcurrentHashMap<String, List<JSONObject>>();
        this.executor.setDelay(0L);
        do {
            try {
                JSONObject response = this.downloadSecurityGuideByDateRangeAndOffset(startDate, endDate, offset);
                if (this.isResponseEmpty(response)) {
                    LOG.info("No more entries found. Stopping download.");
                    break;
                }
                totalEntriesCount = response.getInt("@odata.count");
                currentEntriesCount = response.getJSONArray("value").length();
                offset += currentEntriesCount;
                this.sortEntriesIntoYears(response, entriesToBeWrittenToFile);
                if (++currentRequestsSinceLastWriteToFile < 20) continue;
                long currentTime = System.currentTimeMillis();
                long elapsedTime = currentTime - startTime;
                long expectedRemainingTime = elapsedTime / (long)offset * (long)(totalEntriesCount - offset);
                LOG.info("Downloaded [{} / {}] entries, expected time remaining [{}]", new Object[]{offset, totalEntriesCount, TimeUtils.formatTimeDiff(expectedRemainingTime)});
                this.executor.submit(() -> {
                    try {
                        this.writeEntriesToFile(entriesToBeWrittenToFile);
                    }
                    catch (IOException e) {
                        throw new RuntimeException("Failed to write entries to file", e);
                    }
                });
                this.executor.start();
                currentRequestsSinceLastWriteToFile = 0;
            }
            catch (IOException e) {
                LOG.error("Error while downloading security guide.", (Throwable)e);
                break;
            }
        } while (offset < totalEntriesCount);
        try {
            this.executor.join();
        }
        catch (InterruptedException e) {
            throw new RuntimeException("Failed to join executor.", e);
        }
        try {
            this.writeEntriesToFile(entriesToBeWrittenToFile);
        }
        catch (IOException e) {
            LOG.error("Error while writing entries to file.", (Throwable)e);
        }
        try {
            this.setPreviousTotalEntriesCount(this.getTotalAvailableEntriesCount());
        }
        catch (IOException e) {
            this.setPreviousTotalEntriesCount(totalEntriesCount);
        }
        this.setLastMirrorDate(endDate);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void writeEntriesToFile(Map<String, List<JSONObject>> entriesToBeWrittenToFile) throws IOException {
        HashMap<String, List<JSONObject>> entriesToBeWrittenToFileCopy;
        Map<String, List<JSONObject>> map = entriesToBeWrittenToFile;
        synchronized (map) {
            if (entriesToBeWrittenToFile.isEmpty()) {
                return;
            }
            entriesToBeWrittenToFileCopy = new HashMap<String, List<JSONObject>>(entriesToBeWrittenToFile);
            entriesToBeWrittenToFile.clear();
        }
        for (Map.Entry entry : entriesToBeWrittenToFileCopy.entrySet()) {
            String year = (String)entry.getKey();
            List entries = (List)entry.getValue();
            File yearFile = new File(this.downloadIntoDirectory, year + ".json");
            JSONArray yearFileContent = yearFile.exists() ? new JSONArray(FileUtils.readFileToString((File)yearFile, (Charset)StandardCharsets.UTF_8)) : new JSONArray();
            for (JSONObject newEntry : entries) {
                String newEntryId = newEntry.getString("id");
                boolean entryAlreadyExists = false;
                for (int i = 0; i < yearFileContent.length(); ++i) {
                    JSONObject existingEntry = yearFileContent.getJSONObject(i);
                    String existingEntryId = existingEntry.getString("id");
                    if (!newEntryId.equals(existingEntryId)) continue;
                    entryAlreadyExists = true;
                    break;
                }
                if (entryAlreadyExists) continue;
                yearFileContent.put((Object)newEntry);
            }
            FileUtils.writeStringToFile((File)yearFile, (String)yearFileContent.toString(), (Charset)StandardCharsets.UTF_8);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void sortEntriesIntoYears(JSONObject response, Map<String, List<JSONObject>> entriesByYear) {
        JSONArray entries = response.getJSONArray("value");
        Map<String, List<JSONObject>> map = entriesByYear;
        synchronized (map) {
            for (int i = 0; i < entries.length(); ++i) {
                JSONObject entry = entries.getJSONObject(i);
                String releaseDate = entry.getString("releaseDate");
                String year = releaseDate.substring(0, 4);
                entriesByYear.computeIfAbsent(year, k -> new ArrayList()).add(entry);
            }
        }
    }

    private String determineStartDate() {
        String lastMirrorDate = this.getLastMirrorDate();
        String startDate = StringUtils.hasText(lastMirrorDate) ? lastMirrorDate : START_DATE;
        return startDate;
    }

    private JSONObject downloadSecurityGuideByDateRangeAndOffset(String startDate, String endDate, int offset) throws IOException {
        return new Retry<JSONObject>(() -> {
            HashMap<String, String> getParameters = new HashMap<String, String>();
            getParameters.put("$orderBy", "releaseDate asc");
            getParameters.put("$filter", "(releaseDate gt " + startDate + ") and (releaseDate lt " + endDate + ")");
            getParameters.put("$skip", String.valueOf(offset));
            String getRequestUrl = this.downloader.buildGetRequest(this.getRemoteResourceLocation(ResourceLocationMsrcSecurityUpdateGuide.AFFECTED_PRODUCTS_BASE_URL), getParameters);
            List<String> response = this.downloader.fetchResponseBodyFromUrlAsList(new URL(getRequestUrl));
            return new JSONObject(String.join((CharSequence)"", response));
        }).onException(Throwable.class).withValidator(json -> {
            if (json.has("error")) {
                JSONObject error = json.getJSONObject("error");
                String code = error.getString("code");
                String message = error.getString("message");
                LOG.error("Error while downloading security guide: [{}] {}", (Object)code, (Object)message);
                return false;
            }
            return true;
        }).retryCount(5).withDelay(5000).run();
    }

    private boolean isResponseEmpty(JSONObject response) {
        return response.getJSONArray("value").isEmpty();
    }

    private String getCurrentTimestampForRequest() {
        OffsetDateTime now = OffsetDateTime.now(ZoneId.of("Europe/Paris"));
        DateTimeFormatter formatter = DateTimeFormatter.ofPattern("yyyy-MM-dd'T'HH:mm:ssXXX");
        return now.format(formatter);
    }

    private void setPreviousTotalEntriesCount(int count) {
        this.propertyFiles.set(this.downloadIntoDirectory, "info", Mirror.InfoFileAttributes.MSRC_PREFIX.getKey() + "entries-count", String.valueOf(count));
    }

    private int getPreviousTotalEntriesCount() {
        return Integer.parseInt(this.propertyFiles.getString(this.downloadIntoDirectory, "info", Mirror.InfoFileAttributes.MSRC_PREFIX.getKey() + "entries-count").orElse("0"));
    }

    private void setLastMirrorDate(String date) {
        this.propertyFiles.set(this.downloadIntoDirectory, "info", Mirror.InfoFileAttributes.MSRC_PREFIX.getKey() + "last-mirror-date", date);
    }

    private String getLastMirrorDate() {
        return this.propertyFiles.getString(this.downloadIntoDirectory, "info", Mirror.InfoFileAttributes.MSRC_PREFIX.getKey() + "last-mirror-date").orElse("");
    }

    private int getTotalAvailableEntriesCount() throws IOException {
        JSONObject response = this.downloadSecurityGuideByDateRangeAndOffset(START_DATE, this.getCurrentTimestampForRequest(), 999999999);
        return response.getInt("@odata.count");
    }

    @Override
    protected boolean isResetRequired() {
        return false;
    }

    @Override
    protected boolean additionalIsDownloadRequired() {
        try {
            int totalEntriesCount = this.getTotalAvailableEntriesCount();
            int previousTotalEntriesCount = this.getPreviousTotalEntriesCount();
            if (previousTotalEntriesCount != totalEntriesCount) {
                LOG.info("Total entries count changed from [{}] to [{}], download is required", (Object)previousTotalEntriesCount, (Object)totalEntriesCount);
                return true;
            }
            return false;
        }
        catch (IOException e) {
            return true;
        }
    }

    @Override
    public void setRemoteResourceLocation(String location, String url) {
        super.setRemoteResourceLocation(ResourceLocationMsrcSecurityUpdateGuide.valueOf(location), url);
    }

    public static enum ResourceLocationMsrcSecurityUpdateGuide implements ResourceLocation
    {
        AFFECTED_PRODUCTS_BASE_URL("https://api.msrc.microsoft.com/sug/v2.0/en-US/affectedProduct");

        private final String defaultValue;

        private ResourceLocationMsrcSecurityUpdateGuide(String defaultValue) {
            this.defaultValue = defaultValue;
        }

        @Override
        public String getDefault() {
            return this.defaultValue;
        }
    }
}

