/*
 * Decompiled with CFR 0.152.
 */
package com.metaeffekt.mirror.concurrency;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.CancellationException;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import org.apache.commons.lang3.tuple.Pair;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ScheduledDelayedThreadPoolExecutor {
    private static final Logger LOG = LoggerFactory.getLogger(ScheduledDelayedThreadPoolExecutor.class);
    protected static final int LOG_PROGRESS_EVERY_PERCENT = 25;
    private ScheduledThreadPoolExecutor executor;
    private final BlockingQueue<ThrowingRunnable> backlog;
    private final AtomicInteger remainingScheduledTasks = new AtomicInteger(0);
    private final AtomicInteger maxTaskListSize = new AtomicInteger(0);
    private final AtomicLong batchStartTime = new AtomicLong();
    private final AtomicLong currentBatchEndTime = new AtomicLong();
    private final AtomicLong lastLoggedProgressPercent = new AtomicLong();
    private final List<Pair<Integer, Throwable>> exceptions = Collections.synchronizedList(new ArrayList());
    private long delay;
    private boolean logProgress = true;
    private int logEveryPercent = 25;

    public ScheduledDelayedThreadPoolExecutor(int size, long delay) {
        this.executor = this.createExecutor(size);
        this.backlog = new LinkedBlockingQueue<ThrowingRunnable>();
        this.setDelay(delay);
        this.currentBatchEndTime.set(System.currentTimeMillis());
    }

    public void setSize(int size) {
        if (size <= 0) {
            throw new IllegalArgumentException("Size must be greater than 0");
        }
        int maxThreads = Math.max(1, Math.min(size, Runtime.getRuntime().availableProcessors() - 1));
        this.executor.setCorePoolSize(maxThreads);
    }

    public void setDelay(long delay) {
        if (delay < 0L) {
            throw new IllegalArgumentException("Delay must be non-negative");
        }
        this.delay = delay;
    }

    public void setLogProgress(boolean logProgress) {
        this.logProgress = logProgress;
    }

    public void setLogEveryPercent(int logEveryPercent) {
        this.logEveryPercent = logEveryPercent;
    }

    public void submit(ThrowingRunnable task) {
        if (task == null) {
            throw new IllegalArgumentException("Task must not be null");
        }
        if (task instanceof Thread) {
            LOG.warn("Thread [{}] submitted to executor. Use Runnable instead", (Object)((Thread)((Object)task)).getName());
        }
        this.backlog.offer(task);
    }

    public void start() {
        long startTime = System.currentTimeMillis();
        this.batchStartTime.set(startTime);
        long timeRemainingUntilDone = this.currentBatchEndTime.get() - startTime;
        long delay = timeRemainingUntilDone > 0L ? timeRemainingUntilDone + this.delay : this.delay;
        while (!this.backlog.isEmpty()) {
            this.executor.schedule((Runnable)this.backlog.poll(), delay, TimeUnit.MILLISECONDS);
            delay += this.delay;
            this.remainingScheduledTasks.incrementAndGet();
        }
        this.currentBatchEndTime.set(System.currentTimeMillis() + delay - this.delay);
        this.maxTaskListSize.set(Math.max(this.maxTaskListSize.get(), this.remainingScheduledTasks.get()));
    }

    public void join() throws InterruptedException {
        this.executor.shutdown();
        while (!this.executor.awaitTermination(5L, TimeUnit.MINUTES)) {
            LOG.info("Waiting for executor to terminate");
        }
        int corePoolSize = this.executor.getCorePoolSize();
        this.executor = this.createExecutor(corePoolSize);
        this.remainingScheduledTasks.set(0);
        this.maxTaskListSize.set(0);
        this.lastLoggedProgressPercent.set(0L);
        this.currentBatchEndTime.set(System.currentTimeMillis());
        LOG.info("Executor terminated");
        if (!this.exceptions.isEmpty()) {
            throw new RuntimeException(this.exceptions.size() + " exceptions occurred during execution. See log for details.");
        }
    }

    public boolean isRunning() {
        if (this.executor.isTerminated()) {
            return false;
        }
        return this.remainingScheduledTasks.get() > 0;
    }

    private int calculateRemainingPercent() {
        double remainingTasks = this.remainingScheduledTasks.get();
        double maxTasks = this.maxTaskListSize.get();
        if (maxTasks == 0.0) {
            return 0;
        }
        return (int)(100.0 - remainingTasks / (maxTasks / 100.0));
    }

    private long calculateEstimatedTimeRemaining() {
        long completedTasks = this.maxTaskListSize.get() - this.remainingScheduledTasks.get();
        if (completedTasks == 0L) {
            return 0L;
        }
        long elapsedTime = System.currentTimeMillis() - this.batchStartTime.get();
        long averageTaskTime = elapsedTime / completedTasks;
        return averageTaskTime * (long)this.remainingScheduledTasks.get();
    }

    private boolean shouldProgressBeLogged() {
        if (!this.logProgress) {
            return false;
        }
        long lastLoggedProgress = this.lastLoggedProgressPercent.get();
        long currentProgress = this.calculateRemainingPercent();
        if (currentProgress == 0L) {
            return false;
        }
        if (currentProgress - lastLoggedProgress >= (long)this.logEveryPercent) {
            this.lastLoggedProgressPercent.set(currentProgress);
            return true;
        }
        return false;
    }

    private ScheduledThreadPoolExecutor createExecutor(int size) {
        return new ScheduledThreadPoolExecutor(size){

            @Override
            protected void afterExecute(Runnable r, Throwable t) {
                super.afterExecute(r, t);
                ScheduledDelayedThreadPoolExecutor.this.remainingScheduledTasks.decrementAndGet();
                if (ScheduledDelayedThreadPoolExecutor.this.shouldProgressBeLogged()) {
                    int remainingPercent = ScheduledDelayedThreadPoolExecutor.this.calculateRemainingPercent();
                    long estimatedTimeRemaining = ScheduledDelayedThreadPoolExecutor.this.calculateEstimatedTimeRemaining();
                    long estimatedTimeRemainingSecs = TimeUnit.MILLISECONDS.toSeconds(estimatedTimeRemaining);
                    LOG.info("Started [{} / {}] tasks [{} %]", new Object[]{ScheduledDelayedThreadPoolExecutor.this.maxTaskListSize.get() - ScheduledDelayedThreadPoolExecutor.this.remainingScheduledTasks.get(), ScheduledDelayedThreadPoolExecutor.this.maxTaskListSize.get(), remainingPercent});
                }
                if (t == null && r instanceof Future) {
                    try {
                        Future future = (Future)((Object)r);
                        if (future.isDone()) {
                            future.get();
                        }
                    }
                    catch (CancellationException ce) {
                        t = ce;
                    }
                    catch (ExecutionException ee) {
                        t = ee.getCause();
                    }
                    catch (InterruptedException ie) {
                        Thread.currentThread().interrupt();
                    }
                }
                if (t != null) {
                    if (t instanceof ThrowingRunnableException) {
                        t = t.getCause();
                    }
                    ScheduledDelayedThreadPoolExecutor.this.exceptions.add(Pair.of((Object)(ScheduledDelayedThreadPoolExecutor.this.maxTaskListSize.get() - ScheduledDelayedThreadPoolExecutor.this.remainingScheduledTasks.get()), (Object)t));
                    throw new RuntimeException("Exception in scheduled task at ~" + (ScheduledDelayedThreadPoolExecutor.this.maxTaskListSize.get() - ScheduledDelayedThreadPoolExecutor.this.remainingScheduledTasks.get()), t);
                }
            }
        };
    }

    public static interface ThrowingRunnable
    extends Runnable {
        @Override
        default public void run() {
            try {
                this.runThrows();
            }
            catch (Exception e) {
                throw new ThrowingRunnableException(e);
            }
        }

        public void runThrows() throws Exception;
    }

    private static class ThrowingRunnableException
    extends RuntimeException {
        public ThrowingRunnableException(String message) {
            super(message);
        }

        public ThrowingRunnableException(String message, Throwable cause) {
            super(message, cause);
        }

        public ThrowingRunnableException(Throwable cause) {
            super(cause);
        }
    }
}

