/*
 * Decompiled with CFR 0.152.
 */
package com.metaeffekt.mirror.download.nvd;

import com.metaeffekt.artifact.analysis.utils.FileUtils;
import com.metaeffekt.artifact.analysis.utils.TimeUtils;
import com.metaeffekt.mirror.Retry;
import com.metaeffekt.mirror.download.Download;
import com.metaeffekt.mirror.download.ResourceLocation;
import com.metaeffekt.mirror.download.documentation.MirrorMetadata;
import java.io.File;
import java.io.IOException;
import java.net.URL;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Calendar;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.concurrent.atomic.AtomicReference;
import java.util.stream.Collectors;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@MirrorMetadata(directoryName="nvd", mavenPropertyName="nvdCveDownload")
public class NvdCveApiDownload
extends Download {
    private static final Logger LOG = LoggerFactory.getLogger(NvdCveApiDownload.class);
    private static final SimpleDateFormat ISO_8601_DATE_FORMAT = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss'Z'");
    private static final int API_DELAY_BETWEEN_AUTHORIZED_REQUESTS = 600;
    private static final int API_DELAY_BETWEEN_UNAUTHORIZED_REQUESTS = 6400;
    private String apiKey;

    public NvdCveApiDownload(File baseMirrorDirectory) {
        super(baseMirrorDirectory, NvdCveApiDownload.class);
    }

    public NvdCveApiDownload setApiKey(String apiKey) {
        this.apiKey = apiKey;
        return this;
    }

    @Override
    protected void performDownload() {
        boolean fullMirrorRequired = this.isFullMirrorRequired();
        if (fullMirrorRequired) {
            this.clearDownload();
            LOG.info("Downloading initial NVD data from NVD API");
        } else {
            LOG.info("Existing mirror detected. Downloading incremental NVD data from NVD API");
        }
        LOG.info("Requests {} be authorized with an API key, delay between requests [{}]", (Object)(this.apiKey == null ? "will not" : "will"), (Object)TimeUtils.formatTimeDiff(this.apiKey == null ? 6400L : 600L));
        long lastModified = this.getDownloadDirectoryLastModified();
        Date lastModifiedDate = new Date(lastModified);
        Date now = new Date(TimeUtils.utcNow());
        int currentStartIndex = 0;
        ArrayList<JSONArray> apiResponseDataToBeProcessed = new ArrayList<JSONArray>();
        while (true) {
            long processingStartTime = System.currentTimeMillis();
            int finalCurrentStartIndex = currentStartIndex;
            AtomicReference json = new AtomicReference();
            new Retry(() -> {
                if (fullMirrorRequired) {
                    json.set(this.downloadCvePage(finalCurrentStartIndex));
                } else {
                    json.set(this.downloadCvePage(finalCurrentStartIndex, lastModifiedDate, now));
                }
            }).withDelay(6400).onException(Exception.class).retryCount(8).run();
            int totalResults = ((JSONObject)json.get()).getInt("totalResults");
            int resultsPerPage = ((JSONObject)json.get()).getInt("resultsPerPage");
            currentStartIndex += resultsPerPage;
            if (totalResults == 0) {
                LOG.info("No CVEs found for the given range.");
                break;
            }
            LOG.info("Downloaded CVEs [{}] to [{}] of [{}] [{} %]", new Object[]{currentStartIndex - resultsPerPage, currentStartIndex, totalResults, currentStartIndex * 100 / totalResults});
            JSONArray vulnerabilities = ((JSONObject)json.get()).getJSONArray("vulnerabilities");
            apiResponseDataToBeProcessed.add(vulnerabilities);
            if (apiResponseDataToBeProcessed.size() >= 10) {
                JSONArray merged = this.mergeCveItems(apiResponseDataToBeProcessed);
                this.processApiCveItems(merged);
                apiResponseDataToBeProcessed.clear();
            }
            long processingDuration = System.currentTimeMillis() - processingStartTime;
            if (currentStartIndex >= totalResults) break;
            try {
                long sleepDuration = (long)(this.apiKey == null ? 6400 : 600) - processingDuration;
                if (sleepDuration <= 0L) continue;
                LOG.info("Sleeping for [{}] to avoid rate limit", (Object)TimeUtils.formatTimeDiff(sleepDuration));
                Thread.sleep(sleepDuration);
            }
            catch (InterruptedException interruptedException) {}
        }
        if (apiResponseDataToBeProcessed.size() > 0) {
            JSONArray merged = this.mergeCveItems(apiResponseDataToBeProcessed);
            this.processApiCveItems(merged);
        }
        LOG.info("Finished processing all CVEs");
    }

    private void processApiCveItems(JSONArray jsonArray) {
        Map<Integer, JSONArray> yearCves = this.sortCvesIntoYears(jsonArray);
        LOG.info("Processing CVE data from years: [{}]", (Object)yearCves.entrySet().stream().sorted(Map.Entry.comparingByValue(Comparator.comparing(JSONArray::length).reversed())).map(e -> e.getKey() + " = " + ((JSONArray)e.getValue()).length()).collect(Collectors.joining("; ")));
        for (Map.Entry<Integer, JSONArray> entry : yearCves.entrySet()) {
            int year = entry.getKey();
            JSONArray cves = entry.getValue();
            JSONArray existingJson = this.parseCveItemsFromDownloadedYear(year);
            JSONArray mergedJson = this.mergeCveItems(Arrays.asList(cves, existingJson));
            File cveFile = new File(this.downloadIntoDirectory, year + ".json");
            try {
                FileUtils.write((File)cveFile, (CharSequence)mergedJson.toString(), (Charset)StandardCharsets.UTF_8);
            }
            catch (IOException e2) {
                throw new RuntimeException("Unable to write NVD CVE year file " + cveFile.getAbsolutePath(), e2);
            }
        }
    }

    private JSONArray mergeCveItems(Collection<JSONArray> cves) {
        JSONArray mergedJson = new JSONArray();
        HashSet<String> knownIds = new HashSet<String>();
        for (JSONArray array : cves) {
            for (int i = 0; i < array.length(); ++i) {
                JSONObject cve;
                JSONObject jSONObject = cve = array.getJSONObject(i).has("cve") ? array.getJSONObject(i).getJSONObject("cve") : array.getJSONObject(i);
                if (!knownIds.add(cve.getString("id"))) continue;
                mergedJson.put((Object)cve);
            }
        }
        return mergedJson;
    }

    private Map<Integer, JSONArray> sortCvesIntoYears(JSONArray jsonArray) {
        HashMap<Integer, JSONArray> yearCves = new HashMap<Integer, JSONArray>();
        for (int i = 0; i < jsonArray.length(); ++i) {
            JSONObject jsonObject = jsonArray.getJSONObject(i);
            JSONObject cve = jsonObject.has("cve") ? jsonObject.getJSONObject("cve") : jsonObject;
            String cveId = cve.getString("id");
            int year = Integer.parseInt(cveId.substring(4, 8));
            yearCves.computeIfAbsent(year, k -> new JSONArray()).put((Object)jsonObject);
        }
        return yearCves;
    }

    private JSONArray parseCveItemsFromDownloadedYear(int year) {
        File cveFile = new File(this.downloadIntoDirectory, year + ".json");
        if (!cveFile.exists()) {
            return new JSONArray();
        }
        try {
            String content = FileUtils.readFileToString((File)cveFile, (Charset)StandardCharsets.UTF_8);
            return new JSONArray(content);
        }
        catch (IOException e) {
            throw new RuntimeException("Unable to read NVD CVE year file " + cveFile.getAbsolutePath(), e);
        }
    }

    private JSONObject downloadCvePage(int offset) {
        URL pageUrl = super.getRemoteResourceLocationUrl(ResourceLocationNvd.CVE_API_LIST_ALL, offset);
        List<String> response = this.downloader.fetchResponseBodyFromUrlAsList(pageUrl, Collections.singletonMap("apiKey", this.apiKey));
        try {
            return new JSONObject(String.join((CharSequence)"", response));
        }
        catch (JSONException e) {
            throw new RuntimeException("Unable to parse NVD CVE API response: " + response + "\nRequest URL: " + pageUrl, e);
        }
    }

    private JSONObject downloadCvePage(int offset, Date lastModStartDate, Date lastModEndDate) {
        long diff = lastModEndDate.getTime() - lastModStartDate.getTime();
        if (diff > 10368000000L) {
            throw new IllegalArgumentException("Difference between lastModStartDate and lastModEndDate must not be greater than 120 days");
        }
        String startDate = ISO_8601_DATE_FORMAT.format(lastModStartDate);
        String endDate = ISO_8601_DATE_FORMAT.format(lastModEndDate);
        URL pageUrl = super.getRemoteResourceLocationUrl(ResourceLocationNvd.CVE_API_START_END_DATE, offset, startDate, endDate);
        List<String> response = this.downloader.fetchResponseBodyFromUrlAsList(pageUrl, Collections.singletonMap("apiKey", this.apiKey));
        try {
            return new JSONObject(String.join((CharSequence)"", response));
        }
        catch (JSONException e) {
            throw new RuntimeException("Unable to parse NVD CVE API response: " + response + "\nRequest URL: " + pageUrl, e);
        }
    }

    private boolean isFullMirrorRequired() {
        File[] downloadFiles = this.downloadIntoDirectory.listFiles();
        if (downloadFiles == null) {
            LOG.info("No CVE JSON files found in download directory, performing full mirror");
            return true;
        }
        List files = Arrays.stream(downloadFiles).map(File::getName).collect(Collectors.toList());
        if (files.stream().noneMatch(file -> file.endsWith(".json"))) {
            LOG.info("No CVE JSON files found in download directory, performing full mirror");
            return true;
        }
        int latestCheckYear = Calendar.getInstance().get(1) - 1;
        for (int year = 1999; year <= latestCheckYear; ++year) {
            if (files.contains(year + ".json")) continue;
            LOG.info("Missing CVE JSON file for year [{}], performing full mirror", (Object)year);
            return true;
        }
        long days120 = 10368000000L;
        long directoryLastModified = this.getDownloadDirectoryLastModified();
        if (super.isUpdatedAgeOlderThan(directoryLastModified, days120)) {
            LOG.info("Download directory last modified date is older than 120 days, performing full mirror");
            return true;
        }
        return false;
    }

    @Override
    protected boolean additionalIsDownloadRequired() {
        Date now;
        if (this.isFullMirrorRequired()) {
            return true;
        }
        long lastModified = this.getDownloadDirectoryLastModified();
        Date lastModifiedDate = new Date(lastModified);
        JSONObject changes = this.downloadCvePage(0, lastModifiedDate, now = new Date(TimeUtils.utcNow()));
        if (changes.has("totalResults") && changes.getInt("totalResults") > 0) {
            LOG.info("NVD CVE API reports [{}] new/changed CVEs since last download", (Object)changes.getInt("totalResults"));
            return true;
        }
        return false;
    }

    @Override
    public void setRemoteResourceLocation(String location, String url) {
        super.setRemoteResourceLocation(ResourceLocationNvd.valueOf(location), url);
    }

    public static enum ResourceLocationNvd implements ResourceLocation
    {
        CVE_API_LIST_ALL("https://services.nvd.nist.gov/rest/json/cves/2.0?startIndex=%d"),
        CVE_API_START_END_DATE("https://services.nvd.nist.gov/rest/json/cves/2.0/?startIndex=%d&lastModStartDate=%s&lastModEndDate=%s");

        private final String defaultValue;

        private ResourceLocationNvd(String defaultValue) {
            this.defaultValue = defaultValue;
        }

        @Override
        public String getDefault() {
            return this.defaultValue;
        }
    }
}

