/*
 * Copyright 2021-2024 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.metaeffekt.artifact.analysis.flow;

import com.metaeffekt.artifact.analysis.utils.InventoryUtils;
import com.metaeffekt.artifact.analysis.utils.StringUtils;
import com.metaeffekt.artifact.analysis.workbench.PackageLicenseTransformer;
import com.metaeffekt.resource.InventoryResource;
import org.metaeffekt.core.inventory.processor.model.Artifact;
import org.metaeffekt.core.inventory.processor.model.Constants;
import org.metaeffekt.core.inventory.processor.model.LicenseData;
import org.metaeffekt.core.inventory.processor.model.ReportData;

import java.io.IOException;
import java.util.*;

import static com.metaeffekt.artifact.analysis.metascan.Constants.*;

public class AttachReportFlow {

    public static final String KEY_CATEGORY_USE_NOT_POSSIBLE = "Keine Nutzung möglich";
    public static final String KEY_CATEGORY_USEABLE_WITH_ADDITIONAL_SERVICES = "Nutzung nur mit zusätzlichen Services";
    public static final String KEY_CATEGORY_USEABLE_WITH_MAJOR_OBLIGATIONS = "Nutzung mit Berücksichtigung wesentlicher Einschränkungen";
    public static final String KEY_CATEGORY_USEABLE_WITH_MINOR_OBLIGATIONS = "Nutzung mit Berücksichtigung unwesentlicher Einschränkungen";
    public static final String KEY_LICENSE_COMMENTS = "Kommentare";
    public static final String KEY_ASSESSMENT_PENDING = "Assessment Pending";


    public void attachReport(InventoryResource resource) throws IOException {

        // clear previous report items
        resource.getInventory().getReportData().clear();

        // do it
        int[] id = new int[] {1};
        final Map<String, ReportData> licenceToReportDataMap = new HashMap<>();
        final Map<String, ReportData> commercialLicenceToReportDataMap = new HashMap<>();
        for (Artifact artifact : resource.getInventory().getArtifacts()) {

            reportArtifactsWithoutLicense(resource, artifact, id);
            reportArtifactsWithUnknownLicense(resource, artifact, id);
            reportArtifactsWithKnownButNotAssessedLicense(resource, artifact, id);

            reportArtifactsWithLicenseInCategory(resource, artifact, id, KEY_CATEGORY_USE_NOT_POSSIBLE, "CRITICAL", "ACTIVE", licenceToReportDataMap);
            reportArtifactsWithLicenseInCategory(resource, artifact, id, KEY_CATEGORY_USEABLE_WITH_ADDITIONAL_SERVICES, "HIGH", "ACTIVE", licenceToReportDataMap);
            reportArtifactsWithLicenseInCategory(resource, artifact, id, KEY_CATEGORY_USEABLE_WITH_MAJOR_OBLIGATIONS, "INFO", "PASSIVE", licenceToReportDataMap);

            reportArtifactsWithLicenseInCategory(resource, artifact, id, "Commercial", "CRITICAL", "ACTIVE", commercialLicenceToReportDataMap);

            reportArtifactsWithInsufficientLicenseStatus(resource, artifact, id);

            // FIXME: sync marker names; start maintaining canonicalNameHistory for markers; last verified 28.02.2024
            reportMarker(resource, id, artifact,"Incomplete Match Marker", "ACTIVE", "CRITICAL");
            reportMarker(resource, id, artifact,"Not for Redistribution Marker", "ACTIVE", "CRITICAL");
            reportMarker(resource, id, artifact,"Non-commercial Marker", "ACTIVE", "CRITICAL");
            reportMarker(resource, id, artifact,"Commercial Marker", "ACTIVE", "CRITICAL");
            reportMarker(resource, id, artifact,"Written Permission Required Marker", "ACTIVE", "CRITICAL");

            reportMarker(resource, id, artifact,"Proprietary Marker", "EXPLORE", "CRITICAL");

            reportMarker(resource, id, artifact,"Display Obligation Marker", "ACTIVE", "HIGH");
            reportMarker(resource, id, artifact,"EULA Marker", "ACTIVE", "HIGH");
            reportMarker(resource, id, artifact,"Printed Document Attribution Marker", "ACTIVE", "HIGH");

            reportMarker(resource, id, artifact,"Do Not Promote Marker", "PASSIVE", "MEDIUM");

            reportMarker(resource, id, artifact,"Trademark Marker", "PASSIVE", "INFO");
            reportMarker(resource, id, artifact,"Secondary License Marker", "PASSIVE", "INFO");
            reportMarker(resource, id, artifact,"Changed License Marker", "PASSIVE", "INFO");
            reportMarker(resource, id, artifact,"Commercial Trademark Restriction Marker", "PASSIVE", "INFO");
            reportMarker(resource, id, artifact,"Facebook Patent Grant", "PASSIVE", "INFO");
            reportMarker(resource, id, artifact,"Facebook Patent Rights Grant 2.0", "PASSIVE", "INFO");
            reportMarker(resource, id, artifact,"Licensing Option", "PASSIVE", "INFO");
            reportMarker(resource, id, artifact,"Licensing Option Marker", "PASSIVE", "INFO");
            reportMarker(resource, id, artifact,"Patent Information Marker", "PASSIVE", "INFO");

            // may ease other issues
            reportMarker(resource, id, artifact,"No Eligible Content Marker", "PASSIVE", "INFO");
            reportMarker(resource, id, artifact,"Public Domain Fallback Marker", "PASSIVE", "INFO");
            reportMarker(resource, id, artifact,"Pending License Marker", "PASSIVE", "INFO");
        }
    }

    private void reportArtifactsWithInsufficientLicenseStatus(InventoryResource inventoryResource, Artifact artifact, int[] counter) {
        // NOTE: check is only performed for putative components
        final boolean isComponent = isComponent(artifact);
        if (isComponent) {
            // prevent licenses are reported multiple time in different contexts
            final Set<String> reportedLicenses = new HashSet<>();

            // report on associated/effective licenses if avaialable
            final List<String> licenses = artifact.getLicenses();
            if (!licenses.isEmpty()) {
                reportArtifactsWithInsufficientLicenseStatusInContext(inventoryResource, artifact, counter,
                        licenses, "associated", reportedLicenses);
                reportArtifactsWithInsufficientLicenseStatusInContext(inventoryResource, artifact, counter,
                        inventoryResource.getInventory().getEffectiveLicenses(artifact),
                        "effective", reportedLicenses);
            } else {
                reportArtifactsWithInsufficientLicenseStatusInContext(inventoryResource, artifact, counter,
                        getLicenses(artifact, KEY_DERIVED_LICENSES),
                        "derived/observed", reportedLicenses);

                reportArtifactsWithInsufficientLicenseStatusInContext(inventoryResource, artifact, counter,
                        getLicenses(artifact, KEY_BINARY_ARTIFACT_DERIVED_LICENSES),
                        "derived/binary", reportedLicenses);

                reportArtifactsWithInsufficientLicenseStatusInContext(inventoryResource, artifact, counter,
                        getLicenses(artifact, KEY_SOURCE_ARTIFACT_DERIVED_LICENSES),
                        "derived/source", reportedLicenses);

                reportArtifactsWithInsufficientLicenseStatusInContext(inventoryResource, artifact, counter,
                        getLicenses(artifact, KEY_SOURCE_ARCHIVE_DERIVED_LICENSES),
                        "derived/source-archive", reportedLicenses);

                reportArtifactsWithInsufficientLicenseStatusInContext(inventoryResource, artifact, counter,
                        getLicenses(artifact, KEY_DESCRIPTOR_DERIVED_LICENSES),
                        "derived/pom", reportedLicenses);

                reportArtifactsWithInsufficientLicenseStatusInContext(inventoryResource, artifact, counter,
                        getLicenses(artifact, PackageLicenseTransformer.PACKAGE_SPECIFIED_LICENSE_MAPPED),
                        "derived/package", reportedLicenses);
            }
        }

    }

    private static void reportMarker(InventoryResource inventoryResource, int[] counter, Artifact artifact, String problematicMarker, String policy, String severity) {
        final String markers = artifact.get(KEY_DERIVED_MARKERS);
        if (!StringUtils.isEmpty(markers)) {
            List<String> markerList = InventoryUtils.tokenizeLicense(markers, false, true);
            for (String marker : markerList) {
                if (marker.equals(problematicMarker)) {
                    reportIssueItem(inventoryResource, artifact, counter,
                            "Detected marker [" + problematicMarker + "]", "MARKER",
                            severity, policy, problematicMarker);
                }
            }
        }
    }

    private static void reportArtifactsWithoutLicense(InventoryResource inventoryResource, Artifact artifact, int[] counter) {
        if (isComponent(artifact)) {
            boolean hasDetectedLicenses = false;
            hasDetectedLicenses |= StringUtils.hasText(artifact.getLicense());
            hasDetectedLicenses |= StringUtils.hasText(artifact.get(KEY_DERIVED_LICENSES));
            hasDetectedLicenses |= StringUtils.hasText(artifact.get(KEY_BINARY_ARTIFACT_DERIVED_LICENSES));
            hasDetectedLicenses |= StringUtils.hasText(artifact.get(KEY_SOURCE_ARTIFACT_DERIVED_LICENSES));
            hasDetectedLicenses |= StringUtils.hasText(artifact.get(KEY_SOURCE_ARCHIVE_DERIVED_LICENSES));
            hasDetectedLicenses |= StringUtils.hasText(artifact.get(KEY_DERIVED_LICENSES));

            hasDetectedLicenses |= StringUtils.hasText(artifact.get(PackageLicenseTransformer.PACKAGE_SPECIFIED_LICENSE_MAPPED));
            hasDetectedLicenses |= StringUtils.hasText(artifact.get(PackageLicenseTransformer.COMPONENT_SPECIFIED_LICENSE));

            if (!hasDetectedLicenses) {
                reportIssueItem(inventoryResource, artifact, counter,
                        "No license identified for component.",
                        "NO-LICENSE", "HIGH", "ACTIVE", null);
            }
        }
    }

    private static void reportArtifactsWithUnknownLicense(InventoryResource inventoryResource, Artifact artifact, int[] counter) {
        // NOTE: check is only performed for putative components
        final boolean isComponent = isComponent(artifact);
        if (isComponent) {
            // prevent licenses are reported multiple time in different contexts
            final Set<String> reportedLicenses = new HashSet<>();

            // report on associated/effective licenses if avaialable
            final List<String> licenses = artifact.getLicenses();
            if (!licenses.isEmpty()) {
                reportUnknownLicenseInContext(inventoryResource, artifact, counter,
                        licenses, "associated", reportedLicenses);
                reportUnknownLicenseInContext(inventoryResource, artifact, counter,
                        inventoryResource.getInventory().getEffectiveLicenses(artifact), "effective", reportedLicenses);
            } else {
                reportUnknownLicenseInContext(inventoryResource, artifact, counter,
                        getLicenses(artifact, com.metaeffekt.artifact.analysis.metascan.Constants.KEY_DERIVED_LICENSES),
                        "derived/observed", reportedLicenses);

                reportUnknownLicenseInContext(inventoryResource, artifact, counter,
                        getLicenses(artifact, KEY_BINARY_ARTIFACT_DERIVED_LICENSES),
                        "derived/binary", reportedLicenses);

                reportUnknownLicenseInContext(inventoryResource, artifact, counter,
                        getLicenses(artifact, KEY_SOURCE_ARTIFACT_DERIVED_LICENSES),
                        "derived/source", reportedLicenses);

                reportUnknownLicenseInContext(inventoryResource, artifact, counter,
                        getLicenses(artifact, KEY_SOURCE_ARCHIVE_DERIVED_LICENSES),
                        "derived/source-archive", reportedLicenses);
                reportUnknownLicenseInContext(inventoryResource, artifact, counter,
                        getLicenses(artifact, KEY_DESCRIPTOR_DERIVED_LICENSES),
                        "derived/descriptor", reportedLicenses);
            }
        }
    }

    private static void reportArtifactsWithKnownButNotAssessedLicense(InventoryResource inventoryResource, Artifact artifact, int[] counter) {
        // NOTE: check is only performed for putative components
        final boolean isComponent = isComponent(artifact);
        if (isComponent) {
            // prevent licenses are reported multiple time in different contexts
            final Set<String> reportedLicenses = new HashSet<>();

            // report on associated/effective licenses if available
            final List<String> licenses = artifact.getLicenses();
            if (!licenses.isEmpty()) {
                reportKnownButNotAssessedLicenseInContext(inventoryResource, artifact, counter,
                        licenses, "associated", reportedLicenses);
                reportKnownButNotAssessedLicenseInContext(inventoryResource, artifact, counter,
                        inventoryResource.getInventory().getEffectiveLicenses(artifact), "effective", reportedLicenses);
            } else {
                reportKnownButNotAssessedLicenseInContext(inventoryResource, artifact, counter,
                        getLicenses(artifact, com.metaeffekt.artifact.analysis.metascan.Constants.KEY_DERIVED_LICENSES),
                        "derived/binary", reportedLicenses);

                reportKnownButNotAssessedLicenseInContext(inventoryResource, artifact, counter,
                        getLicenses(artifact, KEY_BINARY_ARTIFACT_DERIVED_LICENSES),
                        "derived/binary", reportedLicenses);
                reportKnownButNotAssessedLicenseInContext(inventoryResource, artifact, counter,
                        getLicenses(artifact,  KEY_SOURCE_ARTIFACT_DERIVED_LICENSES),
                        "derived/source", reportedLicenses);
                reportKnownButNotAssessedLicenseInContext(inventoryResource, artifact, counter,
                        getLicenses(artifact,  KEY_SOURCE_ARCHIVE_DERIVED_LICENSES),
                        "derived/source-archive", reportedLicenses);
                reportKnownButNotAssessedLicenseInContext(inventoryResource, artifact, counter,
                        getLicenses(artifact,  KEY_DESCRIPTOR_DERIVED_LICENSES),
                        "derived/descriptor", reportedLicenses);

            }
        }
    }

    private static void reportUnknownLicenseInContext(InventoryResource inventoryResource, Artifact artifact, int[] counter, List<String> licenses, String licenseContext, Set<String> reportedLicenses) {
        for (String license : licenses) {
            if (reportedLicenses.contains(license)) continue;
            reportedLicenses.add(license);

            if (license.contains(" + ")) continue;
            if (license.contains(("(undefined"))) continue;

            final LicenseData licenseData = inventoryResource.getInventory().findMatchingLicenseData(license);
            if (licenseData != null) {
                final String unknownLicenseTypeMarker = licenseData.get("Unknown License Type");
                if ("x".equalsIgnoreCase(unknownLicenseTypeMarker)) {
                    reportIssueItem(inventoryResource, artifact, counter,
                            "Unknown (" + licenseContext + ") license [" + license + "] detected.",
                            "UNKNOWN-LICENSE", "CRITICAL", "ACTIVE", license);
                }
            }
        }
    }

    private static void reportArtifactsWithInsufficientLicenseStatusInContext(
            InventoryResource inventoryResource, Artifact artifact, int[] counter,
            List<String> licenses, String licenseContext, Set<String> reportedLicenses) {
        for (String license : licenses) {
            if (reportedLicenses.contains(license)) continue;
            reportedLicenses.add(license);

            if (license.contains(" + ")) continue;
            if (license.contains(("(undefined"))) continue;

            final LicenseData licenseData = inventoryResource.getInventory().findMatchingLicenseData(license);
            if (licenseData != null) {

                // OPEN CODE
                final String openCodeStatus = licenseData.get("Open CoDE Status");
                if (!"approved".equalsIgnoreCase(openCodeStatus)) {
                    if ("not approved".equalsIgnoreCase(openCodeStatus)) {
                        reportIssueItem(inventoryResource, artifact, counter,
                                "Detected not Open CoDE approved (" + licenseContext + ") license [" + license + "].",
                                "NOT-APPROVED-LICENSE-OPEN-CODE", "HIGH", "ACTIVE", license);
                    } else {
                        reportIssueItem(inventoryResource, artifact, counter,
                                "Detected (" + licenseContext + ") license [" + license + "] without Open CoDE status.",
                                "MISSING-OPEN-CODE-LICENSE-STATUS", "MEDIUM", "REQUEST", license);
                    }
                }

                // OSI
                final String osiStatus = licenseData.get("OSI Statuss");
                if (osiStatus != null && !"approved".equalsIgnoreCase(osiStatus.trim())) {
                    reportIssueItem(inventoryResource, artifact, counter,
                            "Detected not OSI approved (" + licenseContext + ") license [" + license + "]. Current status [" + osiStatus + "].",
                            "NOT-APPROVED-LICENSE-OSI", "MEDIUM", "PASSIVE", license);
                }
            }
        }
    }

    private static void reportKnownButNotAssessedLicenseInContext(InventoryResource inventoryResource, Artifact artifact, int[] counter,
                                                                  List<String> licenses, String licenseContext, Set<String> reportedLicenses) {

        for (String license : licenses) {
            if (reportedLicenses.contains(license)) continue;
            reportedLicenses.add(license);

            if (license.contains(" + ")) continue;
            if (license.contains(("(undefined"))) continue;

            final LicenseData licenseData = inventoryResource.getInventory().findMatchingLicenseData(license);
            if (licenseData != null) {
                boolean assessed = "x".equalsIgnoreCase(licenseData.get(KEY_CATEGORY_USEABLE_WITH_ADDITIONAL_SERVICES));
                assessed |= "x".equalsIgnoreCase(licenseData.get(KEY_CATEGORY_USEABLE_WITH_MAJOR_OBLIGATIONS));
                assessed |= "x".equalsIgnoreCase(licenseData.get("Nutzung mit Berücksichtigung unwesentlicher Einschränkungen"));
                assessed |= "x".equalsIgnoreCase(licenseData.get(KEY_CATEGORY_USE_NOT_POSSIBLE));

                if (!assessed) {
                    boolean unknown = "x".equalsIgnoreCase(licenseData.get("Unknown License Type"));
                    if (!unknown) {
                        reportIssueItem(inventoryResource, artifact, counter,
                                "Not assessed (" + licenseContext + ") license [" + license + "] detected.",
                                "NOT-ASSESSED-LICENSE", "HIGH", "ACTIVE", license);
                    }
                }
            }
        }
    }

    // FIXME: use artifact.isComponentOrComponentPart()
    private static boolean isComponent(Artifact artifact) {
        final String type = artifact.get(Constants.KEY_TYPE);
        final String suffix = artifact.getType();

        // extract criteria for identifying a qualified component
        return !StringUtils.isEmpty(type) ||
                "jar".equalsIgnoreCase(suffix) ||
                "war".equalsIgnoreCase(suffix) ||
                "sar".equalsIgnoreCase(suffix) ||
                "ear".equalsIgnoreCase(suffix);
    }

    private static void reportArtifactsWithLicenseInCategory(InventoryResource inventoryResource, Artifact artifact, int[] counter,
                                                             String category, String severity, String policy, Map<String, ReportData> licenceToReportDataMap) {

        final Set<String> assetIds = InventoryUtils.collectAssetIdsFromAssetMetaData(inventoryResource.getInventory());

        // check (consolidated) associated licenses
        final List<String> licenses = artifact.getLicenses();
        if (!licenses.isEmpty()) {
            checkAndReportLicensesInCategory(inventoryResource, artifact, counter,
                    category, severity, policy,
                    licenses, "associated", licenceToReportDataMap, assetIds);
            checkAndReportLicensesInCategory(inventoryResource, artifact, counter,
                    category, severity, policy,
                    inventoryResource.getInventory().getEffectiveLicenses(artifact), "effective", licenceToReportDataMap, assetIds);
        } else {
            checkAndReportLicensesInCategory(inventoryResource, artifact, counter,
                    category, severity, policy,
                    getLicenses(artifact, KEY_DERIVED_LICENSES),
                    "derived/binary", licenceToReportDataMap, assetIds);
            checkAndReportLicensesInCategory(inventoryResource, artifact, counter,
                    category, severity, policy,
                    getLicenses(artifact, KEY_BINARY_ARTIFACT_DERIVED_LICENSES),
                    "derived/binary", licenceToReportDataMap, assetIds);
            checkAndReportLicensesInCategory(inventoryResource, artifact, counter,
                    category, severity, policy,
                    getLicenses(artifact, KEY_SOURCE_ARTIFACT_DERIVED_LICENSES),
                    "derived/source", licenceToReportDataMap, assetIds);
            checkAndReportLicensesInCategory(inventoryResource, artifact, counter,
                    category, severity, policy,
                    getLicenses(artifact, KEY_SOURCE_ARCHIVE_DERIVED_LICENSES),
                    "derived/source-archive", licenceToReportDataMap, assetIds);
            checkAndReportLicensesInCategory(inventoryResource, artifact, counter,
                    category, severity, policy,
                    getLicenses(artifact, KEY_DESCRIPTOR_DERIVED_LICENSES),
                    "derived/descriptor", licenceToReportDataMap, assetIds);
        }
    }

    private static List<String> getLicenses(Artifact artifact, String licensesKey) {
        return InventoryUtils.tokenizeLicense(artifact.get(licensesKey), false, false);
    }

    private static void checkAndReportLicensesInCategory(InventoryResource inventoryResource, Artifact artifact, int[] counter,
             String category, String severity, String policy, List<String> licenses, String licenseContext, Map<String, ReportData> reportedMap, Set<String> assetIds) {

        for (String canonicalLicenseName : licenses) {
            final String qualifier = category + "-" + canonicalLicenseName + "-" + artifact.getId() + "-" + artifact.getGroupId() + "-" + artifact.getVersion();

            final LicenseData licenseData = inventoryResource.getInventory().findMatchingLicenseData(canonicalLicenseName);

            if (licenseData != null && "x".equalsIgnoreCase(licenseData.get(category))) {
                if (!reportedMap.containsKey(qualifier)) {
                    final ReportData reportData = reportIssueItem(inventoryResource, artifact, counter,
                            "Categorized (" + licenseContext + ") license [" + canonicalLicenseName + "] in category [" + category + "].",
                            "LICENSE-ISSUE", severity, policy, canonicalLicenseName);

                    reportedMap.put(qualifier, reportData);
                }

                // merge markers
                final ReportData reportData = reportedMap.get(qualifier);

                applyAssetMarkers(artifact, reportData, assetIds);
            }
        }
    }

    private static ReportData reportIssueItem(InventoryResource inventoryResource, Artifact artifact, int[] counter,
                  String cause, String idPart, String severity, String policy, String licenseOrMarker) {
        ReportData reportData = new ReportData();

        // issue identification
        reportData.set("Id", String.format("%04d", counter[0]++));
        reportData.set("Type", idPart);

        // artifact identification
        reportData.set("Artifact Id", artifact.getId());
        reportData.set("Artifact GroupId", artifact.getGroupId());
        reportData.set("Artifact Version", artifact.getVersion());

        // assessment details
        reportData.set("Cause", cause);
        reportData.set("Severity", severity);
        reportData.set("Policy", policy);

        // include markers
        Set<String> assetIds = InventoryUtils.collectAssetIdsFromAssetMetaData(inventoryResource.getInventory());
        applyAssetMarkers(artifact, reportData, assetIds);

        // include license / marker
        reportData.set("License / Marker", licenseOrMarker);

        inventoryResource.getInventory().getReportData().add(reportData);

        return reportData;
    }

    private static void applyAssetMarkers(Artifact artifact, ReportData reportData, Set<String> assetIds) {
        for (String assetId : assetIds) {
            final String assetMarkerForAsset = artifact.get(assetId);
            if (StringUtils.hasText(assetMarkerForAsset)) {
                reportData.set(assetId, assetMarkerForAsset);
            }
            String artifactAssetId = InventoryUtils.deriveAssetIdFromArtifact(artifact);
            if (assetId.equals(artifactAssetId)) {
                reportData.set(assetId, Constants.MARKER_CROSS);
            }
        }
    }

    public static boolean hasCustomerAssessment(LicenseData inputLicenseData) {
        boolean hasAssessment = StringUtils.hasText(inputLicenseData.get(AttachReportFlow.KEY_LICENSE_COMMENTS));
        hasAssessment |= StringUtils.hasText(inputLicenseData.get(AttachReportFlow.KEY_CATEGORY_USE_NOT_POSSIBLE));
        hasAssessment |= StringUtils.hasText(inputLicenseData.get(AttachReportFlow.KEY_CATEGORY_USEABLE_WITH_ADDITIONAL_SERVICES));
        hasAssessment |= StringUtils.hasText(inputLicenseData.get(AttachReportFlow.KEY_CATEGORY_USEABLE_WITH_MAJOR_OBLIGATIONS));
        hasAssessment |= StringUtils.hasText(inputLicenseData.get(AttachReportFlow.KEY_CATEGORY_USEABLE_WITH_MINOR_OBLIGATIONS));
        return hasAssessment;
    }

}
