/*
 * Copyright 2021-2024 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.metaeffekt.artifact.analysis.flow;

import com.metaeffekt.artifact.analysis.utils.InventoryUtils;
import com.metaeffekt.resource.InventoryResource;
import org.metaeffekt.core.inventory.processor.model.Inventory;
import org.metaeffekt.core.inventory.processor.model.InventorySerializationContext;

import java.util.ArrayList;
import java.util.Comparator;
import java.util.List;

import static com.metaeffekt.artifact.analysis.flow.AttachReportFlow.*;
import static com.metaeffekt.artifact.analysis.metascan.Constants.*;

public class FormatInventoryFlow {

    public void filterAndFormat(InventoryResource inventoryResource) {
        final Inventory inventory = inventoryResource.getInventory();

        InventoryUtils.renameArtifactAttributes(inventory, "Vendor", "Organization");
        InventoryUtils.renameArtifactAttributes(inventory, "Vendor URL", "Organization URL");

        InventoryUtils.renameArtifactAttributes(inventory, "Checksum", "Checksum (MD5)");
        InventoryUtils.renameArtifactAttributes(inventory, "Content MD5", "Content Checksum (MD5)");
        InventoryUtils.renameArtifactAttributes(inventory, "Checksum (SHA-1)", "Hash (SHA-1)");

        InventoryUtils.renameArtifactAttributes(inventory, "Alpine Group", "Package Group");

        InventoryUtils.renameArtifactAttributes(inventory, "Specified Package License", "Package Specified License");
        InventoryUtils.renameArtifactAttributes(inventory, "Specified Package License (mapped)", "Package Specified License (mapped)");
        InventoryUtils.renameArtifactAttributes(inventory, "Specified Package License (unknown)", "Package Specified License (unknown)");

        InventoryUtils.removeAssetAttribute("meta_LastTagTime", inventory);

        InventoryUtils.removeArtifactInheritedAttributes(inventory);
//        InventoryUtils.removeArtifactInternalAttributes(inventory);

        // sort all sheets
        inventory.getArtifacts().sort(Comparator.comparing(InventoryUtils::createStringRepresentation));
        inventory.getLicenseData().sort(Comparator.comparing(InventoryUtils::createStringRepresentation));
        inventory.getAssetMetaData().sort(Comparator.comparing(InventoryUtils::createStringRepresentation));

        InventorySerializationContext.initializeArtifactSerializationContext(inventory);

        final InventorySerializationContext serializationContext = inventory.getSerializationContext();
        final List<String> artifactColumnList = serializationContext.get(InventorySerializationContext.CONTEXT_ARTIFACT_COLUMN_LIST);
        final List<String> licenseDataColumnList = InventorySerializationContext.initializeLicenseDataSerializationContext(inventory);

        insertItem(artifactColumnList, "Id", "Checksum (MD5)", 1);
        insertItem(artifactColumnList, "Checksum (MD5)", "Hash (SHA-1)", 1);

        // keep other checksums and hashes independent
        insertItem(artifactColumnList, "Content Checksum (MD5)", "Hash (SHA-256)", 1);
        insertItem(artifactColumnList, "Content Checksum (MD5)", "Hash (SHA-1)", 1);
        insertItem(artifactColumnList, "Hash (SHA-1)", "Hash (SHA-256)", 1);

        insertItem(artifactColumnList, "Analysis Path", "Source - Analysis Path", 1);
        insertItem(artifactColumnList, "Source - Archive Path", "Archive Path", 0);

        if (!artifactColumnList.contains("Notice Parameter")) {
            artifactColumnList.add("Notice Parameter");
        }
        if (!artifactColumnList.contains("Classification")) {
            artifactColumnList.add("Classification");
        }

        insertItem(artifactColumnList, "Version", "Architecture", 1);
        insertItem(artifactColumnList, "Version", "License", 1);
        insertItem(artifactColumnList, "License", "Notice Parameter", 1);
        insertItem(artifactColumnList, "Notice Parameter", "Classification", 1);
        insertItem(artifactColumnList, "Classification", "Rationale", 1);
        insertItem(artifactColumnList, "Rationale", "Summary", 1);
        insertItem(artifactColumnList, "Rationale", "Description", 1);

        insertItem(artifactColumnList, "Package Documentation Path", "Package Group", 0);

        insertItem(artifactColumnList, "Projects", KEY_ARCHIVE_PATH, 1);
        insertItem(artifactColumnList, KEY_ARCHIVE_PATH, KEY_ANALYSIS_PATH, 1);
        insertItem(artifactColumnList, KEY_ANALYSIS_PATH, "Result Path", 1);
        insertItem(artifactColumnList, "Result Path", KEY_DERIVED_LICENSES, 1);
        insertItem(artifactColumnList, KEY_DERIVED_LICENSES, KEY_DERIVED_MARKERS, 1);
        insertItem(artifactColumnList, KEY_DERIVED_MARKERS, KEY_DERIVED_LICENSES_SCANCODE, 1);
        insertItem(artifactColumnList, KEY_DERIVED_LICENSES_SCANCODE, KEY_EXTRACTED_COPYRIGHTS_SCANCODE, 1);

        insertItem(artifactColumnList, "Type", "Errors", 1);
        insertItem(artifactColumnList, "Projects", "Incomplete Match", 1);

        int i = 0;
        for (String attribute : new ArrayList<>(artifactColumnList)) {
            // sort assets before summary
            if (attribute.startsWith("EID-")) {
                insertItem(artifactColumnList, "Analysis Path", attribute, 1 + i);
                insertItem(artifactColumnList, attribute, "Summary", 1);
                i++;
            }

            // remove such that the data is added at the end
            if (attribute.startsWith("AID-")) {
                artifactColumnList.remove(attribute);
            }

        }

        artifactColumnList.remove("Path in Asset");
        artifactColumnList.add("Path in Asset");

        artifactColumnList.remove("Hash (SHA-256)");
        artifactColumnList.remove("Identified Terms");
        artifactColumnList.remove("Organization");
        artifactColumnList.remove("Organization Id");
        artifactColumnList.remove("Packaging");

        // license data
        insertItem(licenseDataColumnList, KEY_LICENSE_COMMENTS, KEY_CATEGORY_USEABLE_WITH_MINOR_OBLIGATIONS, 1);
        insertItem(licenseDataColumnList, KEY_LICENSE_COMMENTS, KEY_CATEGORY_USEABLE_WITH_MAJOR_OBLIGATIONS, 1);
        insertItem(licenseDataColumnList, KEY_LICENSE_COMMENTS, KEY_CATEGORY_USEABLE_WITH_ADDITIONAL_SERVICES, 1);
        insertItem(licenseDataColumnList, KEY_LICENSE_COMMENTS, KEY_CATEGORY_USE_NOT_POSSIBLE, 1);
        insertItem(licenseDataColumnList, KEY_LICENSE_COMMENTS, KEY_ASSESSMENT_PENDING, 1);
        insertItem(licenseDataColumnList, KEY_CATEGORY_USE_NOT_POSSIBLE, KEY_CATEGORY_USEABLE_WITH_ADDITIONAL_SERVICES, 1);
        insertItem(licenseDataColumnList, KEY_CATEGORY_USEABLE_WITH_ADDITIONAL_SERVICES, KEY_CATEGORY_USEABLE_WITH_MAJOR_OBLIGATIONS, 1);
        insertItem(licenseDataColumnList, KEY_CATEGORY_USEABLE_WITH_MAJOR_OBLIGATIONS, KEY_CATEGORY_USEABLE_WITH_MINOR_OBLIGATIONS, 1);
        insertItem(licenseDataColumnList, KEY_CATEGORY_USEABLE_WITH_MINOR_OBLIGATIONS, KEY_ASSESSMENT_PENDING, 1);

        insertItem(licenseDataColumnList, "SPDX Id", "ScanCode Ids", 1);
        insertItem(licenseDataColumnList, "ScanCode Ids", "OSI Status", 1);
        insertItem(licenseDataColumnList, "OSI Status", "Open CoDE Status", 1);

        if (licenseDataColumnList.contains("Commercial")) {
            insertItem(licenseDataColumnList, "Open CoDE Status", "Commercial", 1);
        }
        if (licenseDataColumnList.contains("(Red Hat Only)")) {
            insertItem(licenseDataColumnList, "Open CoDE Status", "(Red Hat Only)", 1);
        }

        serializationContext.put("licensedata.header.[Keine Nutzung möglich].fg", "248,203,173");
        serializationContext.put("licensedata.header.[Keine Nutzung möglich].orientation", "up");
        serializationContext.put("licensedata.header.[Keine Nutzung möglich].width", 42);
        serializationContext.put("licensedata.column.[Keine Nutzung möglich].centered", true);

        serializationContext.put("licensedata.header.[Nutzung nur mit zusätzlichen Services].fg", "255,230,153");
        serializationContext.put("licensedata.header.[Nutzung nur mit zusätzlichen Services].orientation", "up");
        serializationContext.put("licensedata.header.[Nutzung nur mit zusätzlichen Services].width", 42);
        serializationContext.put("licensedata.column.[Nutzung nur mit zusätzlichen Services].centered", true);

        serializationContext.put("licensedata.header.[Nutzung mit Berücksichtigung wesentlicher Einschränkungen].fg", "226,239,218");
        serializationContext.put("licensedata.header.[Nutzung mit Berücksichtigung wesentlicher Einschränkungen].orientation", "up");
        serializationContext.put("licensedata.header.[Nutzung mit Berücksichtigung wesentlicher Einschränkungen].width", 42);
        serializationContext.put("licensedata.column.[Nutzung mit Berücksichtigung wesentlicher Einschränkungen].centered", true);

        serializationContext.put("licensedata.header.[Nutzung mit Berücksichtigung unwesentlicher Einschränkungen].fg", "198,224,180");
        serializationContext.put("licensedata.header.[Nutzung mit Berücksichtigung unwesentlicher Einschränkungen].orientation", "up");
        serializationContext.put("licensedata.header.[Nutzung mit Berücksichtigung unwesentlicher Einschränkungen].width", 42);
        serializationContext.put("licensedata.column.[Nutzung mit Berücksichtigung unwesentlicher Einschränkungen].centered", true);

        serializationContext.put("licensedata.header.[OSI Approved].fg", "221,235,247");
        serializationContext.put("licensedata.header.[OSI Approved].orientation", "up");
        serializationContext.put("licensedata.header.[OSI Approved].width", 42);
        serializationContext.put("licensedata.column.[OSI Approved].centered", true);

        serializationContext.put("licensedata.header.[OSI Status].fg", "221,235,247");
        serializationContext.put("licensedata.header.[OSI Status].orientation", "up");
        serializationContext.put("licensedata.header.[OSI Status].width", 68);
        serializationContext.put("licensedata.column.[OSI Status].centered", true);

        serializationContext.put("licensedata.header.[Open CoDE Status].fg", "231,245,247");
        serializationContext.put("licensedata.header.[Open CoDE Status].orientation", "up");
        serializationContext.put("licensedata.header.[Open CoDE Status].width", 68);
        serializationContext.put("licensedata.column.[Open CoDE Status].centered", true);

        serializationContext.put("licensedata.header.[Commercial].fg", "221,235,247");
        serializationContext.put("licensedata.header.[Commercial].orientation", "up");
        serializationContext.put("licensedata.column.[Commercial].centered", true);

        serializationContext.put("licensedata.header.[(Red Hat Only)].fg", "231,245,247");
        serializationContext.put("licensedata.header.[(Red Hat Only)].orientation", "up");
        serializationContext.put("licensedata.column.[(Red Hat Only)].centered", true);

        serializationContext.put("licensedata.header.[Type Artifact].fg", "219,219,219");
        serializationContext.put("licensedata.header.[Type Artifact].orientation", "up");
        serializationContext.put("licensedata.header.[Type Artifact].width", 20);
        serializationContext.put("licensedata.column.[Type Artifact].centered", true);

        serializationContext.put("licensedata.header.[Type Package].fg", "219,219,219");
        serializationContext.put("licensedata.header.[Type Package].orientation", "up");
        serializationContext.put("licensedata.header.[Type Package].width", 20);
        serializationContext.put("licensedata.column.[Type Package].centered", true);

        serializationContext.put("licensedata.header.[Type Web Module].fg", "219,219,219");
        serializationContext.put("licensedata.header.[Type Web Module].orientation", "up");
        serializationContext.put("licensedata.header.[Type Web Module].width", 20);
        serializationContext.put("licensedata.column.[Type Web Module].centered", true);

        serializationContext.put("licensedata.header.[Unknown License Type].fg", "255,219,219");
        serializationContext.put("licensedata.header.[Unknown License Type].orientation", "up");
        serializationContext.put("licensedata.header.[Unknown License Type].width", 20);
        serializationContext.put("licensedata.column.[Unknown License Type].centered", true);

        serializationContext.put("licensedata.header.[Assessment Pending].orientation", "up");
        serializationContext.put("licensedata.header.[Assessment Pending].width", 20);
        serializationContext.put("licensedata.column.[Assessment Pending].centered", true);

        serializationContext.put("licensedata.header.[Open CoDE Assessment Pending].orientation", "up");
        serializationContext.put("licensedata.header.[Open CoDE Assessment Pending].width", 20);
        serializationContext.put("licensedata.column.[Open CoDE Assessment Pending].centered", true);


        // FIXME: styling of artifacts not supported yet; core change required
        serializationContext.put("artifact.header.[Binary Artifact - Analysis Path].fg", "221,232,251");
        serializationContext.put("artifact.header.[Source Artifact - Analysis Path].fg", "251,232,207");
        serializationContext.put("artifact.header.[Source Archive - Analysis Path].fg", "251,242,207");
        serializationContext.put("artifact.header.[Descriptor - Analysis Path].fg", "187,226,229");
    }

    protected void insertItem(List<String> attributeList, String referenceItem, String insertItem, int offset) {
        if (attributeList == null) return;
        int indexOfReferenceItem = attributeList.indexOf(referenceItem);
        int indexOfItem = attributeList.indexOf(insertItem);

        // do not insert new (empty attributes)
        if (indexOfItem == -1) return;

        if (indexOfReferenceItem >= 0) {
            attributeList.remove(insertItem);
            attributeList.add(Math.min(Math.max(0, indexOfReferenceItem + offset), attributeList.size()), insertItem);
        }
    }

}
