/*
 * Copyright 2021-2024 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.metaeffekt.artifact.analysis.flow;

import com.metaeffekt.resource.InventoryResource;
import org.metaeffekt.artifact.resolver.ArtifactResolverConfig;
import org.metaeffekt.artifact.resolver.ArtifactResolverManager;
import org.metaeffekt.artifact.resolver.download.ProxyConfig;
import org.metaeffekt.artifact.resolver.download.WebAccess;
import org.metaeffekt.artifact.resolver.model.DownloadLocation;
import org.metaeffekt.core.inventory.processor.model.Artifact;
import org.metaeffekt.core.inventory.processor.model.Inventory;

import java.io.File;

public class ResolverFlow {

    public InventoryResult process(ResolverFlowParam resolverFlowParam) {
        final ArtifactResolverManager artifactResolverManager =
                instantiateArtifactResolver(resolverFlowParam);

        final InventoryResource inventoryResource = resolverFlowParam.getInventoryResource();
        final Inventory inventory = inventoryResource.getInventory();

        for (Artifact artifact : inventory.getArtifacts()) {
            artifactResolverManager.resolveAndMergeAll(artifact);
        }

        return new InventoryResult(inventoryResource);
    }

    private ArtifactResolverManager instantiateArtifactResolver(ResolverFlowParam resolverFlowParam) {

        final File downloadBaseDir = new File(resolverFlowParam.getPropertyProvider().
                getRequiredProperty("artifact.resolver.download.dir"));

        final File proxyConfigYaml = new File(resolverFlowParam.getPropertyProvider().
                getRequiredProperty("artifact.resolver.proxy.config"));

        final File artifactResolverConfigYaml = new File(resolverFlowParam.getPropertyProvider().
                getRequiredProperty("artifact.resolver.config"));

        try {
            DownloadLocation downloadLocation = new DownloadLocation(downloadBaseDir);
            WebAccess webAccess = new WebAccess(ProxyConfig.parseFromYaml(proxyConfigYaml));
            ArtifactResolverConfig artifactResolverConfig = ArtifactResolverConfig.parseFromYaml(artifactResolverConfigYaml);
            return new ArtifactResolverManager(downloadLocation, webAccess, artifactResolverConfig);
        } catch (Exception e) {
            throw new IllegalStateException("Cannot create Artifact Resolver.", e);
        }
    }
}
