/*
 * Copyright 2021-2024 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.metaeffekt.artifact.analysis.flow;

import com.metaeffekt.artifact.analysis.model.PropertyProvider;
import com.metaeffekt.artifact.analysis.utils.FileUtils;
import com.metaeffekt.artifact.analysis.utils.InventoryUtils;
import com.metaeffekt.artifact.analysis.utils.TimeUtils;
import com.metaeffekt.artifact.analysis.workbench.ProjectInventoryFilter;
import com.metaeffekt.artifact.extractors.configuration.DirectoryScanExtractorConfiguration;
import com.metaeffekt.artifact.extractors.flow.ExtractorFlow;
import com.metaeffekt.artifact.extractors.flow.ExtractorParam;
import com.metaeffekt.artifact.extractors.flow.ExtractorResult;
import com.metaeffekt.artifact.terms.model.LicenseTextProvider;
import com.metaeffekt.artifact.terms.model.NormalizationMetaData;
import com.metaeffekt.resource.InventoryResource;
import org.metaeffekt.core.inventory.processor.model.*;
import org.metaeffekt.core.inventory.processor.report.DirectoryInventoryScan;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.File;
import java.io.IOException;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.function.Consumer;

public class ScanConsumer implements Consumer<File> {

    private static final Logger LOG = LoggerFactory.getLogger(ScanConsumer.class);

    private final File analysisBaseDir;
    private final File tmpBaseDir;
    private final File resultTargetDir;

    private final InventoryResource referenceInventoryResource;

    private final ScanFlowParam scanFlowParamTemplate;

    public ScanConsumer(File analysisBaseDir, File tmpBaseDir, File resultTargetDir,
                        InventoryResource referenceInventoryResource,
                        ScanFlowParam scanFlowParamTemplate) {

        this.analysisBaseDir = analysisBaseDir;
        this.tmpBaseDir = tmpBaseDir;
        this.resultTargetDir = resultTargetDir;
        this.referenceInventoryResource = referenceInventoryResource;
        this.scanFlowParamTemplate = scanFlowParamTemplate;
    }

    @Override
    public void accept(File file) {
        final String humanReadableTimestamp = TimeUtils.createHumanReadableTimestamp();

        final File intermediateOutputDir = new File(tmpBaseDir, file.getName() + "-" + humanReadableTimestamp);
        final File targetOutputDir = new File(resultTargetDir, file.getName() + "-" + humanReadableTimestamp);

        final File intermediateScanDir = new File(tmpBaseDir, "scan-" + file.getName() );

        try {
            // manage scan dir and output dir
            manageInputAndScanDirForFile(file, intermediateOutputDir, intermediateScanDir);

            final PropertyProvider propertyProvider = scanFlowParamTemplate.getPropertyProvider();
            final String[] licenseScanIncludes = propertyProvider.getProperty("analyze.scan.license.includes", "**/*").split(",");
            final String[] licenseScanExcludes = propertyProvider.getProperty("analyze.scan.license.excludes", "**/.DS_Store*").split(",");

            // reload the reference inventory to anticipate changes
            final Inventory referenceInventory = referenceInventoryResource.reloadInventory();

            // create file scan inventory
            final DirectoryInventoryScan scan = new DirectoryInventoryScan(
                    intermediateScanDir, intermediateScanDir,
                    licenseScanIncludes, licenseScanExcludes, referenceInventory);
            scan.setEnableImplicitUnpack(true);
            scan.setIncludeEmbedded(true);
            scan.setEnableDetectComponentPatterns(true);

            // NOTE: this is the method that does not manage input/scan dir
            final Inventory inventory = scan.performScan();

            // initiate a InventoryResource based on the scan result
            InventoryResource inventoryResource = InventoryResource.fromInventory(inventory);

            // write analysis output (no license scan data yet)
            final File analysedInventoryFile = new File(intermediateOutputDir, "license-scanner-inventory-" + humanReadableTimestamp + "_01-analyse.xlsx");
            final File curatedInventoryFile = new File(intermediateOutputDir, "license-scanner-inventory-" + humanReadableTimestamp + "_02-curate.xlsx");
            final File resolvedInventoryFile = new File(intermediateOutputDir, "license-scanner-inventory-" + humanReadableTimestamp + "_03-resolve.xlsx");
            final File scannedInventoryFile = new File(intermediateOutputDir, "license-scanner-inventory-" + humanReadableTimestamp + "_04-scanned.xlsx");

            final File resultInventoryFile = new File(targetOutputDir, "license-scanner-inventory-" + humanReadableTimestamp + ".xlsx");

            // write out before running the packing procedure
            inventoryResource.sync(analysedInventoryFile);

            final DirectoryScanExtractorConfiguration configuration = new
                    DirectoryScanExtractorConfiguration(file.getName(), referenceInventory,
                    analysedInventoryFile, intermediateScanDir);

            // run extractor to prepare archives for the scan
            final ExtractorParam extractorParam = new ExtractorParam()
                    .using(configuration)
                    .extractArchivesTo(new File(intermediateOutputDir, file.getName()));

            {
                LOG.info("Running ExtractorFlow...");
                final ExtractorFlow extractorFlow = new ExtractorFlow();
                final ExtractorResult extractorResult = extractorFlow.extract(extractorParam);

                // sync once again the analysed file
                // FIXME: enable inventory replacement in resource; the extractor
                inventoryResource = InventoryResource.fromInventory(extractorResult.getAggregatedInventory());
                inventoryResource.sync(analysedInventoryFile);

                // add curation data before resolver (add urls; fix versions; fix groupId)
                final ProjectInventoryFilter projectInventoryFilter = new ProjectInventoryFilter();
                projectInventoryFilter.setEnableApplyCurationData(true);
                projectInventoryFilter.process(inventoryResource.getInventory(), referenceInventory);

                inventoryResource.sync(curatedInventoryFile);
                LOG.info("ExtractorFlow completed.");
            }

            // run resolver to fetch additional artifacts / information
            {

                LOG.info("Running ResolverFlow...");
                final ResolverFlow resolverFlow = new ResolverFlow();
                final ResolverFlowParam resolverFlowParam = new ResolverFlowParam();
                resolverFlowParam.withInventory(inventoryResource);
                resolverFlowParam.configuredBy(scanFlowParamTemplate.getPropertyProvider());

                resolverFlow.process(resolverFlowParam);

                inventoryResource.sync(resolvedInventoryFile);
                LOG.info("ResolverFlow completed.");
            }

            // create ScanFlowParam from template
            final ScanFlowParam scanFlowParam = ScanFlowParam.copy(scanFlowParamTemplate);
            scanFlowParam.withInventory(inventoryResource);

            // run the scan
            final ScanFlowResult scanFlowResult = new ScanFlow().process(scanFlowParam);

            // insert outer asset information
            concludeAssetMetaData(file, scanFlowResult);

            // aggregate analysis results (copy for a reference)
            for (final Artifact artifact : scanFlowResult.getInventory().getArtifacts()) {
                aggregateAnalysisResult(targetOutputDir, artifact, artifact.get("Analysis Path"), "");
                aggregateAnalysisResult(targetOutputDir, artifact, artifact.get("Binary Artifact - Analysis Path"), "Binary Artifact - ");
                aggregateAnalysisResult(targetOutputDir, artifact, artifact.get("Source Artifact - Analysis Path"), "Source Artifact - ");
                aggregateAnalysisResult(targetOutputDir, artifact, artifact.get("Source Archive - Analysis Path"), "Source Archive - ");
                aggregateAnalysisResult(targetOutputDir, artifact, artifact.get("Descriptor - Analysis Path"), "Descriptor - ");
            }

            // apply general filter; implicitly collect the licenses
            final ProjectInventoryFilter projectInventoryFilter = new ProjectInventoryFilter();
            projectInventoryFilter.setEnableInherit(false);
            projectInventoryFilter.setEnableUseDerivedLicense(false);
            projectInventoryFilter.setEnableAddComponentNameAndVersion(false);
            projectInventoryFilter.setEnableApplyCurationData(true);
            projectInventoryFilter.process(inventoryResource.getInventory(), referenceInventory);

            // contribute assessment data
            transferBusinessCaseAssessmentData(referenceInventory, scanFlowResult.getInventory());

            // finally write the inventory
            FileUtils.forceMkDirQuietly(scannedInventoryFile.getParentFile());
            inventoryResource.sync(scannedInventoryFile);

            new AttachReportFlow().attachReport(inventoryResource);
            new FormatInventoryFlow().filterAndFormat(inventoryResource);

            inventoryResource.sync(resultInventoryFile);

            // produce SPDX
            final NormalizationMetaData normalizationMetaData = scanFlowParam.getNormalizationMetaData();
            final LicenseTextProvider licenseTextProvider = scanFlowParam.getLicenseTextProvider();
            final File targetSpdxDocumentFile = new File(targetOutputDir, "license-scanner-spdx-" + humanReadableTimestamp + ".json");
            SpdxDocumentFlow.produceSpdxDocument(inventoryResource, file.getName(), targetSpdxDocumentFile, normalizationMetaData, licenseTextProvider, propertyProvider);

        } catch (RuntimeException|IOException e) {
            LOG.error("Error processing scan for [{}]. [Thread {}]",
                    file, Thread.currentThread().getId(), e);
        }
    }

    private void aggregateAnalysisResult(File targetOutputDir, Artifact artifact, String analysisPath, String context) throws IOException {
        if (analysisPath != null) {
            File analysisResults = new File(analysisPath + "-analysis");
            String intermediateFolder = FileUtils.asRelativePath(analysisBaseDir.getCanonicalFile(), analysisResults.getCanonicalFile());
            File targetFile = new File(targetOutputDir, intermediateFolder);

            if (analysisResults.exists()) {
                FileUtils.copyDirectory(analysisResults, targetFile, f -> {
                    if (f.isDirectory()) return false;
                    if (f.getName().endsWith("_license-scan-segments.txt")) return false;
                    return true;
                });

                artifact.set(context + "Result Path", targetFile.getAbsolutePath());
            }
        }
    }

    private void concludeAssetMetaData(File originalFile, ScanFlowResult scanFlowResult) {

        final Inventory inventory = scanFlowResult.getInventory();

        final Set<String> coveredAssetIds = new HashSet<>();
        final Set<String> assetIdList = InventoryUtils.collectAssetIdsFromArtifacts(inventory);

        // decide for every assetId whether it is regarded a top-level asset id and insert an appropriate asset metadata instance
        for (String assetId : assetIdList) {
            // AID-ch.qos.reload4j_1.2.18.4.jar-5612eec61604d2e8c016085f8a2f394c
            final int dashIndexFirst = assetId.indexOf("-");
            final int dashIndexLast = assetId.lastIndexOf("-");
            final String artifactId = assetId.substring(dashIndexFirst + 1, dashIndexLast);
            final String checksum = assetId.substring(dashIndexLast + 1);

            // when the asset matches an artifact an AssetMetaData items is added
            final Artifact artifact = inventory.findArtifactByIdAndChecksum(artifactId, checksum);
            if (artifact != null) {
                createAssetMetaDataFromArtifact(inventory, artifact, assetId);
                coveredAssetIds.add(assetId);
            }
        }

        // also cover artifacts which are found top-level
        for (Artifact artifact : inventory.getArtifacts()) {
            final String artifactAssetId = InventoryUtils.deriveAssetIdFromArtifact(artifact);

            if (artifactAssetId != null) {
                if (!coveredAssetIds.contains(artifactAssetId)) {
                    final Set<String> artifactAssetIds = InventoryUtils.collectAssetIdsFromArtifact(artifact);
                    if (artifactAssetIds.isEmpty()) {
                        createAssetMetaDataFromArtifact(inventory, artifact, artifactAssetId);
                        coveredAssetIds.add(artifactAssetId);
                    }
                }
            }
        }
    }

    private void createAssetMetaDataFromArtifact(Inventory inventory, Artifact artifact, String assetId) {
        final AssetMetaData assetMetaData = new AssetMetaData();
        assetMetaData.set(AssetMetaData.Attribute.ASSET_ID, assetId);

        assetMetaData.set(Constants.KEY_TYPE, "Artifact");
        assetMetaData.set("Artifact Id", artifact.getId());
        assetMetaData.set("Name", artifact.getId());
        assetMetaData.set(Artifact.Attribute.VERSION, artifact.getVersion());
        assetMetaData.set("Tag", artifact.get("Tag"));
        assetMetaData.set("Checksum (MD5)", artifact.getChecksum());
        assetMetaData.set(Constants.KEY_HASH_SHA256, artifact.get(Constants.KEY_HASH_SHA256));
        assetMetaData.set(Constants.KEY_HASH_SHA1, artifact.get(Constants.KEY_HASH_SHA1));

        assetMetaData.set(Constants.KEY_PATH_IN_ASSET, artifact.get(Constants.KEY_PATH_IN_ASSET));

        inventory.getAssetMetaData().add(assetMetaData);
    }

    private void manageInputAndScanDirForFile(File file, File intermediateOutputDir, File intermediateScanDir) throws IOException {
        // delete the folder if it already exists
        if (intermediateScanDir.exists()) {
            FileUtils.deleteDirectory(intermediateScanDir);
        }

        // create/recreate missing folder structures
        FileUtils.forceMkDirQuietly(intermediateScanDir);
        FileUtils.forceMkDirQuietly(intermediateOutputDir);

        // copy from inputDir; either the folder or the file
        if (file.isDirectory()) {
            FileUtils.copyDirectory(file, intermediateScanDir);
        } else {
            FileUtils.copyFile(file, new File(intermediateScanDir, file.getName()));
        }
    }

    @Override
    public Consumer<File> andThen(Consumer<? super File> after) {
        return Consumer.super.andThen(after);
    }

    // FIXME: move to a more obvious position
    public static void transferBusinessCaseAssessmentData(Inventory referenceInventory, Inventory inventory) throws IOException {

        // prepare detected licenseData map for input inventory as well as an attribute key set from the license data
        final Map<String, LicenseData> licenseDataMap = new HashMap<>();
        final Set<String> attributeKeys = new HashSet<>();

        for (LicenseData licenseData : inventory.getLicenseData()) {
            final String licenseId = licenseData.get(LicenseData.Attribute.CANONICAL_NAME);
            licenseDataMap.put(licenseId, licenseData);
            // also collect all the attributes
            attributeKeys.addAll(licenseData.getAttributes());
        }

        // iterate customer assessment data and merge content
        for (LicenseData customerLicenseData : referenceInventory.getLicenseData()) {
            String licenseId = customerLicenseData.get(LicenseData.Attribute.CANONICAL_NAME);
            String updatedLicenseId = null;

            final LicenseData licenseData = licenseDataMap.get(licenseId);

            // compensate license name changes is expected to be already doen

            if (licenseData != null) {
                // transfer missing data to customerLicenseData
                for (String key : customerLicenseData.getAttributes()) {
                    if (!attributeKeys.contains(key)) {
                        licenseData.set(key, customerLicenseData.get(key));
                    }
                }
            }

            // consume the license; such that no older information gets overwritten
            licenseDataMap.remove(licenseId);

            if (updatedLicenseId != null) {
                licenseDataMap.remove(updatedLicenseId);
            }
        }
    }

}
